// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_font.cc
 *  \brief      MGL フォント
 *  \date       Since: May 27, 2021. 16:38:57 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/font/mgl_font.h>

namespace MGL::Render
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フォントをフォントストレージに追加
 *  \param[in]  key         関連付けるフォントキー
 *  \param[in]  resource    追加するフォントリソース
 *  \return     フォントクラス
 */
/* ------------------------------------------------------------------------- */
Font Font::AddStorage(FontKey key, const SharedFontResource &resource) noexcept
{
    if (!FontStorage::GetInstance().Add(key, resource))
    {
        return {};
    }

    Font font;
    font._resource = resource;
    font._key = key;

    return font;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フォントストレージからフォントを読み込み
 *  \param[in]  key     読み込むフォントのキー
 *  \retval     true    成功
 *  \retval     false   キーに対応したフォントが見つからなかった
 */
/* ------------------------------------------------------------------------- */
bool Font::Load(FontKey key) noexcept
{
    _resource = FontStorage::GetInstance().Get(key);
    _key = key;

    return (_resource != nullptr);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字の表示
 *  \param[in]  text    表示する文字
 *  \param[in]  args    フォーマット引数
 *  \retval     true    文字列を最後まで表示した
 *  \retval     false   文字列の表示を途中で中断した
 */
/* ------------------------------------------------------------------------- */
bool Font::Print(const char *text, const Text::FormatArgs &args) noexcept
{
    if (_resource != nullptr)
    {
        auto result = _resource->Print(_work, _option, text, args);

        if (_isOnceLimit)
        {
            _work.limitCount = -1;
        }

        return result;
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インデックス化した文字の表示
 *  \param[in]  indexedString   表示する文字
 *  \param[in]  args            フォーマット引数
 *  \retval     true            文字列を最後まで表示した
 *  \retval     false           文字列の表示を途中で中断した
 */
/* ------------------------------------------------------------------------- */
bool Font::Print(const Text::IndexedCharacter *indexedString, const Text::FormatArgs &args) noexcept
{
    if (_resource != nullptr)
    {
        auto result = _resource->Print(_work, _option, indexedString, args);

        if (_isOnceLimit)
        {
            _work.limitCount = -1;
        }

        return result;
    }

    return false;
}
}    // namespace MGL::Render

// vim: et ts=4 sw=4 sts=4
