// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_ascii_font.cc
 *  \brief      MGL ASCIIフォント
 *  \date       Since: May 26, 2021. 18:36:26 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/font/mgl_ascii_font.h>

namespace MGL::Render
{
namespace
{
constexpr size_t   kGlyphSize       = 0x80; // グリフの数
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ（テクスチャキーで初期化）
 *  \param[in]  textureKey      テクスチャキー
 *  \param[in]  characterWidth  文字の幅
 *  \param[in]  characterHeight 文字の高さ
 *  \param[in]  sourceOffset    テクスチャの参照元座標のオフセット
 */
/* ------------------------------------------------------------------------- */
AsciiFont::AsciiFont(TextureKey textureKey, uint32_t characterWidth, uint32_t characterHeight, const Vector2 &sourceOffset) noexcept
    : IndexedFontResource(FontOrigin::TopLeft)
    , _texture(textureKey)
    , _characterSize(static_cast<float>(characterWidth), static_cast<float>(characterHeight))
    , _sourceOffset(sourceOffset)
{
    Initialize();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ（領域付きテクスチャで初期化）
 *  \param[in]  textureBounds   領域付きテクスチャ
 */
/* ------------------------------------------------------------------------- */
AsciiFont::AsciiFont(const TextureWithBounds &textureBounds) noexcept
    : IndexedFontResource(FontOrigin::TopLeft)
    , _texture(textureBounds.GetTexture())
    , _characterSize(textureBounds.GetBounds().width / 16.0f, textureBounds.GetBounds().height / 8.0f)
    , _sourceOffset(textureBounds.GetBounds().GetPosition())
{
    Initialize();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ（ファイルから画像を読み込み）
 *  \param[in]  imagePath       画像ファイルのパス
 *  \param[in]  characterWidth  文字の幅
 *  \param[in]  characterHeight 文字の高さ
 *  \param[in]  sourceOffset    テクスチャの参照元座標のオフセット
 *  \param[in]  loaderKey       テクスチャローダーのキー
 */
/* ------------------------------------------------------------------------- */
AsciiFont::AsciiFont(const File::PathView &imagePath, uint32_t characterWidth, uint32_t characterHeight, const Vector2 &sourceOffset, TextureLoaderKey loaderKey) noexcept
    : IndexedFontResource(FontOrigin::TopLeft)
    , _texture(imagePath, false, loaderKey)
    , _characterSize(static_cast<float>(characterWidth), static_cast<float>(characterHeight))
    , _sourceOffset(sourceOffset)
{
    Initialize();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ（メモリ上の画像データから読み込み）
 *  \param[in]  imageData       画像データのアドレス
 *  \param[in]  dataSize        画像データのサイズ
 *  \param[in]  characterWidth  文字の幅
 *  \param[in]  characterHeight 文字の高さ
 *  \param[in]  sourceOffset    テクスチャの参照元座標のオフセット
 *  \param[in]  loaderKey       テクスチャローダーのキー
 */
/* ------------------------------------------------------------------------- */
AsciiFont::AsciiFont(const void *imageData, size_t dataSize, uint32_t characterWidth, uint32_t characterHeight, const Vector2 &sourceOffset, TextureLoaderKey loaderKey) noexcept
    : IndexedFontResource(FontOrigin::TopLeft)
    , _texture(imageData, dataSize, false, loaderKey)
    , _characterSize(static_cast<float>(characterWidth), static_cast<float>(characterHeight))
    , _sourceOffset(sourceOffset)
{
    Initialize();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 */
/* ------------------------------------------------------------------------- */
void AsciiFont::Initialize() noexcept
{
    _glyphs = STL::make_unique<FontGlyph[]>(kGlyphSize);

    for (size_t i = 0; i < kGlyphSize; i++)
    {
        auto &glyph = _glyphs[i];

        glyph.code = static_cast<char32_t>(i);
        glyph.size = _characterSize;
        glyph.advance = _characterSize.x;
        glyph.bearing = Vector2(0.0f, 0.0f);

        if (glyph.code == ' ')
        {
            glyph.texture = TextureWithBounds();
        }
        else
        {
            const Vector2 texturePosition(
                    static_cast<float>(i & 0xFu) * glyph.size.x,
                    static_cast<float>((i >> 4u) & 0xFu) * glyph.size.y);

            glyph.texture = TextureWithBounds(
                    _texture,
                    Rectangle(texturePosition, _characterSize));
        };
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      グリフのインデックスを取得
 *  \param[in]  character   取得するUTF-32文字
 *  \param[in]  faceType    フェイスタイプ
 *  \return     対応したグリフのインデックス．見つからない場合は SIZE_T_MAX
 */
/* ------------------------------------------------------------------------- */
size_t AsciiFont::GetIndex(char32_t character, [[maybe_unused]] FontFaceType faceType) const noexcept
{
    // 非ASCII文字は代わりに'?'を返す
    if (character >= kGlyphSize)
    {
        return static_cast<size_t>('?');
    }
    
    return character;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      グリフを取得
 *  \param[in]  index       取得するグリフのインデックス
 *  \param[in]  faceType    フェイスタイプ
 *  \param[in]  option      描画オプション
 *  \return     対応するグリフ情報．見つからない場合はnullptr
 */
/* ------------------------------------------------------------------------- */
const FontGlyph *AsciiFont::GetGlyph(
        size_t index,
        [[maybe_unused]] FontFaceType faceType,
        [[maybe_unused]] const FontOption &option) const noexcept
{
    if (index >= kGlyphSize)
    {
        return nullptr;
    }

    return &_glyphs[index];
}

}   // namespace MGL::Render

// vim: et ts=4 sw=4 sts=4
