/* ------------------------------------------------------------------------- */
/*!
 *  \file       sprite.hlsl
 *  \brief      スプライト用シェーダ
 *  \date       Since: November 18, 2019. 12:05:35 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

Texture2D texture2d : register(t0);
SamplerState samplerState : register(s0);


//! 変換行列
cbuffer CBViewMatrix : register(b0)
{
    matrix view;    //!< ビュー行列
};

//! 描画アトリビュート
struct Attribute
{
    float4 offset_scale;    //!< オフセット（xy）とスケール（zw）
    float4 texCoord;        //!< テクスチャのUV
    float4 color;           //!< 色
    float  rotate;          //!< 回転角度
    float2 pivot;           //!< 回転ピボット
    uint   flip;            //!< フリップ設定
};

//! アトリビュート用定数バッファ
cbuffer CBAttribute : register(b1)
{
    Attribute attribute[ATTRIBUTE_SIZE];
};

//! 頂点シェーダの入力
struct VS_INPUT
{
    float2 position : POSITION;
    uint2 texIndex : TEX_INDEX;
};

//! ピクセルシェーダの入力
struct PS_INPUT
{
    float4 position : SV_POSITION;
    float2 texCoord : TEX_COORD;
    float4 color : COLOR;
};


//! フリップ用のテーブル
static const float2 kFlipTable[4] =
                        {
                            float2(1.0f, 1.0f),
                            float2(-1.0f, 1.0f),
                            float2(1.0f, -1.0f),
                            float2(-1.0f, -1.0f),
                        };
                        

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      頂点シェーダ
 */
/* ------------------------------------------------------------------------- */
PS_INPUT VS_FUNCTION(VS_INPUT input, uint instID : SV_InstanceID)
{
    PS_INPUT output = (PS_INPUT)0;
    Attribute attr = attribute[instID];

    float cosZ = cos(attr.rotate);
    float sinZ = sin(attr.rotate);
    const float2 flip = kFlipTable[attr.flip];

    float4x4 trMatrix =
    {
        cosZ * flip.x, sinZ * flip.y, 0.0f, 0.0f,
        -sinZ * flip.x, cosZ * flip.y, 0.0f, 0.0f,
        0.0f, 0.0f, flip.x * flip.y, 0.0f,
        attr.offset_scale.x, attr.offset_scale.y, 0.0f, 1.0f
    };

    output.position = mul(view, mul(float4(
                                    (input.position.x - attr.pivot.x) * attr.offset_scale.z,
                                    (input.position.y - attr.pivot.y) * attr.offset_scale.w,
                                    0.0f,
                                    1.0f), trMatrix));
    output.color = attr.color;

    output.texCoord[0] = attr.texCoord[input.texIndex[0]];
    output.texCoord[1] = attr.texCoord[input.texIndex[1]];

    return output;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ピクセルシェーダ
 */
/* ------------------------------------------------------------------------- */
float4 PS(PS_INPUT input) : SV_Target
{
    return texture2d.Sample(samplerState, input.texCoord) * input.color;
}

// vim: et ts=4 sw=4 sts=4

