// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_d3d11_wic_texture_loader.cc
 *  \brief      MGL WICによるテクスチャローダー
 *  \date       Since: March 28, 2021. 3:19:29 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/d3d11/mgl_d3d11_wic_texture_loader.h>
#if defined(MGL_RENDERER_ENABLE_D3D11)

#include <mgl/file/mgl_file.h>
#include <mgl/render/d3d11/mgl_d3d11_defs.h>
#include <mgl/system/mgl_system_debug_macro.h>

#pragma comment(lib, "windowscodecs.lib")

namespace MGL::Render::D3D11
{
namespace
{
    //! デコード用関数テーブルの要素
    struct DecodeTableElement
    {
        WICPixelFormatGUID                  pixelFormat;
        WICTextureLoader::DecodeFunction    decodeFunction;
    };

    //! デコード用関数テーブル
    const DecodeTableElement s_DecodeFunctions[] =
    {
        {GUID_WICPixelFormat32bppRGBA,  WICTextureLoader::Load32bppRGBA},       // 32bit RGBA
        {GUID_WICPixelFormat32bppBGRA,  WICTextureLoader::Load32bppBGRA},       // 32bit BGRA
    };
}   // namespace


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
WICTextureLoader::WICTextureLoader() noexcept
{
    _isAvailable = Initialize();    // NOLINT(cppcoreguidelines-prefer-member-initializer)  Note: 無理（処理順が変わる）
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \date       Since: March 28, 2021. 3:47:22 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */
bool WICTextureLoader::Initialize() noexcept
{
    if (!_comUsing.IsValid())
    {
        return false;
    }

    if (auto hr = CoCreateInstance(CLSID_WICImagingFactory, nullptr, CLSCTX_INPROC_SERVER, IID_PPV_ARGS(&_factory)); FAILED(hr))
    {
        return false;
    }

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルから読み込み
 *  \param[out] textureResource     読み込み先のテクスチャリソース
 *  \param[in]  path                ファイルのパス
 *  \retval     true                成功
 *  \retval     false               失敗
 */
/* ------------------------------------------------------------------------- */
bool WICTextureLoader::Load(Render::TextureResource &textureResource, const File::PathView &path) noexcept
{
    // 初期化に失敗していたら処理しない
    if (!_isAvailable)
    {
        return false;
    }

    // Direct3D11用のリソース以外は受け付けない
    if (textureResource.GetRendererType() != kRendererTypeDirect3D11)
    {
        return false;
    }
    
    // 対象ファイルをバッファに読み込んでデコードする
    try
    {
        File::ThrowingHandle file;
        file.Open(path);
        auto size = file.GetSize();
        auto buffer = STL::make_unique<std::byte[]>(size);
        size = file.Read(buffer.get(), size);
        return Load(textureResource, buffer.get(), size);
    }
    catch (const File::Exception &fileException)
    {
        MGL_ERROR(fileException.what());
        return false;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      メモリ上のバッファから読み込み
 *  \param[out] textureResource     読み込み先のテクスチャリソース
 *  \param[in]  imageData           読み込むイメージデータ
 *  \param[in]  dataSize            読み込むイメージデータのサイズ
 *  \retval     true                成功
 *  \retval     false               失敗
 */
/* ------------------------------------------------------------------------- */
bool WICTextureLoader::Load(Render::TextureResource &textureResource, const void *imageData, size_t dataSize) noexcept
{
    // 初期化に失敗していたら処理しない
    if (!_isAvailable)
    {
        return false;
    }

    // Direct3D11用のリソース以外は受け付けない
    if (textureResource.GetRendererType() != kRendererTypeDirect3D11)
    {
        return false;
    }

    // ストリームを生成
    CComPtr<IWICStream> stream;
    if (auto hr = _factory->CreateStream(&stream.p); FAILED(hr))
    {
        return false;
    }

    // メモリ上のデータでストリームを初期化
    // NOLINTNEXTLINE(bugprone-casting-through-void)    Note: APIの引数にconstが付いていないのでどうしようもない（リファレンスによるとinのみらしい）
    if (auto hr = stream->InitializeFromMemory(reinterpret_cast<WICInProcPointer>(const_cast<void *>(imageData)), static_cast<DWORD>(dataSize)); FAILED(hr))
    {
        return false;
    }

    // ストリームからデコーダを生成
    CComPtr<IWICBitmapDecoder> decoder;
    if (auto hr = _factory->CreateDecoderFromStream(stream, nullptr, WICDecodeMetadataCacheOnLoad, &decoder.p); FAILED(hr))
    {
        return false;
    }

    // デコーダからフレームを取得
    CComPtr<IWICBitmapFrameDecode> frame;
    if (auto hr = decoder->GetFrame(0, &frame.p); FAILED(hr))
    {
        return false;
    }

    // ピクセルフォーマットの取得
    WICPixelFormatGUID pixelFormat;
    if (auto hr = frame->GetPixelFormat(&pixelFormat); FAILED(hr))
    {
        return false;
    }

    // ピクセルフォーマットに対応したデコーダを実行
    for (const auto &element : s_DecodeFunctions)
    {
        if (element.pixelFormat == pixelFormat)
        {
            return element.decodeFunction(textureResource, frame);
        }
    }

    // ループから抜けている場合は対応していないピクセルフォーマットなため失敗
    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      32bitRGBAフォーマットの読み込み
 *  \param[out] textureResource     適用するテクスチャリソース
 *  \param[in]  frame               画像のフレーム情報
 *  \retval     true                成功
 *  \retval     false               失敗
 */
/* ------------------------------------------------------------------------- */
bool WICTextureLoader::Load32bppRGBA(Render::TextureResource &textureResource, IWICBitmapFrameDecode *frame) noexcept
{
    constexpr size_t kPixelSize = 4;

    // 幅と高さを取得
    UINT width = 0;
    UINT height = 0;
    if (auto hr = frame->GetSize(&width, &height); FAILED(hr))
    {
        return false;
    }

    // バッファを準備
    const size_t bufferSize = static_cast<size_t>(width) * static_cast<size_t>(height) * kPixelSize;
    auto buffer = STL::make_unique<BYTE []>(bufferSize);

    // ピクセル情報を取得
    if (auto hr = frame->CopyPixels(nullptr, width * kPixelSize, static_cast<UINT>(bufferSize), buffer.get()); FAILED(hr))
    {
        return false;
    }

    // テクスチャを生成
    return textureResource.Create(buffer.get(), PixelFormat::RGBA8_UNorm, static_cast<uint32_t>(width), static_cast<uint32_t>(height));
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      32bitBGRAフォーマットの読み込み
 *  \param[out] textureResource     適用するテクスチャリソース
 *  \param[in]  frame               画像のフレーム情報
 *  \retval     true                成功
 *  \retval     false               失敗
 */
/* ------------------------------------------------------------------------- */
bool WICTextureLoader::Load32bppBGRA(Render::TextureResource &textureResource, IWICBitmapFrameDecode *frame) noexcept
{
    constexpr size_t kPixelSize = 4;

    // 幅と高さを取得
    UINT width = 0;
    UINT height = 0;
    if (auto hr = frame->GetSize(&width, &height); FAILED(hr))
    {
        return false;
    }

    // バッファを準備
    const size_t bufferSize = static_cast<size_t>(width) * static_cast<size_t>(height) * kPixelSize;
    auto buffer = STL::make_unique<BYTE []>(bufferSize);

    // ピクセル情報を取得
    if (auto hr = frame->CopyPixels(nullptr, width * kPixelSize, static_cast<UINT>(bufferSize), buffer.get()); FAILED(hr))
    {
        return false;
    }

    // テクスチャを生成
    return textureResource.Create(buffer.get(), PixelFormat::BGRA8_UNorm, static_cast<uint32_t>(width), static_cast<uint32_t>(height));
}

}   // namespace MGL::Render::D3D11

#endif  // MGL_RENDERER_ENABLE_D3D11
// vim: et ts=4 sw=4 sts=4
