// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_d3d11_texture_resource.cc
 *  \brief      MGL Direct3D11用テクスチャリソース
 *  \date       Since: March 1, 2021. 18:53:17 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/d3d11/mgl_d3d11_texture_resource.h>
#if defined(MGL_RENDERER_ENABLE_D3D11)

#include <mgl/render/d3d11/mgl_d3d11_device.h>
#include <mgl/math/mgl_matrix4x4.h>

namespace MGL::Render::D3D11
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
TextureResource::TextureResource(bool isMainRenderTarget) noexcept
    : _isMainRenderTarget(isMainRenderTarget)
    , _texture(nullptr)
    , _shaderResourceView(nullptr)
    , _renderTargetView(nullptr)
    , _orthogonalMatrixBuffer(nullptr)
    , _width(0)
    , _height(0)
    , _pixelFormat(PixelFormat::Unknown)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャの生成
 *  \param[in]  pixelData   ピクセルデータ
 *  \param[in]  pixelFormat ピクセルフォーマット
 *  \param[in]  width       幅
 *  \param[in]  height      高さ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool TextureResource::Create(const void *pixelData, PixelFormat pixelFormat, uint32_t width, uint32_t height) noexcept
{
    if (IsValid())
    {
        return false;
    }

    auto &device = Device::GetInstance();

    // ピクセルフォーマットを取得
    auto dxPixelFormat = GetDXPixelFormat(pixelFormat);
    if (dxPixelFormat == DXGI_FORMAT_UNKNOWN)
    {
        return false;
    }

    // テクスチャ記述子
    D3D11_TEXTURE2D_DESC textureDesc = {};
    textureDesc.Width = width;
    textureDesc.Height = height;
    textureDesc.MipLevels = 1;
    textureDesc.ArraySize = 1;
    textureDesc.Format = dxPixelFormat;
    textureDesc.SampleDesc.Count = 1;
    textureDesc.Usage = D3D11_USAGE_DEFAULT;
    textureDesc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
    textureDesc.CPUAccessFlags = 0;

    // サブリソース
    D3D11_SUBRESOURCE_DATA subresource = {};
    subresource.pSysMem = pixelData;
    subresource.SysMemPitch = width * 4;
    subresource.SysMemSlicePitch = 0;

    // テクスチャ生成
    if (FAILED(device.GetD3DDevice()->CreateTexture2D(&textureDesc, &subresource, &_texture.p)))
    {
        return false;
    }

    // シェーダリソースビューの生成
    D3D11_SHADER_RESOURCE_VIEW_DESC resourceViewDesc = {};
    resourceViewDesc.Format = textureDesc.Format;
    resourceViewDesc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE2D;
    resourceViewDesc.Texture2D.MostDetailedMip = 0;
    resourceViewDesc.Texture2D.MipLevels = textureDesc.MipLevels;
    if (FAILED(device.GetD3DDevice()->CreateShaderResourceView(_texture, &resourceViewDesc, &_shaderResourceView.p)))
    {
        return false;
    }

    // 情報を保持
    _width = width;
    _height = height;
    _pixelFormat = pixelFormat;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      レンダーターゲットの生成
 *  \param[in]  width       幅
 *  \param[in]  height      高さ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool TextureResource::CreateRenderTarget(uint32_t width, uint32_t height) noexcept
{
    if (IsValid())
    {
        return false;
    }

    auto &device = Device::GetInstance();

    // テクスチャを生成
    D3D11_TEXTURE2D_DESC textureDesc = {};
    textureDesc.Width = width;
    textureDesc.Height = height;
    textureDesc.MipLevels = 1;
    textureDesc.ArraySize = 1;
    textureDesc.Format = device.GetMainRenderTargetPixelFormat();
    textureDesc.SampleDesc.Count = 1;
    textureDesc.Usage = D3D11_USAGE_DEFAULT;
    textureDesc.BindFlags = D3D11_BIND_RENDER_TARGET | D3D11_BIND_SHADER_RESOURCE;
    textureDesc.CPUAccessFlags = 0;
    if (FAILED(device.GetD3DDevice()->CreateTexture2D(&textureDesc, nullptr, &_texture.p)))
    {
        return false;
    }

    // レンダーターゲットビューの生成
    D3D11_RENDER_TARGET_VIEW_DESC rtDesc = {};
    rtDesc.Format = textureDesc.Format;
    rtDesc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2D;
    if (FAILED(device.GetD3DDevice()->CreateRenderTargetView(_texture, &rtDesc, &_renderTargetView.p)))
    {
        return false;
    }

    // シェーダリソースビューの生成
    D3D11_SHADER_RESOURCE_VIEW_DESC resourceViewDesc = {};
    resourceViewDesc.Format = textureDesc.Format;
    resourceViewDesc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE2D;
    resourceViewDesc.Texture2D.MostDetailedMip = 0;
    resourceViewDesc.Texture2D.MipLevels = textureDesc.MipLevels;
    if (FAILED(device.GetD3DDevice()->CreateShaderResourceView(_texture, &resourceViewDesc, &_shaderResourceView.p)))
    {
        return false;
    }

    // 平行投影行列のバッファを生成
    D3D11_BUFFER_DESC bufferDesc = {};
    bufferDesc.Usage = D3D11_USAGE_DYNAMIC;
    bufferDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
    bufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
    bufferDesc.ByteWidth = sizeof(Matrix4x4);
    if (FAILED(device.GetD3DDevice()->CreateBuffer(&bufferDesc, nullptr, &_orthogonalMatrixBuffer.p)))
    {
        return false;
    }

    // 平行投影行列を書き込み
    Matrix4x4 matrix = kIdentityMatrix4x4;
    matrix.column[0].x = 2.0f / static_cast<float>(width);
    matrix.column[1].y = 2.0f / -static_cast<float>(height);
    matrix.column[3] = Vector4(-1.0f, 1.0f, 0.0f, 1.0f);
    D3D11_MAPPED_SUBRESOURCE resource;
    if (FAILED(device.GetContext()->Map(_orthogonalMatrixBuffer.p, 0, D3D11_MAP_WRITE_DISCARD, 0, &resource)))
    {
        return false;
    }
    memcpy(resource.pData, &matrix, sizeof(Matrix4x4));
    device.GetContext()->Unmap(_orthogonalMatrixBuffer.p, 0);

    // 情報を保持
    _width = width;
    _height = height;
    _pixelFormat = GetPixelFormat(textureDesc.Format);

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      リソースの破棄
 */
/* ------------------------------------------------------------------------- */
void TextureResource::Destroy() noexcept 
{
    if (IsValid() && !_isMainRenderTarget)
    {
        _shaderResourceView.Release();
        _renderTargetView.Release();
        _texture.Release();
    }
}


/* ------------------------------------------------------------------------- */
/*!
　*  \brief      テクスチャのサイズを取得
　*  \return     テクスチャのサイズ
　*/
/* ------------------------------------------------------------------------- */
Vector2 TextureResource::GetSize() const noexcept
{
    if (_isMainRenderTarget)
    {
        return Device::GetInstance().GetMainRenderTargetSize();
    }

    return {static_cast<float>(_width), static_cast<float>(_height)};
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャのピクセルフォーマットを取得
 *  \return     テクスチャのピクセルフォーマット
 */
/* ------------------------------------------------------------------------- */
PixelFormat TextureResource::GetPixelFormat() const noexcept
{
    if (_isMainRenderTarget)
    {
        return GetPixelFormat(Device::GetInstance().GetMainRenderTargetPixelFormat());
    }

    return _pixelFormat;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      MGLのピクセルフォーマットからDirectXのピクセルフォーマットに変換
 *  \param[in]  pixelFormat     変換元のピクセルフォーマット
 *  \return     DirectXのピクセルフォーマット
 */
/* ------------------------------------------------------------------------- */
DXGI_FORMAT TextureResource::GetDXPixelFormat(PixelFormat pixelFormat) noexcept
{
    switch (pixelFormat)
    {
        case PixelFormat::RGBA8_UNorm:
            return DXGI_FORMAT_R8G8B8A8_UNORM;

        case PixelFormat::BGRA8_UNorm:
            return DXGI_FORMAT_B8G8R8A8_UNORM;

        case PixelFormat::B5G6R5_UNorm:
            return DXGI_FORMAT_B5G6R5_UNORM;

        case PixelFormat::A1BGR5_UNorm:
            break;

        case PixelFormat::BGR5A1_UNorm:
            return DXGI_FORMAT_B5G5R5A1_UNORM;

        case PixelFormat::ABGR4_UNorm:
            break;

        case PixelFormat::BGRA4_UNorm:
            return DXGI_FORMAT_B4G4R4A4_UNORM;

        case PixelFormat::A8_UNorm:
            return DXGI_FORMAT_A8_UNORM;

        case PixelFormat::Unknown:
            break;
    }

    return DXGI_FORMAT_UNKNOWN;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      DirectXのピクセルフォーマットからMGLのピクセルフォーマットに変換
 *  \param[in]  dxPixelFormat   変換元のピクセルフォーマット
 *  \return     MGLのピクセルフォーマット
 */
/* ------------------------------------------------------------------------- */
PixelFormat TextureResource::GetPixelFormat(DXGI_FORMAT dxPixelFormat) noexcept
{
    switch (dxPixelFormat)
    {
        case DXGI_FORMAT_R8G8B8A8_UNORM:
            return PixelFormat::RGBA8_UNorm;

        case DXGI_FORMAT_B8G8R8A8_UNORM:
            return PixelFormat::BGRA8_UNorm;

        case DXGI_FORMAT_B5G6R5_UNORM:
            return PixelFormat::B5G6R5_UNorm;

        case DXGI_FORMAT_B5G5R5A1_UNORM:
            return PixelFormat::BGR5A1_UNorm;

        case DXGI_FORMAT_B4G4R4A4_UNORM:
            return PixelFormat::BGRA4_UNorm;

        case DXGI_FORMAT_A8_UNORM:
            return PixelFormat::A8_UNorm;

        default:
            return PixelFormat::Unknown;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      平行投影行列のバッファを取得
 *  \return     平行投影行列のバッファ
 */
/* ------------------------------------------------------------------------- */
ID3D11Buffer *TextureResource::GetOrthogonalMatrixBuffer() const noexcept
{
    if (_isMainRenderTarget)
    {
        return Device::GetInstance().GetMainOrthogonalMatrixBuffer();
    }

    return _orthogonalMatrixBuffer;
}


}   // namespace MGL::Render::D3D11

#endif  // MGL_RENDERER_ENABLE_D3D11

// vim: et ts=4 sw=4 sts=4
