// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_d3d11_sprite_renderer.cc
 *  \brief      MGL Direct3D11用スプライトレンダラ
 *  \date       Since: March 6, 2021. 15:44:44 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/d3d11/mgl_d3d11_sprite_renderer.h>
#if defined(MGL_RENDERER_ENABLE_D3D11)

#include <algorithm>
#include <mgl/render/d3d11/mgl_d3d11_defs.h>
#include <mgl/render/d3d11/mgl_d3d11_device.h>
#include <mgl/render/d3d11/mgl_d3d11_texture_resource.h>
#include <mgl/system/mgl_system_debug_macro.h>

#include "shader/sprite.h"

namespace MGL::Render::D3D11
{
namespace
{
//! 頂点バッファ情報
struct Vertex
{
    float x;
    float y;
    int texIndex[2];
};

constexpr size_t kVertexCountPerAttribute = 4; //! 矩形1つあたりの頂点数

//! 矩形の頂点情報
const Vertex s_Vertex[kVertexCountPerAttribute] =
{
    {0.0f, 0.0f, {0, 1}},
    {1.0f, 0.0f, {2, 1}},
    {0.0f, 1.0f, {0, 3}},
    {1.0f, 1.0f, {2, 3}}
};
}   // namespace

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
SpriteRenderer::SpriteRenderer() noexcept
    : _isAvailable(false)
    , _attributes()
    , _attributeCount(0)
    , _vertexBuffer(nullptr)
    , _pixelShader(nullptr)
    , _inputLayout(nullptr)
    , _currentTexture(nullptr)
    , _currentSamplerType(SamplerType::Invalid)
    , _isEnabledBlending(true)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool SpriteRenderer::Initialize() noexcept
{
    // 初期化済みであれば実行しない
    if (_isAvailable)
    {
        return true;
    }

    // シェーダの初期化
    if (!InitializeShader())
    {
        MGL_ASSERT(0, "[MGL Direct3D11] SpriteRenderer: Failed to initialize shader.");
        return false;
    }

    // 頂点バッファの初期化
    if (!InitializeVertexBuffer())
    {
        MGL_ASSERT(0, "[MGL Direct3D11] SpriteRenderer: Failed to initialize vertex buffer.");
        return false;
    }

    // サンプラーステートの初期化
    if (!InitializeSamplerState())
    {
        MGL_ASSERT(0, "[MGL Direct3D11] SpriteRenderer: Failed to initialize sampler state.");
        return false;
    }

    _isAvailable = true;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      描画の実行
 *  \param[in]  flushParameter  描画パラメータ
 */
/* ------------------------------------------------------------------------- */
void SpriteRenderer::Flush(const FlushParameter &flushParameter) noexcept
{
    // 初期化前は実行しない
    if (!_isAvailable)
    {
        return;
    }

    // アトリビュートが追加されていなければ何もしない
    if (_attributeCount == 0)
    {
        return;
    }

    // テクスチャを取得
    if (_currentTexture->GetRendererType() != kRendererTypeDirect3D11)
    {
        _attributeCount = 0;
        _currentTexture = nullptr;
        _currentSamplerType = SamplerType::Invalid;
        return;
    }
    auto *texture = static_cast<D3D11::TextureResource *>(_currentTexture.get());

    auto &device = Device::GetInstance();
    auto *context = device.GetContext();

    // 定数バッファを取得してAttributeを書き込み
    const size_t requestSize = _attributeCount * sizeof(Attribute);
    auto *attributeBuffer = device.GetConstantBuffer(requestSize);
    context->UpdateSubresource(attributeBuffer, 0, nullptr, _attributes.data(), 0, 0);

    // コンテキストに各種設定
    context->IASetInputLayout(_inputLayout);
    const UINT stride = sizeof(Vertex);
    const UINT offset = 0;
    context->IASetVertexBuffers(0, 1, &_vertexBuffer.p, &stride, &offset);
    context->IASetPrimitiveTopology(D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    ID3D11VertexShader *vertexShader = GetVertexShader(requestSize);
    context->VSSetShader(vertexShader, nullptr, 0);
    context->PSSetShader(_pixelShader, nullptr, 0);
    ID3D11Buffer *constBuffer[] =
    {
        flushParameter.matrix,
        attributeBuffer
    };
    context->VSSetConstantBuffers(0, 2, constBuffer);
    context->RSSetState(flushParameter.rasterizerState);
    if (_isEnabledBlending)
    {
        float blendFactor[4] = { 0.0f, 0.0f, 0.0f, 0.0f };
        context->OMSetBlendState(flushParameter.blendState, blendFactor, 0xFFFFFFFF);
    }
    else
    {
        context->OMSetBlendState(nullptr, nullptr, 0xFFFFFFFF);        
    }
    auto *shaderResourceView = texture->GetShaderResourceView();
    context->PSSetShaderResources(0, 1, &shaderResourceView);
    auto *sampler = GetSamplerState(_currentSamplerType);
    context->PSSetSamplers(0, 1, &sampler);

    // 描画実行
    context->DrawInstanced(static_cast<UINT>(kVertexCountPerAttribute), static_cast<UINT>(_attributeCount), 0, 0);

    // 実行後にはシェーダリソースをクリアしておく
    ID3D11ShaderResourceView *nullView = nullptr;
    context->PSSetShaderResources(0, 1, &nullView);

    // 描画実行後はパラメータをリセットする
    _attributeCount = 0;
    _currentTexture = nullptr;
    _currentSamplerType = SamplerType::Invalid;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アトリビュートの追加
 *  \param[in]  attribute           追加するアトリビュート
 *  \param[in]  flushParameter      描画パラメータ
 *  \param[in]  texture             使用するテクスチャリソース
 *  \param[in]  samplerType         使用するサンプラーの種類
 *  \param[in]  isEnabledBlending   ブレンド設定の有効フラグ
 */
/* ------------------------------------------------------------------------- */
void SpriteRenderer::Add(const Attribute &attribute, const FlushParameter &flushParameter, SharedTextureResource &texture, SamplerType samplerType, bool isEnabledBlending) noexcept
{
    // 初期化前は実行しない
    if (!_isAvailable)
    {
        return;
    }

    // これ以上追加できなかったりテクスチャやサンプラーに変更があった場合は描画を実行する
    if (IsFull() ||
        (_currentTexture != texture) ||
        (_currentSamplerType != samplerType) ||
        (_isEnabledBlending != isEnabledBlending))
    {
        Flush(flushParameter);
    }

    // アトリビュートを追加
    _attributes[_attributeCount] = attribute;
    _attributeCount++;

    // テクスチャとサンプラーの種類を保持しておく
    _currentTexture = texture;
    _currentSamplerType = samplerType;
    _isEnabledBlending = isEnabledBlending;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      シェーダの初期化
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool SpriteRenderer::InitializeShader() noexcept
{
    auto &device = Device::GetInstance();

    // 頂点シェーダ
    for (const auto &vs : s_VS_sprite)
    {
        VertexShader vertexShader = {};
        if (SUCCEEDED(device.GetD3DDevice()->CreateVertexShader(vs.binary, vs.binarySize, nullptr, &vertexShader.shader.p)))
        {
            vertexShader.size = vs.attributeSize * sizeof(Attribute);
            _vertexShaderArray.push_back(vertexShader);
        }
        else
        {
            return false;
        }
    }
    // 頂点シェーダのテーブルはlower_bound()のためにソートしておく
    std::sort(
        _vertexShaderArray.begin(),
        _vertexShaderArray.end(),
        [](const VertexShader &lhs, const VertexShader &rhs)
        {
            return lhs.size < rhs.size;
        }
    );

    // 入力レイアウトオブジェクトも生成
    D3D11_INPUT_ELEMENT_DESC layout[] =
    {
        { "POSITION",  0, DXGI_FORMAT_R32G32_FLOAT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0 },
        { "TEX_INDEX", 0, DXGI_FORMAT_R32G32_UINT,  0, 8, D3D11_INPUT_PER_VERTEX_DATA, 0 }
    };
    constexpr UINT elementCount = sizeof(layout) / sizeof(D3D11_INPUT_ELEMENT_DESC);
    device.GetD3DDevice()->CreateInputLayout(layout, elementCount, s_VS_sprite[0].binary, s_VS_sprite[0].binarySize, &_inputLayout.p);
    if (_inputLayout == nullptr)
    {
        return false;
    }

    // ピクセルシェーダ
    device.GetD3DDevice()->CreatePixelShader(g_PS, sizeof(g_PS), nullptr, &_pixelShader.p);

    return _pixelShader != nullptr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      頂点バッファの初期化
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool SpriteRenderer::InitializeVertexBuffer() noexcept
{
    auto &device = Device::GetInstance();

    // 生成する頂点バッファの設定
    D3D11_BUFFER_DESC desc = {};
    desc.Usage = D3D11_USAGE_DYNAMIC;
    desc.BindFlags = D3D11_BIND_VERTEX_BUFFER;
    desc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
    desc.ByteWidth = sizeof(Vertex) * kVertexCountPerAttribute;

    // 関連付ける頂点情報の設定
    D3D11_SUBRESOURCE_DATA subData;
    subData.pSysMem = s_Vertex;
    subData.SysMemPitch = 0;
    subData.SysMemSlicePitch = 0;

    // 生成
    return SUCCEEDED(device.GetD3DDevice()->CreateBuffer(&desc, &subData, &_vertexBuffer.p));
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サンプラーステートの初期化
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool SpriteRenderer::InitializeSamplerState() noexcept
{
    auto &device = Device::GetInstance();

    D3D11_SAMPLER_DESC desc = {};
    desc.Filter = D3D11_FILTER_MIN_MAG_MIP_POINT;
    desc.AddressU = D3D11_TEXTURE_ADDRESS_WRAP;
    desc.AddressV = D3D11_TEXTURE_ADDRESS_WRAP;
    desc.AddressW = D3D11_TEXTURE_ADDRESS_WRAP;
    desc.ComparisonFunc = D3D11_COMPARISON_NEVER;
    desc.MinLOD = 0;
    desc.MaxLOD = D3D11_FLOAT32_MAX;

    for (size_t i = 0; i < kSamplerStateMax; ++i)
    {
        switch (i)
        {
            case kSamplerStateNearest:
                desc.Filter = D3D11_FILTER_MIN_MAG_MIP_POINT;
                break;

            case kSamplerStateLinear:
                desc.Filter = D3D11_FILTER_MIN_MAG_MIP_LINEAR;
                break;

            default:
                continue;
        }
        if (FAILED(device.GetD3DDevice()->CreateSamplerState(&desc, &_samplers[i].p)))
        {
            return false;
        }
    }

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サンプラーステートを取得
 *  \param[in]  samplerType  サンプラータイプ
 *  \return     引数に対応したサンプラーステート
 */
/* ------------------------------------------------------------------------- */
ID3D11SamplerState *SpriteRenderer::GetSamplerState(SamplerType samplerType) noexcept
{
    switch (samplerType)
    {
        case SamplerType::Nearest:
            return _samplers[kSamplerStateNearest];

        case SamplerType::Linear:
            return _samplers[kSamplerStateLinear];

        default:
            break;
    }

    MGL_ASSERT(0, "[MGL Direct3D11] Sampler %d was not supported.", static_cast<int>(samplerType));

    return nullptr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      頂点シェーダの取得
 *  \param[in]  size    要求する定数バッファのサイズ
 *  \return     サイズに対応したシェーダ
 */
/* ------------------------------------------------------------------------- */
ID3D11VertexShader *SpriteRenderer::GetVertexShader(size_t size) noexcept
{
    auto it = std::lower_bound(
        _vertexShaderArray.begin(),
        _vertexShaderArray.end(),
        VertexShader{ size, nullptr },
        [](const VertexShader &lhs, const VertexShader &rhs)
        {
            return lhs.size < rhs.size;
        }
    );
    if (it == _vertexShaderArray.end())
    {
        return nullptr;
    }

    return it->shader;
}
}   // namespace MGL::Render::D3D11
#endif  // MGL_RENDERER_ENABLE_D3D11

// vim: et ts=4 sw=4 sts=4
