// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_d3d11_renderer_2d.cc
 *  \brief      MGL Direct3D11 2Dレンダラ
 *  \date       Since: March 1, 2021. 15:43:49 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/d3d11/mgl_d3d11_renderer_2d.h>
#include <mgl/render/d3d11/mgl_d3d11_defs.h>
#include <mgl/render/d3d11/mgl_d3d11_device.h>
#include <mgl/render/d3d11/mgl_d3d11_texture_resource.h>

#if defined(MGL_RENDERER_ENABLE_D3D11)
namespace MGL::Render::D3D11
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
Renderer2D::Renderer2D() noexcept
    : _mainRasterizerState(nullptr)
    , _clippedRasterizerState(nullptr)
    , _blendState(nullptr)
    , _isEnabledScissor(false)
    , _mainRenderTarget(STL::make_shared<D3D11::TextureResource>(true))
    , _currentRenderTarget(_mainRenderTarget)
    , _currentObjectRenderer(nullptr)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief  レンダラが初期化可能な状態かを取得
 *  \retval true    初期化可能
 *  \retval false   初期化不可能
 */
/* ------------------------------------------------------------------------- */
bool Renderer2D::CanInitialize() const noexcept
{
    return true;    // 常にtrue
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief  初期化処理
 *  \retval true    成功
 *  \retval false   失敗
 */
/* ------------------------------------------------------------------------- */
bool Renderer2D::Initialize() noexcept
{
    // デバイスが初期化されていなければここで初期化
    if (!Device::HasInstance())
    {
        Device::CreateInstance();
        if (!Device::GetInstance().Initialize())
        {
            return false;
        }
    }

    // ラスタライザステートの初期化
    if (!InitializeRasterizerState())
    {
        return false;
    }

    // ブレンドステートの初期化
    if (!InitializeBlendState())
    {
        return false;
    }

    // オブジェクトレンダラを初期化
    ObjectRenderer *objectRenderers[] = 
    {
        &_lineRenderer,
        &_rectangleRenderer,
        &_spriteRenderer
    };
    for (auto *objectRenderer : objectRenderers)
    {
        if (!objectRenderer->Initialize())
        {
            return false;
        }
    }

    SetScissor(false, {});

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief  描画の実行
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::Flush() noexcept
{
    ChangeObjectRenderer(nullptr);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      描画領域のクリア
 *  \param[in]  color   クリアする色
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::Clear(const Color &color) noexcept
{
    Device::GetInstance().ClearRenderTarget(color);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ラインの描画
 *  \param[in]  start   開始点
 *  \param[in]  end     終了点
 *  \param[in]  color   描画する色
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::DrawLine(const Vector2 &start, const Vector2 &end, const Color &color) noexcept
{
    // 使用するオブジェクトレンダラを変更
    ChangeObjectRenderer(&_lineRenderer);

    _lineRenderer.Add(start, end, color, GetFlushParameter());
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      矩形の描画
 *  \param[in]  rectangle   描画する矩形
 *  \param[in]  color       描画する色
 *  \param[in]  option      描画オプション
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::DrawRectangle(const Rectangle &rectangle, const Color &color, const DrawOption2D &option) noexcept
{
    // 使用するオブジェクトレンダラを変更
    ChangeObjectRenderer(&_rectangleRenderer);

    RectangleRenderer::Attribute attribute = {};

    // オフセットを設定
    auto offset = option.GetOffset();
    attribute.offset[0] = rectangle.x + offset.x;
    attribute.offset[1] = rectangle.y + offset.y;
    attribute.offset[2] = attribute.offset[3] = 0.0f;
    ApplyAnchor(attribute.offset, rectangle.GetSize(), option);

    // スケールを設定
    auto scale = option.GetScale();
    attribute.scale[0] = rectangle.width * scale.x;
    attribute.scale[1] = rectangle.height * scale.y;
    attribute.scale[2] = attribute.scale[3] = 0.0f;

    // 補間
    attribute.scale[0] += 0.0001f;
    attribute.scale[1] += 0.0001f;

    // 色を設定
    auto maskColor = option.GetMaskColor();
    attribute.color[0] = color.red * maskColor.red;
    attribute.color[1] = color.green * maskColor.green;
    attribute.color[2] = color.blue * maskColor.blue;
    attribute.color[3] = color.alpha * maskColor.alpha;

    // 回転角度を設定
    attribute.rotate = option.GetRotate() * Math::kDegreeToRadian;
    attribute.pivot[0] = option.GetPivot().x;
    attribute.pivot[1] = option.GetPivot().y;

    // レンダラに追加
    _rectangleRenderer.Add(attribute, GetFlushParameter());
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スプライトの描画
 *  \param[in]  textureResource     描画するテクスチャのリソース
 *  \param[in]  position            描画する位置
 *  \param[in]  sourceRectangle     描画するテクスチャの領域
 *  \param[in]  option              描画オプション
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::DrawSprite(
        const Vector2 &position,
        SharedTextureResource textureResource,
        const Rectangle &sourceRectangle,
        const DrawOption2D &option) noexcept
{
    if (textureResource == nullptr)
    {
        return;
    }

    if (!textureResource->IsValid())
    {
        return;
    }
    
    // 使用するオブジェクトレンダラを変更
    ChangeObjectRenderer(&_spriteRenderer);

    SpriteRenderer::Attribute attribute = {};

    // オフセットを設定
    auto offset = option.GetOffset();
    attribute.offset[0] = position.x + offset.x;
    attribute.offset[1] = position.y + offset.y;
    ApplyAnchor(attribute.offset, sourceRectangle.GetSize(), option);

    // スケールを設定
    auto scale = option.GetScale();
    attribute.scale[0] = sourceRectangle.width * scale.x;
    attribute.scale[1] = sourceRectangle.height * scale.y;

    // 補間
    attribute.scale[0] += 0.0001f;
    attribute.scale[1] += 0.0001f;

    // 色を設定
    auto maskColor = option.GetMaskColor();
    attribute.color[0] = maskColor.red;
    attribute.color[1] = maskColor.green;
    attribute.color[2] = maskColor.blue;
    attribute.color[3] = maskColor.alpha;

    // 回転角度を設定
    attribute.rotate = option.GetRotate() * Math::kDegreeToRadian;
    attribute.pivot[0] = option.GetPivot().x;
    attribute.pivot[1] = option.GetPivot().y;

    // フリップ設定
    // NOTE:
    //  下位2bitで0から3のインデックスを設定して，それを
    //  シェーダ側のテーブルへのインデックスとして扱う．
    if (option.IsHorizontalFlip())
    {
        attribute.flip |= 1u;
    }
    if (option.IsVerticalFlip())
    {
        attribute.flip |= 2u;
    }

    // フリップの際に回転方向が反対にならないよう対処
    if (option.IsHorizontalFlip() != option.IsVerticalFlip())
    {
        attribute.rotate = -attribute.rotate;
    }

    // テクスチャ座標を設定
    auto textureSize = textureResource->GetSize();
    attribute.textureCoord[0] = sourceRectangle.x / textureSize.x;
    attribute.textureCoord[1] = sourceRectangle.y / textureSize.y;
    attribute.textureCoord[2] = attribute.textureCoord[0] + sourceRectangle.width / textureSize.x;
    attribute.textureCoord[3] = attribute.textureCoord[1] + sourceRectangle.height / textureSize.y;

    // レンダラに追加
    _spriteRenderer.Add(attribute, GetFlushParameter(), textureResource, option.GetSamplerType(), option.GetBlendMode() != BlendMode::None);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      レンダーターゲットの設定
 *  \param[in]  renderTarget    設定するレンダーターゲット
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool Renderer2D::SetRenderTarget(SharedTextureResource renderTarget) noexcept
{
    if (_currentRenderTarget == renderTarget)
    {
        return true;
    }

    if (renderTarget == nullptr)
    {
        return false;
    }

    if (!renderTarget->IsRenderTarget())
    {
        return false;
    }

    if (renderTarget->GetRendererType() != kRendererTypeDirect3D11)
    {
        return false;
    }

    Flush();

    auto *textureResource = static_cast<D3D11::TextureResource *>(renderTarget.get());
    auto size = textureResource->GetSize();
    Device::GetInstance().SetRenderTarget(textureResource->GetRenderTargetView(), size.x, size.y);

    _currentRenderTarget = renderTarget;
    
    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      現在のレンダーターゲットを取得
 *  \return     現在のレンダーターゲット
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource Renderer2D::GetRenderTarget() const noexcept
{
    return _currentRenderTarget;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      メインレンダーターゲットを取得
 *  \return     メインレンダーターゲット
 */
/* ------------------------------------------------------------------------- */
SharedTextureResource Renderer2D::GetMainRenderTarget() const noexcept
{
    return _mainRenderTarget;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      シザーを設定
 *  \param[in]  isEnabled   有効フラグ
 *  \param[in]  rectangle   設定する矩形
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::SetScissor(bool isEnabled, const Rectangle &rectangle) noexcept
{
    const bool isSameEnable = _isEnabledScissor == isEnabled;

    // 設定済みのフラグと設定しようとしているフラグがどちらも無効であれば何もしない
    if (isSameEnable && !isEnabled)
    {
        return;
    }

    // 設定されているパラメータが同一であれば何もしない
    if (isSameEnable)
    {
        if ((rectangle.x == _scissorRectangle.x) &&
            (rectangle.y == _scissorRectangle.y) &&
            (rectangle.width == _scissorRectangle.width) &&
            (rectangle.height == _scissorRectangle.height))
        {
            return;
        }
    }

    // 変更前に一旦描画を実行する
    Flush();

    // 変更を反映
    if (isEnabled)
    {
        D3D11_RECT rect;
        rect.top = static_cast<LONG>(rectangle.y);
        rect.bottom = static_cast<LONG>(rectangle.y + rectangle.height);
        rect.left = static_cast<LONG>(rectangle.x);
        rect.right = static_cast<LONG>(rectangle.x + rectangle.width);

        Device::GetInstance().GetContext()->RSSetScissorRects(1, &rect);
    }
    else
    {
        Device::GetInstance().GetContext()->RSSetScissorRects(0, nullptr);
    }

    _isEnabledScissor = isEnabled;
    _scissorRectangle = rectangle;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      シザーが有効かどうかを取得
 *  \retval     true    有効
 *  \retval     false   無効
 */
/* ------------------------------------------------------------------------- */
bool Renderer2D::IsEnabledScissor() const noexcept
{
    return _isEnabledScissor;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      シザー矩形を取得
 *  \return     シザー矩形
 */
/* ------------------------------------------------------------------------- */
Rectangle Renderer2D::GetScissorRectangle() const noexcept
{
    return _scissorRectangle;
}


/* ----------------------------------------------------------------------------- */
/**
 *  \brief	    ラスタライザの初期化
 *  \retval	    true	成功
 *  \retval	    false	失敗
 */
/* ----------------------------------------------------------------------------- */
bool Renderer2D::InitializeRasterizerState() noexcept
{
    D3D11_RASTERIZER_DESC desc = {};
    desc.AntialiasedLineEnable = false;
    desc.CullMode = D3D11_CULL_NONE;
    desc.DepthBias = 0;
    desc.DepthBiasClamp = 0.0f;
    desc.DepthClipEnable = true;
    desc.FillMode = D3D11_FILL_SOLID;
    desc.FrontCounterClockwise = false;
    desc.MultisampleEnable = false;
    desc.ScissorEnable = false;
    desc.SlopeScaledDepthBias = 0.0f;

    auto &device = Device::GetInstance();

    if (FAILED(device.GetD3DDevice()->CreateRasterizerState(&desc, &_mainRasterizerState)))
    {
        return false;
    }

    desc.ScissorEnable = true;
    return SUCCEEDED(device.GetD3DDevice()->CreateRasterizerState(&desc, &_clippedRasterizerState));
}


/* ----------------------------------------------------------------------------- */
/**
 *  \brief	    ブレンドステートの初期化
 *  \retval	    true	成功
 *  \retval	    false	失敗
 */
/* ----------------------------------------------------------------------------- */
bool Renderer2D::InitializeBlendState() noexcept
{
    D3D11_BLEND_DESC desc = {};
    desc.AlphaToCoverageEnable = false;
    desc.IndependentBlendEnable = false;
    desc.RenderTarget[0].BlendEnable = true;
    desc.RenderTarget[0].SrcBlend = D3D11_BLEND_SRC_ALPHA;
    desc.RenderTarget[0].DestBlend = D3D11_BLEND_INV_SRC_ALPHA;
    desc.RenderTarget[0].BlendOp = D3D11_BLEND_OP_ADD;
    desc.RenderTarget[0].SrcBlendAlpha = D3D11_BLEND_ONE;
    desc.RenderTarget[0].DestBlendAlpha = D3D11_BLEND_ZERO;
    desc.RenderTarget[0].BlendOpAlpha = D3D11_BLEND_OP_ADD;
    desc.RenderTarget[0].RenderTargetWriteMask = D3D11_COLOR_WRITE_ENABLE_ALL;

    auto &device = Device::GetInstance();

    return SUCCEEDED(device.GetD3DDevice()->CreateBlendState(&desc, &_blendState));
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      描画の実行に必要なパラメータを取得
 *  \return     描画の実行に必要なパラメータ
 */
/* ------------------------------------------------------------------------- */
ObjectRenderer::FlushParameter Renderer2D::GetFlushParameter() noexcept
{
    ObjectRenderer::FlushParameter flushParameter = {};
    flushParameter.matrix = GetOrthogonalMatrixBuffer();
    flushParameter.rasterizerState = (!_isEnabledScissor) ? _mainRasterizerState : _clippedRasterizerState;
    flushParameter.blendState = _blendState;

    return flushParameter;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      現在の平行投影行列を取得
 *  \return     平行投影行列
 */
/* ------------------------------------------------------------------------- */
ID3D11Buffer *Renderer2D::GetOrthogonalMatrixBuffer() const noexcept
{
    if (_currentRenderTarget->GetRendererType() != kRendererTypeDirect3D11)
    {
        return nullptr;
    }

    auto *renderTarget = static_cast<D3D11::TextureResource *>(_currentRenderTarget.get());

    return renderTarget->GetOrthogonalMatrixBuffer();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      オブジェクトレンダラの変更
 *  \param[in]  objectRenderer  新たに設定するオブジェクトレンダラ
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::ChangeObjectRenderer(ObjectRenderer *objectRenderer) noexcept
{
    if (objectRenderer != _currentObjectRenderer)
    {
        if (_currentObjectRenderer != nullptr)
        {
            _currentObjectRenderer->Flush(GetFlushParameter());
        }
    }
    
    _currentObjectRenderer = objectRenderer;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          オフセットにアンカーを反映
 *  \param[in,out]  offset  反映させるオフセット
 *  \param[in]      size    矩形のサイズ
 *  \param[in]      option  描画オプション
 */
/* ------------------------------------------------------------------------- */
void Renderer2D::ApplyAnchor(float *offset, const Vector2 &size, const DrawOption2D &option) noexcept
{
    auto scale = option.GetScale();
    auto width = size.x * scale.x;
    auto height = size.y * scale.y;

    // 水平方向の補正
    switch (option.GetAlignment().horizontal)
    {
        case Alignment::Horizontal::Center:
            offset[0] -= width * 0.5f;
            break;

        case Alignment::Horizontal::Right:
            offset[0] -= width;
            break;

        default:
            break;
    }

    // 垂直方向の補正
    switch (option.GetAlignment().vertical)
    {
        case Alignment::Vertical::Middle:
            offset[1] -= height * 0.5f;
            break;

        case Alignment::Vertical::Bottom:
            offset[1] -= height;
            break;

        default:
            break;
    }

    const auto &pivot = option.GetPivot();
    offset[0] += pivot.x * width;
    offset[1] += pivot.y * height;
}

};  // namespace MGL::Render::D3D11
#endif  // MGL_RENDERER_ENABLE_D3D11
// vim: et ts=4 sw=4 sts=4
