// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_d3d11_rectangle_renderer.cc
 *  \brief      MGL Direct3D11用矩形レンダラ
 *  \date       Since: March 5, 2021. 17:17:59 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/d3d11/mgl_d3d11_rectangle_renderer.h>
#if defined(MGL_RENDERER_ENABLE_D3D11)

#include <algorithm>
#include <mgl/render/d3d11/mgl_d3d11_device.h>
#include <mgl/system/mgl_system_debug_macro.h>

#include "shader/rectangle.h"

namespace MGL::Render::D3D11
{
namespace
{
//! 頂点バッファ情報
struct Vertex
{
    float x;
    float y;
};

constexpr size_t kVertexCountPerAttribute = 4; //! 矩形1つあたりの頂点数

//! 矩形の頂点情報
const Vertex s_Vertex[kVertexCountPerAttribute] =
{
    {0.0f, 0.0f},
    {1.0f, 0.0f},
    {0.0f, 1.0f},
    {1.0f, 1.0f}
};
}   // namespace

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
RectangleRenderer::RectangleRenderer() noexcept
    : _isAvailable(false)
    , _attributes()
    , _attributeCount(0)
    , _pixelShader(nullptr)
    , _inputLayout(nullptr)
    , _vertexBuffer(nullptr)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool RectangleRenderer::Initialize() noexcept
{
    // 初期化済みなら何もしない
    if (_isAvailable)
    {
        return true;
    }

    // シェーダの初期化
    if (!InitializeShader())
    {
        MGL_ASSERT(0, "[MGL Direct3D11] RectangleRenderer: Failed to initialize shader.");
        return false;
    }

    // 頂点バッファの初期化
    if (!InitializeVertexBuffer())
    {
        MGL_ASSERT(0, "[MGL Direct3D11] RectangleRenderer: Failed to initialize vertex buffer.");
        return false;
    }

    _isAvailable = true;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      描画の実行
 *  \param[in]  flushParameter  描画パラメータ
 */
/* ------------------------------------------------------------------------- */
void RectangleRenderer::Flush(const FlushParameter &flushParameter) noexcept
{
    // 初期化前は実行しない
    if (!_isAvailable)
    {
        return;
    }

    // 描画するものがなければ実行しない
    if (_attributeCount == 0)
    {
        return;
    }

    auto &device = Device::GetInstance();
    auto *context = device.GetContext();

    // 定数バッファを取得してAttributeを書き込み
    const size_t requestSize = _attributeCount * sizeof(Attribute);
    auto *attributeBuffer = device.GetConstantBuffer(requestSize);
    context->UpdateSubresource(attributeBuffer, 0, nullptr, _attributes.data(), 0, 0);

    // コンテキストに各種設定
    context->IASetInputLayout(_inputLayout);
    const UINT stride = sizeof(Vertex);
    const UINT offset = 0;
    context->IASetVertexBuffers(0, 1, &_vertexBuffer.p, &stride, &offset);
    context->IASetPrimitiveTopology(D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);
    ID3D11VertexShader *vertexShader = GetVertexShader(requestSize);
    context->VSSetShader(vertexShader, nullptr, 0);
    context->PSSetShader(_pixelShader, nullptr, 0);
    ID3D11Buffer *constBuffer[] =
    {
        flushParameter.matrix,
        attributeBuffer,
    };
    context->VSSetConstantBuffers(0, 2, constBuffer);
    context->RSSetState(flushParameter.rasterizerState);
    float blendFactor[4] = { 0.0f, 0.0f, 0.0f, 0.0f };
    context->OMSetBlendState(flushParameter.blendState, blendFactor, 0xFFFFFFFF);

    // 描画実行
    context->DrawInstanced(static_cast<UINT>(kVertexCountPerAttribute), static_cast<UINT>(_attributeCount), 0, 0);

    _attributeCount = 0;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アトリビュートの追加
 *  \param[in]  attribute       追加するアトリビュート
 *  \param[in]  flushParameter  描画パラメータ
 */
/* ------------------------------------------------------------------------- */
void RectangleRenderer::Add(const Attribute &attribute, const FlushParameter &flushParameter) noexcept
{
    // 初期化前は実行しない
    if (!_isAvailable)
    {
        return;
    }

    // これ以上追加できない場合は一旦描画を実行する
    if (IsFull())
    {
        Flush(flushParameter);
    }

    // アトリビュートを追加
    _attributes[_attributeCount] = attribute;
    _attributeCount++;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      シェーダの初期化
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool RectangleRenderer::InitializeShader() noexcept
{
    auto &device = Device::GetInstance();

    // 頂点シェーダ
    for (const auto &vs : s_VS_rectangle)
    {
        VertexShader vertexShader = {};
        if (SUCCEEDED(device.GetD3DDevice()->CreateVertexShader(vs.binary, vs.binarySize, nullptr, &vertexShader.shader.p)))
        {
            vertexShader.size = vs.attributeSize * sizeof(Attribute);
            _vertexShaderArray.push_back(vertexShader);
        }
        else
        {
            return false;
        }
    }
    // 頂点シェーダのテーブルはlower_bound()のためにソートしておく
    std::sort(
        _vertexShaderArray.begin(),
        _vertexShaderArray.end(),
        [](const VertexShader &lhs, const VertexShader &rhs)
        {
            return lhs.size < rhs.size;
        }
    );

    // 入力レイアウトオブジェクトも生成
    D3D11_INPUT_ELEMENT_DESC layout[] =
    {
        { "POSITION", 0, DXGI_FORMAT_R32G32_FLOAT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0 }
    };
    constexpr UINT elementCount = sizeof(layout) / sizeof(D3D11_INPUT_ELEMENT_DESC);
    device.GetD3DDevice()->CreateInputLayout(layout, elementCount, s_VS_rectangle[0].binary, s_VS_rectangle[0].binarySize, &_inputLayout.p);
    if (_inputLayout == nullptr)
    {
        return false;
    }

    // ピクセルシェーダ
    device.GetD3DDevice()->CreatePixelShader(g_PS, sizeof(g_PS), nullptr, &_pixelShader.p);

    return _pixelShader != nullptr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      頂点バッファの初期化
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool RectangleRenderer::InitializeVertexBuffer() noexcept
{
    // 生成する頂点バッファの設定
    D3D11_BUFFER_DESC desc = {};
    desc.Usage = D3D11_USAGE_DYNAMIC;
    desc.BindFlags = D3D11_BIND_VERTEX_BUFFER;
    desc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
    desc.ByteWidth = sizeof(Vertex) * kVertexCountPerAttribute;

    // 関連付ける頂点情報の設定
    D3D11_SUBRESOURCE_DATA subData;
    subData.pSysMem = s_Vertex;
    subData.SysMemPitch = 0;
    subData.SysMemSlicePitch = 0;

    // 生成
    return SUCCEEDED(Device::GetInstance().GetD3DDevice()->CreateBuffer(&desc, &subData, &_vertexBuffer.p));
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      頂点シェーダの取得
 *  \param[in]  size    要求する定数バッファのサイズ
 *  \return     サイズに対応したシェーダ
 */
/* ------------------------------------------------------------------------- */
ID3D11VertexShader *RectangleRenderer::GetVertexShader(size_t size) noexcept
{
    auto it = std::lower_bound(
        _vertexShaderArray.begin(),
        _vertexShaderArray.end(),
        VertexShader{ size, nullptr },
        [](const VertexShader &lhs, const VertexShader &rhs)
        {
            return lhs.size < rhs.size;
        }
    );
    if (it == _vertexShaderArray.end())
    {
        return nullptr;
    }

    return it->shader;
}

}   // namespace MGL::Render::D3D11
#endif  // MGL_RENDERER_ENABLE_D3D11
// vim: et ts=4 sw=4 sts=4
