// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_d3d11_line_renderer.cc
 *  \brief      MGL Direct3D11用ラインレンダラ
 *  \date       Since: April 3, 2021. 14:57:16 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/d3d11/mgl_d3d11_line_renderer.h>
#if defined(MGL_RENDERER_ENABLE_D3D11)

#include <algorithm>
#include <mgl/render/d3d11/mgl_d3d11_device.h>
#include <mgl/system/mgl_system_debug_macro.h>

#include "shader/line_vs.h"
#include "shader/line_ps.h"

namespace MGL::Render::D3D11
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \date       Since: April 3, 2021. 15:05:06 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */
LineRenderer::LineRenderer() noexcept
    : _isAvailable(false)
    , _attributes()
    , _attributeCount(0)
    , _vertexShader(nullptr)
    , _pixelShader(nullptr)
    , _inputLayout(nullptr)
    , _vertexBuffer(nullptr)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool LineRenderer::Initialize() noexcept
{
    // 初期化済みなら何もしない
    if (_isAvailable)
    {
        return true;
    }

    // シェーダの初期化
    if (!InitializeShader())
    {
        MGL_ASSERT(0, "[MGL Direct3D11] LineRenderer: Failed to initialize shader.");
        return false;
    }

    // 頂点バッファの初期化
    if (!InitializeVertexBuffer())
    {
        MGL_ASSERT(0, "[MGL Direct3D11] LineRenderer: Failed to initialize vertex buffer.");
        return false;
    }

    _isAvailable = true;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      描画の実行
 *  \param[in]  flushParameter  描画パラメータ
 */
/* ------------------------------------------------------------------------- */
void LineRenderer::Flush(const FlushParameter &flushParameter) noexcept
{
    // 初期化前は実行しない
    if (!_isAvailable)
    {
        return;
    }

    // 描画するものがなければ実行しない
    if (_attributeCount == 0)
    {
        return;
    }

    auto &device = Device::GetInstance();
    auto *context = device.GetContext();

    // アトリビュートの内容を頂点バッファにコピー
    D3D11_MAPPED_SUBRESOURCE resource;
	if (FAILED(context->Map(_vertexBuffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &resource)))
	{
		_attributeCount = 0;
		return;
	}
    memcpy(resource.pData, _attributes.data(), sizeof(Attribute) * _attributeCount);
    context->Unmap(_vertexBuffer, 0);

    // コンテキストに各種設定
	context->IASetInputLayout(_inputLayout);
	const UINT stride = sizeof(Attribute);
	const UINT offset = 0;
	context->IASetVertexBuffers(0, 1, &_vertexBuffer.p, &stride, &offset);
	context->IASetPrimitiveTopology(D3D11_PRIMITIVE_TOPOLOGY_LINELIST);
	context->VSSetShader(_vertexShader, nullptr, 0);
	context->PSSetShader(_pixelShader, nullptr, 0);
	ID3D11Buffer *constBuffer[] =
	{
		flushParameter.matrix
	};
	context->VSSetConstantBuffers(0, 1, constBuffer);
	context->RSSetState(flushParameter.rasterizerState);
	float blendFactor[4] = { 0.0f, 0.0f, 0.0f, 0.0f };
	context->OMSetBlendState(flushParameter.blendState, blendFactor, 0xFFFFFFFF);

	// 描画実行
	context->Draw(static_cast<UINT>(_attributeCount) * 2, 0);

	_attributeCount = 0;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アトリビュートの追加
 *  \param[in]  start           開始点
 *  \param[in]  end             終了点
 *  \param[in]  color           色
 *  \param[in]  flushParameter  描画パラメータ
 */
/* ------------------------------------------------------------------------- */
void LineRenderer::Add(const Vector2 &start, const Vector2 &end, const Color &color, const FlushParameter &flushParameter) noexcept
{
    // 初期化前は実行しない
    if (!_isAvailable)
    {
        return;
    }

    // これ以上追加できない場合は一旦描画を実行する
    if (IsFull())
    {
        Flush(flushParameter);
    }

    // アトリビュートを追加
    _attributes[_attributeCount] = {start.x, start.y, color.red, color.green, color.blue, color.alpha};
    _attributes[_attributeCount + 1] = {end.x, end.y, color.red, color.green, color.blue, color.alpha};
    _attributeCount += 2;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      シェーダの初期化
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool LineRenderer::InitializeShader() noexcept
{
    auto &device = Device::GetInstance();

    // 頂点シェーダの生成
    if (auto hr = device.GetD3DDevice()->CreateVertexShader(g_VS, sizeof(g_VS), nullptr, &_vertexShader.p); FAILED(hr))
    {
        return false;
    }

    // ピクセルシェーダの生成
    if (auto hr = device.GetD3DDevice()->CreatePixelShader(g_PS, sizeof(g_PS), nullptr, &_pixelShader.p); FAILED(hr))
    {
        return false;
    }

    // 入力レイアウトオブジェクトも生成
    D3D11_INPUT_ELEMENT_DESC layout[] =
    {
	    { "POSITION", 0, DXGI_FORMAT_R32G32_FLOAT,		 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0 },
	    { "COLOR",	  0, DXGI_FORMAT_R32G32B32A32_FLOAT, 0, 8, D3D11_INPUT_PER_VERTEX_DATA, 0 }
    };
    constexpr UINT elementCount = sizeof(layout) / sizeof(D3D11_INPUT_ELEMENT_DESC);
    device.GetD3DDevice()->CreateInputLayout(layout, elementCount, g_VS, sizeof(g_VS), &_inputLayout.p);

    return _inputLayout != nullptr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      頂点バッファの初期化
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool LineRenderer::InitializeVertexBuffer() noexcept
{
    // 生成する頂点バッファの設定
    D3D11_BUFFER_DESC desc = {};
    desc.Usage = D3D11_USAGE_DYNAMIC;
    desc.BindFlags = D3D11_BIND_VERTEX_BUFFER;
    desc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
    desc.ByteWidth = sizeof(Attribute) * kAttributeMax;

    // 生成
    return SUCCEEDED(Device::GetInstance().GetD3DDevice()->CreateBuffer(&desc, nullptr, &_vertexBuffer.p));
}
}   // namespace MGL::Render::D3D11

#endif  // MGL_RENDERER_ENABLE_D3D11
// vim: et ts=4 sw=4 sts=4
