// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_d3d11_device.cc
 *  \brief      MGL Direct3D11 デバイス管理
 *  \date       Since: March 1, 2021. 15:03:37 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/render/d3d11/mgl_d3d11_device.h>
#if defined(MGL_RENDERER_ENABLE_D3D11)

#include <mgl/platform/win32/mgl_win32_window.h>
#include <mgl/math/mgl_matrix4x4.h>
#include <algorithm>

#pragma comment(lib, "d3d11.lib")

namespace MGL::Render::D3D11
{
namespace
{
    constexpr size_t kDynamicVertexBufferSize = 16ull * 1024 * 1024;    //!< 動的頂点バッファのサイズ
    constexpr size_t kConstantBufferSize = 16ull * 4096;                //!< 定数バッファのサイズ（4096ベクターが最大）
    constexpr size_t kConstantBufferMinSize = 64;                       //!< 定数バッファの最小サイズ
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インスタンスの取得
 *  \return     インスタンスの参照
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Device> &Device::GetInstanceRef() noexcept
{
    static STL::unique_ptr<Device> sInstance = nullptr;
    return sInstance;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \date       Since: November 14, 2019. 17:53:10 JST.
 *  \author     Acerola.
 */
/* ------------------------------------------------------------------------- */
Device::Device() noexcept
    : _driverType(D3D_DRIVER_TYPE_NULL)
    , _featureLevel(D3D_FEATURE_LEVEL_11_0)
    , _d3dDevice(nullptr)
    , _d3dDevice1(nullptr)
    , _immediateContext(nullptr)
    , _immediateContext1(nullptr)
    , _swapChain(nullptr)
    , _swapChain1(nullptr)
    , _mainRenderTargetView(nullptr)
    , _mainTexture(nullptr)
    , _mainOrthogonalMatrixBuffer(nullptr)
    , _currentRenderTarget(nullptr)
    , _eventPostFrameUpdate(Event::NotifyType::PostFrameUpdate, OnEventPostFrameUpdate, this)
    , _eventChangeClientSize(Event::NotifyType::ChangeClientSize, OnEventChangeClientSize, this)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デストラクタ
 */
/* ------------------------------------------------------------------------- */
Device::~Device() noexcept
{
    // 解放前にコンテキストのステートは初期化しておいたほうが良いらしい
    if (_immediateContext != nullptr)
    {
        _immediateContext->ClearState();
    }

    // フルスクリーン状態ではスワップチェーンが解放できない
    SetFullscreen(false);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \retval     true		成功
 *  \retval     false		失敗
 */
/* ------------------------------------------------------------------------- */
bool Device::Initialize() noexcept
{
    // デバイス生成用フラグの準備
    UINT createDeviceFlags = 0;
#if defined(MGL_DEBUG)
    createDeviceFlags |= D3D11_CREATE_DEVICE_DEBUG;
#endif

    // 初期化するドライバタイプを指定
    // 上から順に試行して成功したものを使用する
    const D3D_DRIVER_TYPE driverTypes[] =
    {
        D3D_DRIVER_TYPE_HARDWARE,		// ハードウェアドライバ
        D3D_DRIVER_TYPE_WARP,			// Wrapドライバ（高速ソフトウェアドライバ）
        D3D_DRIVER_TYPE_REFERENCE,		// リファレンスドライバ（低速高精度ソフトウェアドライバ）
    };

    // 機能レベルの指定
    // こちらも上から順に試行して対応しているものを使用する
    D3D_FEATURE_LEVEL featureLevels[] =
    {
        D3D_FEATURE_LEVEL_11_1,		// 11.1
        D3D_FEATURE_LEVEL_11_0,		// 11.0
        D3D_FEATURE_LEVEL_10_1,		// 10.1
        D3D_FEATURE_LEVEL_10_0,		// 10.0
    };
    const UINT numFeatureLevels = sizeof(featureLevels) / sizeof(D3D_FEATURE_LEVEL);

    // ドライバタイプを走査してデバイスを生成
    for (auto driverType : driverTypes)
    {
        // デバイス生成
        auto result = D3D11CreateDevice(
                nullptr,
                driverType,
                nullptr,
                createDeviceFlags,
                featureLevels,
                numFeatureLevels,
                D3D11_SDK_VERSION,
                &_d3dDevice.p,
                &_featureLevel,
                &_immediateContext.p
        );

        // 引数が不正と言われた場合、機能レベル11.1を削って再試行する
        // MSDNによるとそういうものらしい
        if (result == E_INVALIDARG)
        {
            result = D3D11CreateDevice(
                    nullptr,
                    driverType,
                    nullptr,
                    createDeviceFlags,
                    &featureLevels[1],
                    numFeatureLevels - 1,
                    D3D11_SDK_VERSION,
                    &_d3dDevice.p,
                    &_featureLevel,
                    &_immediateContext.p
            );
        }

        // 成功していればループを抜ける
        if (SUCCEEDED(result))
        {
            _driverType = driverType;
            break;
        }
    }
    if (_driverType == D3D_DRIVER_TYPE_NULL)
    {
        return false;
    }

    // デバイスからDXGIFactoryを取得
    CComPtr<IDXGIFactory1> dxgiFactory = nullptr;
    CComPtr<IDXGIDevice> dxgiDevice = nullptr;
    bool isGetFactory = false;
    if (SUCCEEDED(_d3dDevice->QueryInterface(IID_PPV_ARGS(&dxgiDevice))))
    {
        CComPtr<IDXGIAdapter> adapter = nullptr;
        if (SUCCEEDED(dxgiDevice->GetAdapter(&adapter)))
        {
            if (SUCCEEDED(adapter->GetParent(IID_PPV_ARGS(&dxgiFactory))))
            {
                isGetFactory = true;
            }
        }
    }
    if (!isGetFactory)
    {
        return false;
    }

    // スワップチェーンの生成
    auto &window = Win32::Window::GetInstance();
    bool isCreatedSwapChain = false;
    CComPtr<IDXGIFactory2> dxgiFactory2 = nullptr;
    if (SUCCEEDED(dxgiFactory->QueryInterface(IID_PPV_ARGS(&dxgiFactory2))))
    {
        // DXGIFactory2が取得できていればDirectX 11.1以降の初期化を行う

        // _d3DDevice1 と _immediateContext1 を取得
        if (SUCCEEDED(_d3dDevice->QueryInterface(IID_PPV_ARGS(&_d3dDevice1))))
        {
            _immediateContext->QueryInterface(IID_PPV_ARGS(&_immediateContext1));
        }

        // スワップチェーンの記述子の初期化
        DXGI_SWAP_CHAIN_DESC1 desc{};
        desc.Width = window.GetWidth();						    // 幅
        desc.Height = window.GetHeight();						// 高さ
        desc.Format = DXGI_FORMAT_B8G8R8A8_UNORM;				// フォーマット
        desc.SampleDesc.Count = 1;								// マルチサンプルの数
        desc.SampleDesc.Quality = 0;							// マルチサンプルのクオリティ
        desc.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;		// バッファの用途
        desc.BufferCount = 1;									// バッファの数

        // スワップチェーンの生成
        if (SUCCEEDED(dxgiFactory2->CreateSwapChainForHwnd(_d3dDevice, window.GetWindowHandler(), &desc, nullptr, nullptr, &_swapChain1.p)))
        {
            // _swapChain も取得しておく
            if (SUCCEEDED(_swapChain1->QueryInterface(IID_PPV_ARGS(&_swapChain))))
            {
                isCreatedSwapChain = true;
            }
        }
    }
    else
    {
        // DXGIFactory2が取得できなければ11.0までの初期化を行う

        // スワップチェーンの記述子を初期化
        DXGI_SWAP_CHAIN_DESC desc{};
        desc.BufferDesc.Width = window.GetWidth();				// 幅
        desc.BufferDesc.Height = window.GetHeight();			// 高さ
        desc.BufferDesc.Format = DXGI_FORMAT_B8G8R8A8_UNORM;	// フォーマット
        desc.BufferDesc.RefreshRate.Numerator = 60;				// リフレッシュレート
        desc.BufferDesc.RefreshRate.Denominator = 1;			// リフレッシュレートに対する更新間隔
        desc.OutputWindow = window.GetWindowHandler();			// 出力先ウィンドウ
        desc.SampleDesc.Count = 1;								// マルチサンプルの数
        desc.SampleDesc.Quality = 0;							// マルチサンプルのクオリティ
        desc.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;		// バッファの用途
        desc.BufferCount = 1;									// バッファの数
        desc.Windowed = true;									// ウィンドウモード

        // スワップチェーンの生成
        if (SUCCEEDED(dxgiFactory->CreateSwapChain(_d3dDevice, &desc, &_swapChain.p)))
        {
            isCreatedSwapChain = true;
        }
    }

    // スワップチェーンの生成に失敗していたら初期化失敗
    if (!isCreatedSwapChain)
    {
        return false;
    }

    // Alt + Enterによるフルスクリーン切り替えは無効化しておく
    dxgiFactory->MakeWindowAssociation(window.GetWindowHandler(), DXGI_MWA_NO_WINDOW_CHANGES | DXGI_MWA_NO_ALT_ENTER);

    // 描画先を生成
    if (FAILED(_swapChain->GetBuffer(0, IID_PPV_ARGS(&_mainTexture))))
    {
        return false;
    }
    if (FAILED(_d3dDevice->CreateRenderTargetView(_mainTexture, nullptr, &_mainRenderTargetView.p)))
    {
        return false;
    }

    // 平行投影行列を生成
    if (!InitializeOrthogonalMatrixBuffer())
    {
        return false;
    }

    // 描画先を設定
    SetRenderTarget(_mainRenderTargetView, static_cast<float>(window.GetWidth()), static_cast<float>(window.GetHeight()));

    // 頂点バッファを生成
    if (!InitializeVertexBuffer())
    {
        return false;
    }

    // 定数バッファを生成
    if (!InitializeConstantBuffer())
    {
        return false;
    }

    // フレーム遅延を1フレームに設定
    SetFrameLatency(1);

    // 初回のみ画面を黒でクリア
    ClearRenderTarget(kColorBlack);

    return true;
}


/* ----------------------------------------------------------------------------- */
/**
 *  \brief		フレーム遅延の最大値を設定
 *  \param[in]	latency		遅延フレーム数（0でシステムデフォルト値）
 *  \retval		true		成功
 *  \retval		false		失敗
 */
/* ----------------------------------------------------------------------------- */
bool Device::SetFrameLatency(uint32_t latency) noexcept
{
    bool isSucceeded = false;

    CComPtr<IDXGIDevice1> dxgiDevice1 = nullptr;
    if (SUCCEEDED(_d3dDevice->QueryInterface(IID_PPV_ARGS(&dxgiDevice1))))
    {
        if (SUCCEEDED(dxgiDevice1->SetMaximumFrameLatency(latency)))
        {
            isSucceeded = true;
        }
    }

    return isSucceeded;
}


/* ----------------------------------------------------------------------------- */
/**
 *  \brief		フルスクリーンモードに切り替える
 *  \param[in]	isEnabled   trueでフルスクリーン
 *  \retval		true		成功
 *  \retval		false		失敗
 */
/* ----------------------------------------------------------------------------- */
bool Device::SetFullscreen(bool isEnabled) noexcept
{
    bool isSucceeded = false;

    if (_swapChain != nullptr)
    {
        if (SUCCEEDED(_swapChain->SetFullscreenState(isEnabled, nullptr)))
        {
            isSucceeded = true;
        }
    }

    return isSucceeded;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フルスクリーン状態の取得
 *  \retval     true    フルスクリーン
 *  \retval     false   ウィンドウ
 */
/* ------------------------------------------------------------------------- */
bool Device::IsFullscreen() const noexcept
{
    if (_swapChain == nullptr)
    {
        return false;
    }

    BOOL isFullscreen = FALSE;
    if (SUCCEEDED(_swapChain->GetFullscreenState(&isFullscreen, nullptr)))
    {
        return isFullscreen;
    }

    return false;
}


/* ----------------------------------------------------------------------------- */
/**
 *  \brief		レンダーターゲットのクリア
 *  \param[in]	color	クリアする色
 */
/* ----------------------------------------------------------------------------- */
void Device::ClearRenderTarget(const Color& color) noexcept
{
    float colors[] =
    {
        color.red, color.green, color.blue, 1.0f
    };

    _immediateContext->ClearRenderTargetView(_currentRenderTarget, colors);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief		描画の更新
 */
/* ------------------------------------------------------------------------- */
void Device::Present() noexcept
{
    _swapChain->Present(1, 0);
}


/* ----------------------------------------------------------------------------- */
/**
 *  \brief		平行投影行列のバッファを初期化
 *  \retval		true		成功
 *  \retval		false		失敗
 */
/* ----------------------------------------------------------------------------- */
bool Device::InitializeOrthogonalMatrixBuffer() noexcept
{
    D3D11_BUFFER_DESC desc = {};

    desc.Usage = D3D11_USAGE_DYNAMIC;
    desc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
    desc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
    desc.ByteWidth = sizeof(Matrix4x4);

    if (FAILED(_d3dDevice->CreateBuffer(&desc, nullptr, &_mainOrthogonalMatrixBuffer.p)))
    {
        return false;
    }

    UpdateOrthogonalMatrix();

    return true;
}


/* ----------------------------------------------------------------------------- */
/**
 *  \brief		頂点バッファを初期化
 *  \retval		true		成功
 *  \retval		false		失敗
 */
/* ----------------------------------------------------------------------------- */
bool Device::InitializeVertexBuffer() noexcept
{
    D3D11_BUFFER_DESC desc{};
    desc.Usage = D3D11_USAGE_DYNAMIC;
    desc.ByteWidth = kDynamicVertexBufferSize;
    desc.BindFlags = D3D11_BIND_VERTEX_BUFFER;
    desc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;

    if (FAILED(_d3dDevice->CreateBuffer(&desc, nullptr, &_vertexBuffer.buffer.p)))
    {
        return false;
    }
    _vertexBuffer.offset = 0;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      定数バッファの初期化
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool Device::InitializeConstantBuffer() noexcept
{
    size_t size = kConstantBufferSize;

    D3D11_BUFFER_DESC desc{};
    desc.Usage = D3D11_USAGE_DEFAULT;
    desc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
    desc.CPUAccessFlags = 0;

    while (size >= 16)
    {
        desc.ByteWidth = static_cast<UINT>(size);

        ConstBuffer constBuffer;
        if (FAILED(_d3dDevice->CreateBuffer(&desc, nullptr, &constBuffer.buffer.p)))
        {
            return false;
        }
        constBuffer.size = size;
        _constBufferArray.push_back(constBuffer);

        size >>= 1u;
    }

    std::sort(
        _constBufferArray.begin(),
        _constBufferArray.end(),
        [](const ConstBuffer &lhs, const ConstBuffer &rhs)
        {
            return lhs.size < rhs.size;
        }
    );

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      定数バッファを取得
 *  \param[in]  size    サイズ
 *  \return     サイズに対応したバッファ．取得できない場合はnullptr．
 */
/* ------------------------------------------------------------------------- */
ID3D11Buffer *Device::GetConstantBuffer(size_t size) noexcept
{
    auto it = std::lower_bound(
        _constBufferArray.begin(),
        _constBufferArray.end(),
        ConstBuffer(size),
        [](const ConstBuffer &lhs, const ConstBuffer &rhs)
        {
            return lhs.size < rhs.size;
        }
    );
    if (it == _constBufferArray.end())
    {
        return nullptr;
    }

    return it->buffer;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      描画先の変更
 *  \param[in]  renderTargetView    変更する描画先．nullptrでメインの描画先．
 *  \param[in]  width               描画先の幅
 *  \param[in]  height              描画先の高さ
 *  \retval     true                成功
 *  \retval     false               失敗
 */
/* ------------------------------------------------------------------------- */
bool Device::SetRenderTarget(ID3D11RenderTargetView *renderTargetView, float width, float height) noexcept
{
    if (renderTargetView == nullptr)
    {
        _immediateContext->OMSetRenderTargets(1, &_mainRenderTargetView.p, nullptr);
        _currentRenderTarget = _mainRenderTargetView;
    }
    else
    {
        _immediateContext->OMSetRenderTargets(1, &renderTargetView, nullptr);
        _currentRenderTarget = renderTargetView;
    }

    UpdateViewport(width, height);

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      クライアントバッファのリサイズ
 *  \param[in]  width   リサイズ後の幅
 *  \param[in]  height  リサイズ後の高さ
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool Device::ResizeClientBuffer(float width, float height) noexcept
{
    const bool isMainRenderTarget = (_currentRenderTarget == _mainRenderTargetView);

    // 現在の描画先がメインなら一旦外す
    if (isMainRenderTarget)
    {
        _currentRenderTarget = nullptr;
        _immediateContext->OMSetRenderTargets(0, nullptr, nullptr);
    }                

    // メインの描画先を削除
    _mainRenderTargetView.Release();
    _mainTexture.Release();

    // バックバッファをリサイズ
    DXGI_SWAP_CHAIN_DESC desc;
    _swapChain->GetDesc(&desc);
    if (FAILED(_swapChain->ResizeBuffers(1, static_cast<UINT>(width), static_cast<UINT>(height), desc.BufferDesc.Format, desc.Flags)))
    {
        return false;
    }

    // 描画先を再生成
    if (FAILED(_swapChain->GetBuffer(0, IID_PPV_ARGS(&_mainTexture))))
    {
        return false;
    }
    if (FAILED(_d3dDevice->CreateRenderTargetView(_mainTexture, nullptr, &_mainRenderTargetView.p)))
    {
        return false;
    }

    // 描画先を元に戻す
    if (isMainRenderTarget)
    {
        SetRenderTarget(nullptr, width, height);
    }

    // 行列の更新
    UpdateOrthogonalMatrix();

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      メインレンダーターゲットのピクセルフォーマットを取得
 *  \return     ピクセルフォーマット
 */
/* ------------------------------------------------------------------------- */
DXGI_FORMAT Device::GetMainRenderTargetPixelFormat() const noexcept
{
    D3D11_RENDER_TARGET_VIEW_DESC desc;
    _mainRenderTargetView->GetDesc(&desc);

    return desc.Format;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      メインレンダーターゲットのサイズを取得
 *  \return     サイズ
 */
/* ------------------------------------------------------------------------- */
Vector2 Device::GetMainRenderTargetSize() const noexcept
{
    D3D11_TEXTURE2D_DESC desc;
    _mainTexture->GetDesc(&desc);

    return {static_cast<float>(desc.Width), static_cast<float>(desc.Height)};
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ビューポートの更新
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool Device::UpdateViewport(float width, float height) noexcept
{
    D3D11_VIEWPORT viewport = {};
    viewport.Width = width;
    viewport.Height = height;
    viewport.MinDepth = 0.0f;
    viewport.MaxDepth = 1.0f;
    viewport.TopLeftX = 0.0f;
    viewport.TopLeftY = 0.0f;
    _immediateContext->RSSetViewports(1, &viewport);

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      平行投影行列の更新
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool Device::UpdateOrthogonalMatrix() noexcept
{
    auto size = GetMainRenderTargetSize();

    Matrix4x4 matrix = kIdentityMatrix4x4;
    matrix.column[0].x = 2.0f / size.x;
    matrix.column[1].y = 2.0f / -size.y;
    matrix.column[3] = Vector4(-1.0f, 1.0f, 0.0f, 1.0f);

    D3D11_MAPPED_SUBRESOURCE resource;
    if (FAILED(_immediateContext->Map(_mainOrthogonalMatrixBuffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &resource)))
    {
        return false;
    }
    memcpy(resource.pData, &matrix, sizeof(Matrix4x4));
    _immediateContext->Unmap(_mainOrthogonalMatrixBuffer, 0);

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フレーム更新後のイベント処理
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       未使用
 */
/* ------------------------------------------------------------------------- */
void Device::OnEventPostFrameUpdate(void *callbackArg, [[maybe_unused]] void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<Device *>(callbackArg);

    thisPtr->Present();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      クライアントサイズ変更時の処理
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       新しいサイズ（Vector2型）
 */
/* ------------------------------------------------------------------------- */
void Device::OnEventChangeClientSize(void *callbackArg, void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<Device *>(callbackArg);
    auto *newSize = static_cast<Vector2 *>(notifyArg);

    thisPtr->ResizeClientBuffer(newSize->x, newSize->y);
}

}   // namespace MGL::Render::D3D11

#endif  // MGL_RENDERER_ENABLE_D3D11
// vim: et ts=4 sw=4 sts=4
