// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_win32_window.cc
 *  \brief      MGL Win32用ウィンドウ
 *  \date       Since: March 1, 2021. 12:04:30 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/platform/win32/mgl_win32_window.h>
#if defined(MGL_TARGET_WIN32)

namespace MGL::Win32
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インスタンスの取得
 *  \return     インスタンスの参照
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Window> &Window::GetInstanceRef() noexcept
{
    static STL::unique_ptr<Window> sInstance = nullptr;
    return sInstance;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \param[in]  hInstance       アプリケーションインスタンス
 *  \param[in]  nCmdShow        ウィンドウの表示方法
 *  \param[in]  windowProc      ウィンドウプロシージャのアドレス
 *  \param[in]  descriptor      初期化用の記述子
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool Window::Initialize(HINSTANCE hInstance, int nCmdShow, WNDPROC windowProc, const Descriptor &descriptor) noexcept
{
    _hInstance = hInstance;

    // クラス名が指定されていなければウィンドウタイトルをそのまま使用する
    const auto *className = descriptor.className != nullptr ? descriptor.className : descriptor.windowTitle;

    // ウィンドウクラスの登録
    // NOLINTBEGIN(cppcoreguidelines-pro-type-cstyle-cast, google-readability-casting)
    WNDCLASSEX wcex;
    wcex.cbSize = sizeof(WNDCLASSEX);                           // この構造体のサイズ
    wcex.style = CS_HREDRAW | CS_VREDRAW;                       // ウィンドウのスタイル
    wcex.lpfnWndProc = windowProc;                              // ウィンドウプロシージャのコールバック関数
    wcex.cbClsExtra = 0;                                        // 構造体の余分なメモリの初期化値
    wcex.cbWndExtra = 0;                                        // ウィンドウインスタンスの余分なメモリの初期化値
    wcex.hInstance = _hInstance;                                // アプリケーションインスタンス
    wcex.hIcon = descriptor.icon;                               // ウィンドウアイコン
    wcex.hCursor = LoadCursor(nullptr, IDC_ARROW);              // カーソル
    wcex.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);            // 背景
    wcex.lpszMenuName = nullptr;                                // メニュー名
    wcex.lpszClassName = className;                             // クラス名
    wcex.hIconSm = descriptor.smallIcon;                        // スモールアイコン
    if (!RegisterClassEx(&wcex))
    {
        return false;
    }
    // NOLINTEND(cppcoreguidelines-pro-type-cstyle-cast, google-readability-casting)

    // ウィンドウの生成
    RECT rc = { 0, 0, descriptor.width, descriptor.height };
    AdjustWindowRect(&rc, WS_OVERLAPPEDWINDOW, FALSE);
    //DWORD windowStyle = WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX;   // ウィンドウスタイル
    const DWORD windowStyle = WS_OVERLAPPEDWINDOW;
    _hWnd = CreateWindow(
            className,              // クラス名
            descriptor.windowTitle, // ウィンドウタイトル
            windowStyle,            // ウィンドウスタイルの設定
            CW_USEDEFAULT,          // X座標
            CW_USEDEFAULT,          // Y座標
            rc.right - rc.left,     // 幅
            rc.bottom - rc.top,     // 高さ
            nullptr,                // 親ウィンドウ
            nullptr,                // メニュー
            _hInstance,             // アプリケーションインスタンス
            nullptr                 // パラメータ
    );
    if (_hWnd == nullptr)
    {
        return false;
    }

    ShowWindow(_hWnd, nCmdShow);

    _width = descriptor.width;
    _height = descriptor.height;

    _scaleFactor = GetLogicalScaleFactor();

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ウィンドウタイトルの設定
 *  \param[in]  title   ウィンドウタイトル
 */
/* ------------------------------------------------------------------------- */
void Window::SetWindowTitle(const char *title) noexcept
{
    constexpr size_t kBufferSize = 4096;
    wchar_t wStr[kBufferSize];

    MultiByteToWideChar(CP_UTF8, 0, title, -1, wStr, kBufferSize);

    SetWindowText(_hWnd, wStr);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ウィンドウのリサイズ
 *  \param[in]  width   幅
 *  \param[in]  height  高さ
 */
/* ------------------------------------------------------------------------- */
bool Window::Resize(int width, int height) noexcept
{
    RECT rc = { 0, 0, width, height };
    AdjustWindowRectExForDpi(&rc, WS_OVERLAPPEDWINDOW, FALSE, 0, GetDpiForWindow(_hWnd));

    if (SetWindowPos(_hWnd, nullptr, 0, 0, rc.right - rc.left, rc.bottom - rc.top, SWP_NOMOVE | SWP_NOZORDER))
    {
        _width = width;
        _height = height;

        return true;
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ウィンドウの論理サイズの倍率を取得
 *  \return     論理サイズの倍率
 */
/* ------------------------------------------------------------------------- */
float Window::GetLogicalScaleFactor() const noexcept
{
    return static_cast<float>(GetDpiForWindow(_hWnd)) / static_cast<float>(USER_DEFAULT_SCREEN_DPI);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ウィンドウの更新処理
 */
/* ------------------------------------------------------------------------- */
void Window::Update() noexcept
{
    // DPIが変更されたタイミングでウィンドウサイズを調整する
    // （WM_DPICHANGEDはマニフェストを設定しても呼ばれた試しがないので，ここで監視する）
    auto currentScaleFactor = GetLogicalScaleFactor();
    if (_scaleFactor != currentScaleFactor)
    {
        auto ratio = currentScaleFactor / _scaleFactor;
        auto width = static_cast<float>(_width) * ratio;
        auto height = static_cast<float>(_height) * ratio;
        Resize(static_cast<int>(width), static_cast<int>(height));

        _scaleFactor = currentScaleFactor;
    }
}

}   // namespace MGL::Win32

#endif  // MGL_TARGET_WIN32

// vim: et ts=4 sw=4 sts=4
