// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_win32_main.cc
 *  \brief      MGL Win32用メイン
 *  \date       Since: March 1, 2021. 11:56:04 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/platform/win32/mgl_win32_main.h>
#if defined(MGL_TARGET_WIN32)

#include <Dbt.h>

#include <mgl/initialize/mgl_initialize.h>
#include <mgl/platform/win32/mgl_win32_window.h>
#include <mgl/event/mgl_event.h>
#include <mgl/render/mgl_render.h>
#include <mgl/system/mgl_system_debug_macro.h>

namespace MGL::Win32
{
namespace
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ウィンドウプロシージャ
 *  \param[in]  hWnd        ウィンドウハンドラ
 *  \param[in]  message     ウィンドウへのイベントメッセージ
 *  \param[in]  wParam      メッセージのパラメータ
 *  \param[in]  lParam      メッセージのパラメータ
 *  \return     常に0
 */
/* ------------------------------------------------------------------------- */
LRESULT CALLBACK WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch (message)
    {
        // ウィンドウを閉じたら終了
        case WM_DESTROY:
            PostQuitMessage(0);
            break;

        // ウィンドウ生成直前
        case WM_NCCREATE:
            EnableNonClientDpiScaling(hWnd);
            return DefWindowProc(hWnd, message, wParam, lParam);

        // デバイス接続・切断通知
        case WM_DEVICECHANGE:
            if ((wParam == DBT_DEVICEARRIVAL) || (wParam == DBT_DEVICEREMOVECOMPLETE))
            {
                auto *data = reinterpret_cast<DEV_BROADCAST_HDR *>(lParam);
                if (data->dbch_devicetype == DBT_DEVTYP_DEVICEINTERFACE)
                {
                    auto *interfaceData = reinterpret_cast<DEV_BROADCAST_DEVICEINTERFACE *>(data);
                    if (wParam == DBT_DEVICEARRIVAL)
                    {
                        MGL::Event::Notify(MGL::Event::NotifyType::DeviceArrival, &interfaceData->dbcc_classguid);
                    }
                    else if (wParam == DBT_DEVICEREMOVECOMPLETE)
                    {
                        MGL::Event::Notify(MGL::Event::NotifyType::DeviceRemove, &interfaceData->dbcc_classguid);
                    }
                }
            }
            break;

        // ウィンドウのリサイズ
        case WM_SIZE:
            if (MGL::IsInitialized())
            {
                MGL::Vector2 newSize(static_cast<float>(LOWORD(lParam)), static_cast<float>(HIWORD(lParam)));
                MGL::Event::Notify(MGL::Event::NotifyType::ChangeClientSize, &newSize);
            }
            break;

        // システムコマンド
        case WM_SYSCOMMAND:
            // ALTキーを単体で入力した場合にゲームが停止する問題への対処
            if ((wParam == SC_KEYMENU) && (LOWORD(lParam) <= 0))
            {
                return 0;
            }
            return DefWindowProc(hWnd, message, wParam, lParam);

        default:
            return DefWindowProc(hWnd, message, wParam, lParam);
    }

    return 0;
}
}   // namespace

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      Win32メイン関数
 *  \param[in]  hInstance           アプリケーションインスタンス
 *  \param[in]  hPrevInstance       未使用
 *  \param[in]  lpCmdLine           未使用
 *  \param[in]  nCmdShow            ウィンドウの表示状態
 *  \param[in]  initializer         MGLイニシャライザ
 *  \param[in]  windowDescriptor    ウィンドウ初期化用記述子
 *  \param[in]  makeDelegate        アプリケーションデリゲートを生成する関数
 *  \return     終了コード
 */
 /* ------------------------------------------------------------------------- */
int Main(
    HINSTANCE hInstance,
    [[maybe_unused]] HINSTANCE hPrevInstance,
    [[maybe_unused]] LPSTR lpCmdLine,
    int nCmdShow,
    MGL::InitializerDelegate &initializer,
    const Window::Descriptor &windowDescriptor,
    MakeAppDelegateFunction makeDelegate
) noexcept
{
    // 有効なメモリアロケータが設定されていない場合はデフォルトアロケータを設定
    // MGLの初期化処理内でも行っているけど，ここでやらないと次のウィンドウの生成が行えない
    if (!Memory::IsAvailableAllocator())
    {
        Memory::SetAllocator(nullptr);
    }

    // ウィンドウの生成
    Window::CreateInstance();
    auto &window = Window::GetInstance();
    window.Initialize(hInstance, nCmdShow, WindowProc, windowDescriptor);

    // MGLの初期化
    if (!MGL::Initialize(initializer, makeDelegate))
    {
        return -1;
    }

    // ウィンドウの初期サイズを通知しておく
    MGL::Vector2 windowSize(static_cast<float>(window.GetWidth()), static_cast<float>(window.GetHeight()));
    MGL::Event::Notify(MGL::Event::NotifyType::ChangeClientSize, &windowSize);

    // ループ処理
    MSG msg{};
    while (msg.message != WM_QUIT)
    {
        if (PeekMessage(&msg, nullptr, 0, 0, PM_REMOVE))
        {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }

        window.Update();

        MGL::Event::Notify(MGL::Event::NotifyType::PreFrameUpdate);
        MGL::Event::Notify(MGL::Event::NotifyType::AppFrameUpdate);
        MGL::Render::GetRenderer2D().Flush();
        MGL::Event::Notify(MGL::Event::NotifyType::PostFrameUpdate);
    }

    MGL::Terminate();

    return static_cast<int>(msg.wParam);
}
}   // namespace MGL::Win32

#endif  // MGL_TARGET_WIN32

// vim: et ts=4 sw=4 sts=4
