// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_macos_view_controller.mm
 *  \brief      MGL macOS用ビューコントローラ
 *  \date       Since: February 11, 2021. 23:45:20 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#import <mgl/platform/apple/macos/mgl_macos_view_controller.h>
#if defined(MGL_TARGET_MACOS)

#import <Carbon/Carbon.h>
#import <mgl/event/mgl_event.h>
#import <mgl/platform/apple/macos/mgl_macos_keycode.h>
#import <mgl/input/mouse/mgl_mouse_event.h>

static NSInteger s_WindowNumber = -1;

@implementation MGLmacOSViewController

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ビュー読み込み後の処理
 */
/* ------------------------------------------------------------------------- */
- (void)viewDidLoad
{
    [super viewDidLoad];
    
    // 自身をFirstResponderに設定
    [self.view.window makeFirstResponder:self];
    
    // 特殊なキーに対するイベントを定義
    [NSEvent addLocalMonitorForEventsMatchingMask:NSEventMaskKeyUp
                                          handler:^(NSEvent *event)
     {
        if (event.type == NSEventTypeKeyUp)
        {
            // デフォルトではCommandキーで修飾されているとkeyUpが呼ばれないので明示的に呼び出す
            if ((event.modifierFlags & NSEventModifierFlagCommand) != 0)
            {
                [self keyUp: event];
            }
        }
        return event;
    }];
    
    // マウス移動時のイベント発生を有効にする
    self.view.window.acceptsMouseMovedEvents = YES;
}

- (void)viewWillAppear
{
    s_WindowNumber = self.view.window.windowNumber;
}


- (void)viewWillDisappear
{
    s_WindowNumber = -1;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      メインウィンドウの取得
 *  \return     メインウィンドウ．生成されていない場合はnullptr
 */
/* ------------------------------------------------------------------------- */
+ (nullable NSWindow *)mainWindow
{
    return [NSApplication.sharedApplication windowWithWindowNumber:s_WindowNumber];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      キーダウンイベント
 *  \param[in]  event   イベントの詳細
 */
/* ------------------------------------------------------------------------- */
- (void)keyDown:(NSEvent *)event
{
    // リピート入力は無視する
    if (event.isARepeat)
    {
        return;
    }
    
    // コマンドキーで修飾されているキーは無視する（システム側の処理と被るため）
    if ((event.modifierFlags & NSEventModifierFlagCommand) != 0)
    {
        return;
    }
    
    // MGLのイベントに通知
    if (auto keycode = MGL::Platform::MacOS::ConvertKeycode(event.keyCode); keycode != MGL::Input::Keycode::Invalid)
    {
        MGL::Event::Notify(MGL::Event::NotifyType::InputKeyDown, &keycode);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      キーアップイベント
 *  \param[in]  event   イベントの詳細
 */
/* ------------------------------------------------------------------------- */
- (void)keyUp:(NSEvent *)event
{
    // リピート入力は無視する
    if (event.isARepeat)
    {
        return;
    }
    
    // MGLのイベントに通知
    if (auto keycode = MGL::Platform::MacOS::ConvertKeycode(event.keyCode); keycode != MGL::Input::Keycode::Invalid)
    {
        MGL::Event::Notify(MGL::Event::NotifyType::InputKeyUp, &keycode);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      修飾キー操作時のイベント
 *  \param[in]  event   イベントの詳細
 */
/* ------------------------------------------------------------------------- */
- (void)flagsChanged:(NSEvent *)event
{
    MGL::Input::Keycode keycode = MGL::Input::Keycode::Invalid;
    bool isPressed = false;
    
    switch(event.keyCode)
    {
        // Shiftキー
        case kVK_Shift:
        case kVK_RightShift:
            isPressed = (event.modifierFlags & NSEventModifierFlagShift) != 0;
            keycode = MGL::Platform::MacOS::ConvertKeycode(event.keyCode);
            break;
            
        // Controlキー
        case kVK_Control:
        case kVK_RightControl:
            isPressed = (event.modifierFlags & NSEventModifierFlagControl) != 0;
            keycode = MGL::Platform::MacOS::ConvertKeycode(event.keyCode);
            break;
            
        // Optionキー
        case kVK_Option:
        case kVK_RightOption:
            isPressed = (event.modifierFlags & NSEventModifierFlagOption) != 0;
            keycode = MGL::Platform::MacOS::ConvertKeycode(event.keyCode);
            break;
            
        // Commandキー
        case kVK_Command:
        case kVK_RightCommand:
            isPressed = (event.modifierFlags & NSEventModifierFlagCommand) != 0;
            keycode = MGL::Platform::MacOS::ConvertKeycode(event.keyCode);
            break;
            
        default:
            break;
    }
    
    // MGLのイベントに通知
    if (keycode != MGL::Input::Keycode::Invalid)
    {
        auto notifyType = isPressed ? MGL::Event::NotifyType::InputKeyDown : MGL::Event::NotifyType::InputKeyUp;
        MGL::Event::Notify(notifyType, &keycode);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの左ボタンを押した際のイベント
 *  \param[in]  event   イベントの詳細
 */
/* ------------------------------------------------------------------------- */
- (void)mouseDown:(NSEvent *)event
{
    [self notifyMouseButton:event
                  isPressed:true];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの左ボタンを離した際のイベント
 *  \param[in]  event   イベントの詳細
 */
/* ------------------------------------------------------------------------- */
- (void)mouseUp:(NSEvent *)event
{
    [self notifyMouseButton:event
                  isPressed:false];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの左ボタンのドラッグイベント（押したまま移動した際のイベント）
 *  \param[in]  event   イベントの詳細
 */
/* ------------------------------------------------------------------------- */
- (void)mouseDragged:(NSEvent *)event
{
    [self notifyMousePosition:[self getMousePosition:event]
                    deltaMove:[self getMouseDeltaMove:event]];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの右ボタンを押した際のイベント
 *  \param[in]  event   イベントの詳細
 */
/* ------------------------------------------------------------------------- */
- (void)rightMouseDown:(NSEvent *)event
{
    [self notifyMouseButton:event
                  isPressed:true];

}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの右ボタンを離した際のイベント
 *  \param[in]  event   イベントの詳細
 */
/* ------------------------------------------------------------------------- */
- (void)rightMouseUp:(NSEvent *)event
{
    [self notifyMouseButton:event
                  isPressed:false];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの右ボタンのドラッグイベント（押したまま移動した際のイベント）
 *  \param[in]  event   イベントの詳細
 */
/* ------------------------------------------------------------------------- */
- (void)rightMouseDragged:(NSEvent *)event
{
    [self notifyMousePosition:[self getMousePosition:event]
                    deltaMove:[self getMouseDeltaMove:event]];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの左右以外のボタンを押した際のイベント
 *  \param[in]  event   イベントの詳細
 */
/* ------------------------------------------------------------------------- */
- (void)otherMouseDown:(NSEvent *)event
{
    [self notifyMouseButton:event
                  isPressed:true];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの左右以外のボタンを離した際のイベント
 *  \param[in]  event   イベントの詳細
 */
/* ------------------------------------------------------------------------- */
- (void)otherMouseUp:(NSEvent *)event
{
    [self notifyMouseButton:event
                  isPressed:false];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの左右以外のボタンのドラッグイベント（押したまま移動した際のイベント）
 *  \param[in]  event   イベントの詳細
 */
/* ------------------------------------------------------------------------- */
- (void)otherMouseDragged:(NSEvent *)event
{
    [self notifyMousePosition:[self getMousePosition:event]
                    deltaMove:[self getMouseDeltaMove:event]];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウス移動イベント（何も押さずに移動した際のイベント）
 *  \param[in]  event   イベントの詳細
 */
/* ------------------------------------------------------------------------- */
- (void)mouseMoved:(NSEvent *)event
{
    [self notifyMousePosition:[self getMousePosition:event]
                    deltaMove:[self getMouseDeltaMove:event]];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      イベントからマウスの位置を取得
 *  \param[in]  event   イベントの詳細
 *  \return     スクリーン座標に変換したマウスの位置
 */
/* ------------------------------------------------------------------------- */
- (NSPoint)getMousePosition:(const NSEvent *)event
{
    NSPoint point = event.locationInWindow;
    CGSize size = self.view.frame.size;
    point.x = point.x / size.width;
    point.y = (size.height - point.y) / size.height;
        
    return point;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      イベントからマウスの移動量を取得
 *  \param[in]  event   イベントの詳細
 *  \return     スクリーン座標に変換したマウスの移動量
 */
/* ------------------------------------------------------------------------- */
- (NSPoint)getMouseDeltaMove:(const NSEvent *)event
{
    NSPoint deltaMove = CGPointMake(event.deltaX, event.deltaY);
    CGSize size = self.view.window.screen.frame.size;
    deltaMove.x = deltaMove.x / size.width;
    deltaMove.y = deltaMove.y / size.height;
        
    return deltaMove;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボタンイベントの通知
 *  \param[in]  event       イベントの詳細
 *  \param[in]  isPressed   押したかのフラグ
 */
/* ------------------------------------------------------------------------- */
- (void)notifyMouseButton:(const NSEvent *)event
                isPressed:(bool)isPressed
{
    MGL::Input::MouseEventArgument eventArg;
    
    // ボタンダウンを通知
    MGL::Input::MouseButton button = [self getMouseButton:event.buttonNumber];
    eventArg.type = isPressed ? MGL::Input::MouseEventType::ButtonDown : MGL::Input::MouseEventType::ButtonUp;
    eventArg.button = button;
    MGL::Event::Notify(MGL::Event::NotifyType::InputMouse, &eventArg);
        
    // 位置の更新も通知
    [self notifyMousePosition:[self getMousePosition:event]
                    deltaMove:[self getMouseDeltaMove:event]];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      NSEventのマウスボタン番号からMGLのマウスボタンの種類を取得
 *  \param[in]  buttonNumber    NSEventのマウスボタン番号
 *  \return     MGLのマウスボタンの種類．未対応のボタンの場合はNone
 */
/* ------------------------------------------------------------------------- */
- (MGL::Input::MouseButton)getMouseButton:(NSInteger)buttonNumber
{
    static MGL::Input::MouseButton buttonArray[] =
    {
        MGL::Input::MouseButton::Left,
        MGL::Input::MouseButton::Right,
        MGL::Input::MouseButton::Middle,
        MGL::Input::MouseButton::Other1,
        MGL::Input::MouseButton::Other2,
        MGL::Input::MouseButton::Other3,
        MGL::Input::MouseButton::Other4,
        MGL::Input::MouseButton::Other5,
        MGL::Input::MouseButton::Other6,
        MGL::Input::MouseButton::Other7,
        MGL::Input::MouseButton::Other8,
    };
    static size_t arraySize = sizeof(buttonArray) / sizeof(MGL::Input::MouseButton);

    auto index = static_cast<size_t>(buttonNumber);
    if (index >= arraySize)
    {
        return MGL::Input::MouseButton::None;
    }
    
    return buttonArray[index];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの位置が移動した際のイベント通知
 *  \param[in]  position    押した際の座標
 */
/* ------------------------------------------------------------------------- */
- (void)notifyMousePosition:(NSPoint)position
                  deltaMove:(NSPoint)deltaMove
{
    MGL::Input::MouseEventArgument eventArg;
    
    // 位置の更新を通知
    eventArg.type = MGL::Input::MouseEventType::MovePosition;
    eventArg.position = MGL::Vector2(static_cast<float>(position.x), static_cast<float>(position.y));
    eventArg.deltaMove = MGL::Vector2(static_cast<float>(deltaMove.x), static_cast<float>(deltaMove.y));
    MGL::Event::Notify(MGL::Event::NotifyType::InputMouse, &eventArg);
}

@end

#endif  // MGL_TARGET_MACOS
// vim: et ts=4 sw=4 sts=4
