// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_macos_keycode.cc
 *  \brief      MGL macOS用キーコード変換
 *  \date       Since: December 27, 2020. 18:55:49 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/platform/apple/macos/mgl_macos_keycode.h>
#if defined(MGL_TARGET_MACOS)

#include <Carbon/Carbon.h>

namespace MGL::Platform::MacOS
{
namespace
{
constexpr Input::Keycode kKeycodeTable[] =
{
    [kVK_ANSI_A]        = Input::Keycode::KeyA,         //!< A
    [kVK_ANSI_B]        = Input::Keycode::KeyB,         //!< B
    [kVK_ANSI_C]        = Input::Keycode::KeyC,         //!< C
    [kVK_ANSI_D]        = Input::Keycode::KeyD,         //!< D
    [kVK_ANSI_E]        = Input::Keycode::KeyE,         //!< E
    [kVK_ANSI_F]        = Input::Keycode::KeyF,         //!< F
    [kVK_ANSI_G]        = Input::Keycode::KeyG,         //!< G
    [kVK_ANSI_H]        = Input::Keycode::KeyH,         //!< H
    [kVK_ANSI_I]        = Input::Keycode::KeyI,         //!< I
    [kVK_ANSI_J]        = Input::Keycode::KeyJ,         //!< J
    [kVK_ANSI_K]        = Input::Keycode::KeyK,         //!< K
    [kVK_ANSI_L]        = Input::Keycode::KeyL,         //!< L
    [kVK_ANSI_M]        = Input::Keycode::KeyM,         //!< M
    [kVK_ANSI_N]        = Input::Keycode::KeyN,         //!< N
    [kVK_ANSI_O]        = Input::Keycode::KeyO,         //!< O
    [kVK_ANSI_P]        = Input::Keycode::KeyP,         //!< P
    [kVK_ANSI_Q]        = Input::Keycode::KeyQ,         //!< Q
    [kVK_ANSI_R]        = Input::Keycode::KeyR,         //!< R
    [kVK_ANSI_S]        = Input::Keycode::KeyS,         //!< S
    [kVK_ANSI_T]        = Input::Keycode::KeyT,         //!< T
    [kVK_ANSI_U]        = Input::Keycode::KeyU,         //!< U
    [kVK_ANSI_V]        = Input::Keycode::KeyV,         //!< V
    [kVK_ANSI_W]        = Input::Keycode::KeyW,         //!< W
    [kVK_ANSI_X]        = Input::Keycode::KeyX,         //!< X
    [kVK_ANSI_Y]        = Input::Keycode::KeyY,         //!< Y
    [kVK_ANSI_Z]        = Input::Keycode::KeyZ,         //!< Z
    [kVK_ANSI_1]        = Input::Keycode::Key1,         //!< 1
    [kVK_ANSI_2]        = Input::Keycode::Key2,         //!< 2
    [kVK_ANSI_3]        = Input::Keycode::Key3,         //!< 3
    [kVK_ANSI_4]        = Input::Keycode::Key4,         //!< 4
    [kVK_ANSI_5]        = Input::Keycode::Key5,         //!< 5
    [kVK_ANSI_6]        = Input::Keycode::Key6,         //!< 6
    [kVK_ANSI_7]        = Input::Keycode::Key7,         //!< 7
    [kVK_ANSI_8]        = Input::Keycode::Key8,         //!< 8
    [kVK_ANSI_9]        = Input::Keycode::Key9,         //!< 9
    [kVK_ANSI_0]        = Input::Keycode::Key0,         //!< 0
    [kVK_Return]        = Input::Keycode::Return,       //!< Return or Enter
    [kVK_Escape]        = Input::Keycode::Escape,       //!< Escape
    [kVK_Delete]        = Input::Keycode::Backspace,    //!< Backspace
    [kVK_Tab]           = Input::Keycode::Tab,          //!< Tab
    [kVK_Space]         = Input::Keycode::Space,        //!< Space
    [kVK_F1]            = Input::Keycode::F1,           //!< F1
    [kVK_F2]            = Input::Keycode::F2,           //!< F2
    [kVK_F3]            = Input::Keycode::F3,           //!< F3
    [kVK_F4]            = Input::Keycode::F4,           //!< F4
    [kVK_F5]            = Input::Keycode::F5,           //!< F5
    [kVK_F6]            = Input::Keycode::F6,           //!< F6
    [kVK_F7]            = Input::Keycode::F7,           //!< F7
    [kVK_F8]            = Input::Keycode::F8,           //!< F8
    [kVK_F9]            = Input::Keycode::F9,           //!< F9
    [kVK_F10]           = Input::Keycode::F10,          //!< F10
    [kVK_F11]           = Input::Keycode::F11,          //!< F11
    [kVK_F12]           = Input::Keycode::F12,          //!< F12
    [kVK_ForwardDelete] = Input::Keycode::Delete,       //!< Delete
    [kVK_RightArrow]    = Input::Keycode::Right,        //!< →
    [kVK_LeftArrow]     = Input::Keycode::Left,         //!< ←
    [kVK_DownArrow]     = Input::Keycode::Down,         //!< ↓
    [kVK_UpArrow]       = Input::Keycode::Up,           //!< ↑
    [kVK_Control]       = Input::Keycode::LeftControl,  //!< 左Ctrl
    [kVK_Shift]         = Input::Keycode::LeftShift,    //!< 左Shift
    [kVK_Option]        = Input::Keycode::LeftAlt,      //!< 左Option or 左Alt
    [kVK_Command]       = Input::Keycode::LeftGUI,      //!< 左Command or 左Windows
    [kVK_RightControl]  = Input::Keycode::RightControl, //!< 右Ctrl
    [kVK_RightShift]    = Input::Keycode::RightShift,   //!< 右Shift
    [kVK_RightOption]   = Input::Keycode::RightAlt,     //!< 右Option or 右Alt
    [kVK_RightCommand]  = Input::Keycode::RightGUI,     //!< 右Command or 右Windows
};
constexpr size_t kArraySize = sizeof(kKeycodeTable) / sizeof(Input::Keycode);
}  // namespace

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      macOSのキーコードからMGLのキーコードへ変換
 *  \param[in]  macosKeycode    macOSのキーコード
 *  \return     キーコードに対応した
 */
/* ------------------------------------------------------------------------- */
MGL::Input::Keycode ConvertKeycode(unsigned short macosKeycode)
{
    if (macosKeycode >= kArraySize)
    {
        return MGL::Input::Keycode::Invalid;
    }
    
    return kKeycodeTable[macosKeycode];
}

}   // namespace MGL::Platform::MacOS

#endif  // MGL_TARGET_MACOS

// vim: et ts=4 sw=4 sts=4
