// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_ios_view_controller.mm
 *  \brief      MGL iOS用ビューコントローラ
 *  \date       Since: February 14, 2021. 23:39:31 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */
#import <mgl/platform/apple/ios/mgl_ios_view_controller.h>

#if defined(MGL_TARGET_IOS)

#import <mgl/event/mgl_event.h>
#import <mgl/input/touch/mgl_touch_delegate_event.h>
#import <mgl/platform/apple/ios/mgl_ios_notification_observer.h>

@implementation MGLiOSViewController
{
    MGLiOSNotificationObserver *_observer;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ビュー読み込み後の処理
 */
/* ------------------------------------------------------------------------- */
- (void)viewDidLoad
{
    [super viewDidLoad];

    // マルチタッチを有効化
    self.view.multipleTouchEnabled = YES;
    
    // イベント通知を登録
    _observer = [MGLiOSNotificationObserver new];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タッチ開始イベント
 */
/* ------------------------------------------------------------------------- */
- (void)touchesBegan:(NSSet *)touches withEvent:(UIEvent *)event
{
    for (UITouch *touch in touches)
    {
        CGPoint location = [touch locationInView:self.view ];
        CGRect viewRectPixels = [[self view] bounds];

        MGL::Input::TouchEventArgument argument;
        argument.deviceID = touch.hash;
        argument.position.x = static_cast<float>(location.x / viewRectPixels.size.width);
        argument.position.y = static_cast<float>(location.y / viewRectPixels.size.height);
        argument.type = MGL::Input::TouchEventType::Begin;
        
        MGL::Event::Notify(MGL::Event::NotifyType::InputTouch, &argument);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タッチ移動イベント
 */
/* ------------------------------------------------------------------------- */
- (void)touchesMoved:(NSSet *)touches withEvent:(UIEvent *)event
{
    for (UITouch *touch in touches)
    {
        CGPoint location = [touch locationInView:self.view ];
        CGRect viewRectPixels = [[self view] bounds];

        MGL::Input::TouchEventArgument argument;
        argument.deviceID = touch.hash;
        argument.position.x = static_cast<float>(location.x / viewRectPixels.size.width);
        argument.position.y = static_cast<float>(location.y / viewRectPixels.size.height);
        argument.type = MGL::Input::TouchEventType::Move;
        
        MGL::Event::Notify(MGL::Event::NotifyType::InputTouch, &argument);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タッチ終了イベント
 */
/* ------------------------------------------------------------------------- */
- (void)touchesEnded:(NSSet *)touches withEvent:(UIEvent *)event
{
    for (UITouch *touch in touches)
    {
        MGL::Input::TouchEventArgument argument;
        argument.deviceID = touch.hash;
        argument.type = MGL::Input::TouchEventType::End;
        
        MGL::Event::Notify(MGL::Event::NotifyType::InputTouch, &argument);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タッチキャンセルイベント
 */
/* ------------------------------------------------------------------------- */
- (void)touchesCancelled:(NSSet<UITouch *> *)touches withEvent:(UIEvent *)event
{
    for (UITouch *touch in touches)
    {
        MGL::Input::TouchEventArgument argument;
        argument.deviceID = touch.hash;
        argument.type = MGL::Input::TouchEventType::End;
        
        MGL::Event::Notify(MGL::Event::NotifyType::InputTouch, &argument);
    }
}

@end

#endif  // MGL_TARGET_IOS
// vim: et ts=4 sw=4 sts=4
