// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_memory_fixed_size_allocator.cc
 *  \brief      MGL 固定サイズアロケータ
 *  \date       Since: May 22, 2022. 18:20:50 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/memory/mgl_memory_fixed_size_allocator.h>
#include <mgl/system/mgl_system_debug_macro.h>

namespace MGL::Memory
{
// NOLINTBEGIN(cppcoreguidelines-pro-bounds-avoid-unchecked-container-access)
// Note: メモリアロケータの実装でこれを有効化するのは困難

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化
 *  \param[in]  tag         このアロケータの種類を表すタグ（ヘッダに利用）
 *  \param[in]  blockSize   ブロックサイズ
 *  \param[in]  blockCount  ブロック数
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool FixedSizeAllocator::Initialize(Tag tag, size_t blockSize, size_t blockCount) noexcept
{
    if (_poolSize != 0)
    {
        return false;
    }

    _tag = tag;

    // ブロックサイズと数を保持
    _blockSize = blockSize;
    _blockCount = blockCount;
    _freeCount = _blockCount;

    // メモリプールを確保
    _poolSize = _blockSize * _blockCount;
    _pool = std::make_unique<std::byte[]>(_poolSize);

    // メモリブロックの配列を初期化
    _blockArray = std::make_unique<Block[]>(_blockCount);
    auto *blockAddress = _pool.get();
    for (size_t i = 0; i < _blockCount; i++)
    {
        _blockArray[i].address = blockAddress;

        if (i < (_blockCount - 1))
        {
            _blockArray[i].next = &_blockArray[i + 1];
        }
        else
        {
            _blockArray[i].next = nullptr;
        }

        blockAddress += _blockSize;
    }

    _topBlock = &_blockArray[0];

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アロケート
 *  \param[in]  size    要求サイズ
 *  \return     成功時には利用可能な先頭アドレス。失敗時にはnullptr
 */
/* ------------------------------------------------------------------------- */
void *FixedSizeAllocator::Allocate(size_t size) noexcept
{
    // デフォルトアロケータがサイズチェックを行っているはずなので、
    // 確保できないサイズを要求されないことをアサーション。
    // デフォルトアロケータを介さずに利用するならサイズチェックが必要。
    MGL_ASSERT((size + sizeof(Header) <= _blockSize), "[MGL Memory] Bad algorithm.");

    const std::scoped_lock lock(_mutex);

    // メモリブロックに空きがなければ代替アロケータに任せる
    if (_topBlock == nullptr)
    {
        if (_altAllocator != nullptr)
        {
            return _altAllocator->Allocate(size);
        }

        return nullptr;
    }

    // メモリブロックの先頭にヘッダに情報を書き込み
    auto *header = new (_topBlock->address) Header();
    header->tag = _tag;
    header->usedSize = static_cast<uint16_t>(size);
    header->block = _topBlock;

    // 先頭ブロックと未使用ブロック数を更新
    _topBlock = _topBlock->next;
    _freeCount--;

    // ヘッダの次のアドレスが利用可能な先頭アドレス
    return ++header;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デアロケート
 *  \param[in]  buffer  デアロケートするバッファ
 */
/* ------------------------------------------------------------------------- */
void FixedSizeAllocator::Deallocate(void *buffer) noexcept
{
    // nullptrのデアロケーションもここには到達しないはず。
    // NULLチェックはMGL::Memory::Deallocate()が行うべき処理。
    MGL_ASSERT(buffer != nullptr, "[MGL Memory] Bad algorithm.");

    // でも一応チェック
    if (buffer == nullptr)
    {
        return;
    }

    // ヘッダのチェック
    auto *header = static_cast<Header *>(buffer);
    header--;
    if (header->tag != _tag)
    {
        // ヘッダに書き込まれがアロケータが自身ではない場合、
        // このアロケータが発行していないアドレスを解放しようとしている。
        MGL_ASSERT(0, "[MGL Memory] Detects illegal deallocation: %p", buffer);
        return;
    }

    const std::scoped_lock lock(_mutex);

    // メモリブロックを返却
    header->tag = Tag::None;
    header->block->next = _topBlock;
    _topBlock = header->block;
    _freeCount++;
}
// NOLINTEND(cppcoreguidelines-pro-bounds-avoid-unchecked-container-access)
}    // namespace MGL::Memory

// vim: et ts=4 sw=4 sts=4
