// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_memory_default_allocator.cc
 *  \brief      MGL デフォルトのメモリアロケータ
 *  \date       Since: May 22, 2022. 17:29:52 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <cstring>

#include <mgl/memory/mgl_memory_default_allocator.h>
#include <mgl/system/mgl_system_debug_macro.h>

#if defined(MGL_DEBUG)
#define MGL_MEMORY_DEBUG_ENABLE
#endif

namespace MGL::Memory
{
namespace
{
// どのアロケータを使用するかを表すテーブル。アルゴリズムに関わるので変更不可。
// clang-format off
constexpr uint8_t kAllocatorTable[] = {0, 1, 2, 2, 3, 3, 3, 3};
// clang-format on
}    // namespace

// NOLINTBEGIN(cppcoreguidelines-pro-bounds-avoid-unchecked-container-access)
// Note: メモリアロケータの実装でこれを有効化するのは困難

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
DefaultAllocator::DefaultAllocator() noexcept
{
    // デバッグが無効ならこれらは利用しない
#if !defined(MGL_MEMORY_DEBUG_ENABLE)
    (void)_systemUsedCount;
    (void)_systemUsedSize;
#endif
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \param[in]  config  このアロケータの設定
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool DefaultAllocator::Initialize(const Configuration &config) noexcept
{
    if (_isInitialized)
    {
        return true;
    }

    // メモリプールのサイズがブロックを1つも格納できないサイズの場合は初期化できない
    if ((config.fixedSizeAllocatorMinimumBlockSize << (kFixedSizeAllocatorCount - 1)) > config.fixedSizeAllocatorMemoryPoolSize)
    {
        return false;
    }

    size_t blockSize = config.fixedSizeAllocatorMinimumBlockSize;

    // 最小ブロックサイズが1を何ビットシフトした値なのかを計算
    // この値は要求サイズから適切なアロケータを判別する際に使用する
    // NOTE: ffsとか__builtin_ffsとか_BitScanReverseの方が良いけど処理系の判別がめんどくさい
    _shiftBit = 0;
    for (uint8_t i = 0; i < 32; i++)
    {
        if ((blockSize >> i) == 1)
        {
            _shiftBit = i;
            break;
        }
    }
    if (_shiftBit == 0)
    {
        return false;    // 最小ブロックサイズが2のn乗でない値だとここに到達する
    }

    // 固定サイズアロケータを初期化
    for (size_t index = 0; index < _fixedSizeAllocatorArray.size(); index++)
    {
        auto &allocator = _fixedSizeAllocatorArray[index];

        const size_t blockCount = config.fixedSizeAllocatorMemoryPoolSize / blockSize;

        const auto tag = static_cast<FixedSizeAllocator::Tag>(index);
        allocator.Initialize(tag, blockSize, blockCount);
        _fixedSizeBlockMax = blockSize;

        blockSize *= 2;
    }

    // アロケータに代替アロケータを設定
    for (size_t i = 0; i < _fixedSizeAllocatorArray.size() - 1; i++)
    {
        _fixedSizeAllocatorArray[i].SetAltAllocator(&_fixedSizeAllocatorArray[i + 1]);
    }

    _config = config;

    // 外部アロケータが指定されていない場合はmalloc()、free()、realloc()を使用する
    if (_config.allocator == nullptr)
    {
        _config.allocator = malloc;
    }
    if (_config.deallocator == nullptr)
    {
        _config.deallocator = free;
    }
    if (_config.reallocator == nullptr)
    {
        _config.reallocator = realloc;
    }

    _isInitialized = true;

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool DefaultAllocator::Initialize() noexcept
{
    return Initialize(Configuration());
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化状態を取得
 *  \retval     true    初期化済み
 *  \retval     false   未初期化
 */
/* ------------------------------------------------------------------------- */
bool DefaultAllocator::IsInitialized() const noexcept
{
    return _isInitialized;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アロケート
 *  \param[in]  size    アロケートサイズ
 *  \return     アロケートしたメモリのアドレス
 */
/* ------------------------------------------------------------------------- */
void *DefaultAllocator::Allocate(size_t size) noexcept
{
    void *ptr = nullptr;    // NOLINT(misc-const-correctness)  constにするとコンパイル通らないよ。
    const auto requireSize = size + sizeof(FixedSizeAllocator::Header);

    // 要求サイズが固定サイズアロケータで賄えるならそちらに要求
    if (requireSize <= _fixedSizeBlockMax)
    {
        const auto index = (requireSize - 1) >> _shiftBit;
        MGL_ASSERT(index <= 7, "[MGL Memory] Bad algorithm.");
        ptr = _fixedSizeAllocatorArray[kAllocatorTable[index]].Allocate(size);
    }

    // 固定サイズアロケータで確保できなければ外部アロケータを使用
    if (ptr == nullptr)
    {
        auto *header = new (_config.allocator(requireSize)) FixedSizeAllocator::Header();
        if (header != nullptr)
        {
            // 外部アロケータで確保した場合もヘッダ情報は必要（デアロケート、リアロケート時に参照する）
            header->tag = FixedSizeAllocator::Tag::Extra;
            header->requestSize = size;

            // デバッグが有効であればシステムのアロケータの使用状況を更新
#if defined(MGL_MEMORY_DEBUG_ENABLE)
            _systemUsedCount++;
            _systemUsedSize += size;
#endif
            ptr = ++header;
        }
    }

    return ptr;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デアロケート
 *  \param[in]  buffer  デアロケートするメモリアドレス
 */
/* ------------------------------------------------------------------------- */
void DefaultAllocator::Deallocate(void *buffer) noexcept
{
    auto *header = static_cast<FixedSizeAllocator::Header *>(buffer);
    
    header--;

    // ヘッダから対応するアロケータを取得して解放
    if (auto *allocator = GetFixedSizeAllocator(*header); allocator != nullptr)
    {
        allocator->Deallocate(buffer);
    }
    // 取得に失敗した場合、外部アロケータを使用しているのであればそれを使用して解放
    else if (header->tag == FixedSizeAllocator::Tag::Extra)
    {
        // デバッグが有効であればシステムのアロケータの使用状況を更新
#if defined(MGL_MEMORY_DEBUG_ENABLE)
        _systemUsedCount--;
        _systemUsedSize -= header->requestSize;
#endif
        _config.deallocator(header);
    }
    else
    {
        MGL_ERROR("[MGL Memory] Detected unmanaged address deallocation: %p", buffer);
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      リアロケート
 *  \param[in]  buffer  リアロケートするメモリアドレス
 *  \param[in]  newSize 新たに確保するメモリサイズ
 *  \return     アロケートしたメモリのアドレス
 */
/* ------------------------------------------------------------------------- */
void *DefaultAllocator::Reallocate(void *buffer, size_t newSize) noexcept
{
    // ヘッダ情報を取得
    auto *header = static_cast<FixedSizeAllocator::Header *>(buffer);
    header--;

    void *ptr = nullptr;

    // ヘッダから対応するアロケータを取得してリアロケート
    if (auto *allocator = GetFixedSizeAllocator(*header); allocator != nullptr)
    {
        // 要求サイズがブロックサイズに収まっていれば同じブロックを継続して利用可能
        const auto requireSize = newSize + sizeof(FixedSizeAllocator::Header);
        if (allocator->GetBlockSize() >= requireSize)
        {
            header->usedSize = static_cast<uint16_t>(newSize);
            return buffer;
        }

        // 新たな格納先を新規に確保
        ptr = Allocate(newSize);
        if (ptr == nullptr)
        {
            // 失敗した場合はnullptrを返し、bufferの解放は行わない
            // （この挙動はrealloc()準拠）
            return nullptr;
        }

        // 内容をコピーして元のメモリは解放
        const auto copySize = std::min(static_cast<size_t>(header->usedSize), newSize);
        memcpy(ptr, buffer, copySize);
        allocator->Deallocate(buffer);
    }
    // 外部アロケータを使用している場合は指定されたリアロケータを使用
    else if (header->tag == FixedSizeAllocator::Tag::Extra)
    {
#if defined(MGL_MEMORY_DEBUG_ENABLE)
        _systemUsedSize -= header->requestSize;
#endif
        auto *newHeader = new (_config.reallocator(buffer, newSize)) FixedSizeAllocator::Header();
        newHeader->tag = FixedSizeAllocator::Tag::Extra;
        newHeader->requestSize = newSize;
        ptr = ++newHeader;

#if defined(MGL_MEMORY_DEBUG_ENABLE)
        _systemUsedSize += newSize;
#endif
    }
    else
    {
        MGL_ERROR("[MGL Memory] Detected unmanaged address reallocation: %p", buffer);
    }

    return ptr;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サイズ情報を取得
 *  \param[out] dest    取得したサイズの格納先
 *  \param[in]  key     取得するサイズの種類を表すキー。内容は実装先依存
 *  \param[in]  arg     取得の際に使用する引数
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool DefaultAllocator::GetSizeInfo(size_t &dest, uint32_t key, uint32_t arg) noexcept
{
    switch (key)
    {
        // 固定サイズアロケータの数
        case MGL::Hash::FNV1a("FixedBlockCount"):
            dest = _fixedSizeAllocatorArray.size();
            return true;

        // 固定サイズアロケータのブロックサイズ
        case MGL::Hash::FNV1a("FixedBlockSize"):
            if (arg < _fixedSizeAllocatorArray.size())
            {
                dest = _fixedSizeAllocatorArray[arg].GetBlockSize();
                return true;
            }
            break;

        // 固定サイズアロケータの利用可能な最大ブロック数
        case MGL::Hash::FNV1a("FixedBlockCapacity"):
            if (arg < _fixedSizeAllocatorArray.size())
            {
                dest = _fixedSizeAllocatorArray[arg].GetCapacity();
                return true;
            }
            break;

        // 固定サイズアロケータの使用中のブロック数
        case MGL::Hash::FNV1a("FixedBlockUsedCount"):
            if (arg < _fixedSizeAllocatorArray.size())
            {
                dest = _fixedSizeAllocatorArray[arg].GetUsedCount();
                return true;
            }
            break;

        // 固定サイズアロケータの空きブロック数
        case MGL::Hash::FNV1a("FixedBlockFreeCount"):
            if (arg < _fixedSizeAllocatorArray.size())
            {
                dest = _fixedSizeAllocatorArray[arg].GetFreeCount();
                return true;
            }
            break;

#if defined(MGL_MEMORY_DEBUG_ENABLE)
        // システムからアロケートしている数
        case MGL::Hash::FNV1a("SystemUsedCount"):
            dest = _systemUsedCount;
            return true;

        // システムからアロケートしている合計サイズ
        case MGL::Hash::FNV1a("SystemUsedSize"):
            dest = _systemUsedSize;
            return true;
#endif

        default:
            break;
    }

    return false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ヘッダから対応する固定サイズアロケータを取得
 *  \param[in]  header  ヘッダ
 *  \return     対応する固定サイズアロケータ。見つからない場合はnullptr。
 */
/* ------------------------------------------------------------------------- */
FixedSizeAllocator *DefaultAllocator::GetFixedSizeAllocator(const FixedSizeAllocator::Header &header) noexcept
{
    const auto index = static_cast<size_t>(header.tag);
    if (index < _fixedSizeAllocatorArray.size())
    {
        return &_fixedSizeAllocatorArray[index];
    }

    return nullptr;
}
// NOLINTEND(cppcoreguidelines-pro-bounds-avoid-unchecked-container-access)
}    // namespace MGL::Memory

// vim: et ts=4 sw=4 sts=4
