// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_memory.cc
 *  \brief      MGL メモリ関連
 *  \date       Since: April 29, 2022. 20:29:38 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/memory/mgl_memory.h>
#include <mgl/system/mgl_system_debug_macro.h>

namespace MGL::Memory
{
namespace
{
//! デフォルトのアロケータ
DefaultAllocator s_DefaultAllocator;

//! 現在のアロケータを示すポインタ
Allocator *s_Allocator = nullptr;
};    // namespace

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デフォルトアロケータの設定
 *  \param[in]  config  デフォルトアロケータの設定パラメータ
 */
/* ------------------------------------------------------------------------- */
bool SetDefaultAllocator(const DefaultAllocator::Configuration &config) noexcept
{
    // アロケータ初期化後は実行できない
    if (s_Allocator != nullptr)
    {
        return false;
    }

    // デフォルトアロケータ初期化後は実行できない（必要ない？）
    if (s_DefaultAllocator.IsInitialized())
    {
        return false;
    }

    // デフォルトアロケータを初期化
    if (!s_DefaultAllocator.Initialize(config))
    {
        return false;
    }

    // デフォルトアロケータを設定
    return SetAllocator(nullptr);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アロケータの設定
 *  \param[in]  allocator   設定するアロケータ
 */
/* ------------------------------------------------------------------------- */
bool SetAllocator(Allocator *allocator) noexcept
{
    if (s_Allocator != nullptr)
    {
        return false;
    }

    if (allocator == nullptr)
    {
        allocator = &s_DefaultAllocator;
    }

    if (!allocator->IsInitialized())
    {
        if (!allocator->Initialize())
        {
            return false;
        }
    }

    s_Allocator = allocator;

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アロケート
 *  \param[in]  size    アロケートサイズ
 *  \return     アロケートしたメモリのアドレス
 */
/* ------------------------------------------------------------------------- */
void *Allocate(size_t size) noexcept
{
    MGL_ASSERT(s_Allocator, "[MGL Memory] allocator is not initialized.");

    // サイズが0の場合はnullptrを返す
    if (size == 0)
    {
        return nullptr;
    }

    auto *ptr = s_Allocator->Allocate(size);
    MGL_ASSERT(ptr, "[MGL Memory] allocation failed.");

    return ptr;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デアロケート
 *  \param[in]  buffer  デアロケートするメモリアドレス
 */
/* ------------------------------------------------------------------------- */
void Deallocate(void *buffer) noexcept
{
    MGL_ASSERT(s_Allocator, "[MGL Memory] allocator is not initialized.");

    if (buffer != nullptr)
    {
        s_Allocator->Deallocate(buffer);
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      リアロケート
 *  \param[in]  buffer  リアロケートするメモリアドレス
 *  \param[in]  newSize 新たに確保するメモリサイズ
 *  \return     アロケートしたメモリのアドレス
 */
/* ------------------------------------------------------------------------- */
void *Reallocate(void *buffer, size_t newSize) noexcept
{
    MGL_ASSERT(s_Allocator, "[MGL Memory] allocator is not initialized.");

    // バッファにnullptrが指定された場合は通常のアロケートを行う
    if (buffer == nullptr)
    {
        return Allocate(newSize);
    }

    // バッファが非nullptrでサイズが0の場合はデアロケートを行う
    if (newSize == 0)
    {
        Deallocate(buffer);
    }

    auto *ptr = s_Allocator->Reallocate(buffer, newSize);
    MGL_ASSERT(ptr, "[MGL Memory] reallocation faield");

    return ptr;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アロケータの有効状態を取得
 *  \retval     true    有効
 *  \retval     false   無効
 */
/* ------------------------------------------------------------------------- */
bool IsAvailableAllocator() noexcept
{
    return s_Allocator != nullptr;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      現在のアロケータタイプを取得
 *  \return     現在のアロケータタイプ
 */
/* ------------------------------------------------------------------------- */
AllocatorType GetAllocatorType() noexcept
{
    if (s_Allocator == nullptr)
    {
        return kInvalidAllocatorType;
    }

    return s_Allocator->GetType();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アロケータからサイズ情報を取得
 *  \param[out] dest    取得したサイズ情報の格納先
 *  \param[in]  key     取得するサイズの種類を表すキー。内容は実装先依存
 *  \param[in]  arg     取得の際に使用する引数
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool GetSizeInfo(size_t &dest, uint32_t key, uint32_t arg) noexcept
{
    if (s_Allocator == nullptr)
    {
        return false;
    }

    return s_Allocator->GetSizeInfo(dest, key, arg);
}
}    // namespace MGL::Memory

// vim: et ts=4 sw=4 sts=4
