// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_leaderboard_server.cc
 *  \brief      MGL リーダーボードサーバ
 *  \date       Since: September 4, 2021. 1:27:25 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/leaderboard/mgl_leaderboard_server.h>

namespace MGL::Leaderboard
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インスタンスの取得
 *  \return     インスタンスの参照
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Server> &Server::GetInstanceRef() noexcept
{
    static STL::unique_ptr<Server> sInstance = nullptr;
    return sInstance;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
Server::Server() noexcept
    : _eventPreFrameUpdate(Event::NotifyType::PreFrameUpdate, OnPreFrameUpdate, this)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \param[in]  delegate    デリゲートクラス
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Server::Initialize(STL::unique_ptr<ServerDelegate> &delegate) noexcept
{
    if (_delegate != nullptr)
    {
        return false;
    }
    
    _delegate = std::move(delegate);
    
    return _delegate->Initialize();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スコアの送信
 *  \param[in]  identifier      リーダーボードの識別子
 *  \param[in]  score           送信するスコア
 *  \param[in]  options         送信するオプションデータ
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool Server::ReportScore(Identifier identifier, uint64_t score, const OptionDataArray &options) noexcept
{
    if (IsProcessing())
    {
        return false;
    }

    if (!_delegate->ReportScore(identifier, score, options))
    {
        return false;
    }

    _state = State::ReportScore;

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      エントリデータを取得
 *  \param[in]  identifier      リーダーボードの識別子
 *  \param[in]  startRank       取得する最初の順位
 *  \param[in]  requestSize     取得するエントリ数
 *  \param[in]  playerScope     取得するプレイヤーのスコープ
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool Server::FetchEntryData(Identifier identifier, uint32_t startRank, uint32_t requestSize, PlayerScope playerScope) noexcept
{
    if (IsProcessing())
    {
        return false;
    }

    if (!_delegate->FetchEntryData(identifier, startRank, requestSize, playerScope))
    {
        return false;
    }

    _state = State::FetchEntryData;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      プレイヤーの順位を中心にエントリデータを取得
 *  \param[in]  identifier      リーダーボードの識別子
 *  \param[in]  requestSize     取得するエントリ数
 *  \param[in]  playerScope     取得するプレイヤーのスコープ
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool Server::FetchEntryDataAroundUser(Identifier identifier, uint32_t requestSize, PlayerScope playerScope) noexcept
{
    if (IsProcessing())
    {
        return false;
    }

    if (!_delegate->FetchEntryDataAroundUser(identifier, requestSize, playerScope))
    {
        return false;
    }

    _state = State::FetchEntryData;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      クリップデータの送信
 *  \param[in]  identifier      リーダーボードの識別子
 *  \param[in]  clipData        送信するクリップデータ
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool Server::UploadClipData(Identifier identifier, const ClipData &clipData) noexcept
{
    if (IsProcessing())
    {
        return false;
    }

    if (!_delegate->UploadClipData(identifier, clipData))
    {
        return false;
    }

    _state = State::UploadClipData;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      クリップデータのダウンロード
 *  \param[in]  clipID          クリップデータの識別子
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool Server::DownloadClipData(uint64_t clipID) noexcept
{
    if (IsProcessing())
    {
        return false;
    }

    if (!_delegate->DownloadClipData(clipID))
    {
        return false;
    }

    _state = State::DownloadClipData;

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      更新処理
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       未使用
 */
/* ------------------------------------------------------------------------- */
void Server::OnPreFrameUpdate(void *callbackArg, [[maybe_unused]] void *notifyArg) noexcept
{
    auto *server = static_cast<Server *>(callbackArg);

    if (server->IsProcessing())
    {
        if (!server->_delegate->IsProcessing())
        {
            server->_state = State::Ready;
        }
    }
}
}   // namespace MGL::Leaderboard

// vim: et ts=4 sw=4 sts=4
