// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_touch_server.cc
 *  \brief      MGL タッチ入力サーバ
 *  \date       Since: February 16, 2021. 3:29:14 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/touch/mgl_touch_server.h>

namespace MGL::Input
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インスタンスの取得
 *  \return     インスタンスの参照
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<TouchServer> &TouchServer::GetInstanceRef() noexcept
{
    static STL::unique_ptr<TouchServer> sInstance = nullptr;
    return sInstance;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
TouchServer::TouchServer() noexcept
    : _eventPreFrameUpdate(Event::NotifyType::PreFrameUpdate, OnEventPreFrameUpdate, this)
    , _eventChangeClientSize(Event::NotifyType::ChangeClientSize, OnEventChangeClientSize, this)
{
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \param[in]  delegate    タッチ入力デリゲート
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool TouchServer::Initialize(STL::unique_ptr<TouchDelegate> &delegate) noexcept
{
    if (IsAvailable())
    {
        return false;
    }

    _delegate = std::move(delegate);

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フレーム更新前のコールバック関数
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       未使用
 */
/* ------------------------------------------------------------------------- */
void TouchServer::OnEventPreFrameUpdate(void *callbackArg, [[maybe_unused]] void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<TouchServer *>(callbackArg);

    if (thisPtr->_delegate != nullptr)
    {
        thisPtr->_delegate->UpdateState(thisPtr->_stateArray);
        thisPtr->UpdatePosition();
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      クライアントサイズ変更時のコールバック関数
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       変更後のサイズ（Vector2型）
 */
/* ------------------------------------------------------------------------- */
void TouchServer::OnEventChangeClientSize(void *callbackArg, void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<TouchServer *>(callbackArg);
    const auto *newSize = static_cast<Vector2 *>(notifyArg);

    thisPtr->_clientSize = *newSize;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タッチ座標の更新処理
 */
/* ------------------------------------------------------------------------- */
void TouchServer::UpdatePosition() noexcept
{
    for (auto &state : _stateArray)
    {
        if (!state.isEnabled)
        {
            continue;
        }

        if (!_clientSize.IsZero())
        {
            state.position = Vector2(state.uvPosition.x * _clientSize.x, state.uvPosition.y * _clientSize.y);
            state.prevPosition = Vector2(state.uvPrevPosition.x * _clientSize.x, state.uvPrevPosition.y * _clientSize.y);
            state.deltaMove = Vector2(state.uvDeltaMove.x * _clientSize.x, state.uvDeltaMove.y * _clientSize.y);

            if (_bounds.IsValid())
            {
                state.position = _bounds.GetPosition(state.position);
                state.prevPosition = _bounds.GetPosition(state.prevPosition);
                state.deltaMove = state.prevPosition - state.position;
            }
        }
        else
        {
            state.position = state.uvPosition;
            state.prevPosition = state.uvPrevPosition;
            state.deltaMove = state.uvDeltaMove;
        }

        if (state.touchFrameCount == 1)
        {
            state.deltaMove = state.uvDeltaMove = MGL::Vector2(0.0f, 0.0f);
        }
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インデックスからタッチステートを取得
 *  \param[in]  index   インデックス（0〜kMultiTouchMax）
 *  \return     対応したタッチステート
 */
/* ------------------------------------------------------------------------- */
const TouchState &TouchServer::GetState(size_t index) const noexcept
{
    // 範囲外は取れない
    if (index >= _stateArray.size())
    {
        return _invalidState;
    }

    // ステートが有効でない場合は無効なステートを返す
    if (!_stateArray[index].isEnabled)
    {
        return _invalidState;
    }

    return _stateArray[index];
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タッチ識別番号からタッチステートを取得
 *  \param[in]  touchID     タッチ識別番号
 *  \return     対応したタッチステート．見つからない場合は無効なステートが返る
 */
/* ------------------------------------------------------------------------- */
const TouchState &TouchServer::GetState(TouchID touchID) const noexcept
{
    for (const auto &state : _stateArray)
    {
        if (state.touchID == touchID)
        {
            return state.isEnabled ? state : _invalidState;
        }
    }

    return _invalidState;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      矩形の範囲内にあるタッチステートを取得
 *  \param[in]  rectangle   判定する矩形
 *  \return     矩形の中にある有効なタッチステート．見つからない場合は無効なステートが返る
 *  \note       矩形の中にタッチステートが複数存在する場合，最もインデックスの小さいものが返る
 */
/* ------------------------------------------------------------------------- */
const TouchState &TouchServer::GetState(const Rectangle &rectangle) const noexcept
{
    for (const auto &state : _stateArray)
    {
        if (!state.isEnabled)
        {
            continue;
        }

        if (state.position.x >= rectangle.x)
        {
            if (state.position.x <= (rectangle.x + rectangle.width))
            {
                if (state.position.y >= rectangle.y)
                {
                    if (state.position.y <= (rectangle.y + rectangle.height))
                    {
                        return state;
                    }
                }
            }
        }
    }

    return _invalidState;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タッチの入力範囲を設定
 *  \param[in]  bounds      入力範囲
 */
/* ------------------------------------------------------------------------- */
void TouchServer::SetBounds(const Rectangle &bounds) noexcept
{
    _bounds.Set(bounds);
    UpdatePosition();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タッチの入力範囲を論理サイズで設定
 *  \param[in]  bounds      入力範囲
 *  \param[in]  logicalSize 論理サイズ
 */
/* ------------------------------------------------------------------------- */
void TouchServer::SetBounds(const Rectangle &bounds, const Vector2 &logicalSize) noexcept
{
    _bounds.SetWithLogicalSize(bounds, logicalSize);
    UpdatePosition();
}
}    // namespace MGL::Input

// vim: et ts=4 sw=4 sts=4
