// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_touch_delegate_event.cc
 *  \brief      MGL イベントコールバックによるタッチ入力デリゲート
 *  \date       Since: February 16, 2021. 4:03:46 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/touch/mgl_touch_delegate_event.h>
#if defined(MGL_TOUCH_DELEGATE_ENABLE_EVENT)

namespace MGL::Input
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
EventTouchDelegate::EventTouchDelegate() noexcept
    : _event(Event::NotifyType::InputTouch, OnEventInputTouch, this)
    , _stateArray()
    , _currentTouchID(TouchID{0})
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タッチ入力状態の更新処理
 *  \param[out] stateArray   更新するステート
 */
/* ------------------------------------------------------------------------- */
void EventTouchDelegate::UpdateState(TouchStateArray &stateArray) noexcept
{
    // こちら側で管理しているステートをコピー
    for (size_t i = 0; i < stateArray.size(); ++i)
    {
        auto &state = _stateArray[i];

        stateArray[i].Update(state);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タッチ入力のイベントコールバック
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       タッチ入力イベントの引数
 */
/* ------------------------------------------------------------------------- */
void EventTouchDelegate::OnEventInputTouch(void *callbackArg, void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<EventTouchDelegate *>(callbackArg);
    auto *argument = static_cast<TouchEventArgument *>(notifyArg);

    switch (argument->type)
    {
        // 開始
        case TouchEventType::Begin:
            thisPtr->OnBegin(argument->deviceID, argument->position);
            break;

        // 移動
        case TouchEventType::Move:
            thisPtr->OnMove(argument->deviceID, argument->position);
            break;

        // 終了
        case TouchEventType::End:
            thisPtr->OnEnd(argument->deviceID);
            break;

        default:
            break;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タッチ位置の設定
 *  \param[in]  deviceId    デバイス側の識別番号
 *  \param[in]  position    タッチ位置
 */
/* ------------------------------------------------------------------------- */
void EventTouchDelegate::OnBegin(uint64_t deviceId, const Vector2 &position) noexcept
{
    for (auto &state : _stateArray)
    {
        if (!state.isEnabled)
        {
            state.isEnabled = true;
            state.uvPosition = position;
            state.deviceID = deviceId;
            state.touchID = _currentTouchID++;
            state.touchFrameCount = 0;

            break;
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タッチ位置の移動
 *  \param[in]  deviceID    デバイス側の識別番号
 *  \param[in]  position    タッチ位置
 */
/* ------------------------------------------------------------------------- */
void EventTouchDelegate::OnMove(uint64_t deviceID, const Vector2 &position) noexcept
{
    for (auto &state : _stateArray)
    {
        if (state.isEnabled && (state.deviceID == deviceID))
        {
            state.uvPosition = position;

            break;
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      タッチが離された際の処理
 *  \param[in]  deviceID    デバイス側の識別番号
 */
/* ------------------------------------------------------------------------- */
void EventTouchDelegate::OnEnd(uint64_t deviceID) noexcept
{
    for (auto &state : _stateArray)
    {
        if (state.isEnabled && (state.deviceID == deviceID))
        {
            state.isEnabled = false;

            break;
        }
    }
}

}    // namespace MGL::Input
#endif    // MGL_TOUCH_DELEGATE_ENABLE_EVENT

// vim: et ts=4 sw=4 sts=4
