// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_mouse_server.cc
 *  \brief      MGL マウスサーバ
 *  \date       Since: December 28, 2020. 17:37:53 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/mouse/mgl_mouse_server.h>

#include <mgl/mgl_environment.h>

namespace MGL::Input
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インスタンスの取得
 *  \return     インスタンスの参照
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<MouseServer> &MouseServer::GetInstanceRef() noexcept
{
    static STL::unique_ptr<MouseServer> sInstance = nullptr;
    return sInstance;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
MouseServer::MouseServer() noexcept
    : _eventPreFrameUpdate(Event::NotifyType::PreFrameUpdate, OnEventPreFrameUpdate, this)
    , _eventChangeClientSize(Event::NotifyType::ChangeClientSize, OnEventChangeClientSize, this)
{
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \param[in]  delegate    マウス入力デリゲート
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool MouseServer::Initialize(STL::unique_ptr<MouseDelegate> &delegate) noexcept
{
    if (IsAvailable())
    {
        return false;
    }

    _delegate = std::move(delegate);

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フレーム更新前のコールバック関数
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       未使用
 */
/* ------------------------------------------------------------------------- */
void MouseServer::OnEventPreFrameUpdate(void *callbackArg, [[maybe_unused]] void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<MouseServer *>(callbackArg);

    if (thisPtr->_delegate != nullptr)
    {
        thisPtr->_delegate->UpdateState(thisPtr->_state);
        thisPtr->_state.buttonFlags.Clear(MouseButton::None);
        thisPtr->UpdatePosition();
        thisPtr->_delegate->PostUpdatePosition(thisPtr->_state);
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      クライアントサイズ変更時のコールバック関数
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       変更後のサイズ（Vector2型）
 */
/* ------------------------------------------------------------------------- */
void MouseServer::OnEventChangeClientSize(void *callbackArg, void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<MouseServer *>(callbackArg);
    const auto *newSize = static_cast<Vector2 *>(notifyArg);

    thisPtr->_clientSize = *newSize;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウス座標の更新処理
 */
/* ------------------------------------------------------------------------- */
void MouseServer::UpdatePosition() noexcept
{
    if (!_clientSize.IsZero())
    {
        _state.position = Vector2(_state.uvPosition.x * _clientSize.x, _state.uvPosition.y * _clientSize.y);
        _state.prevPosition = Vector2(_state.uvPrevPosition.x * _clientSize.x, _state.uvPrevPosition.y * _clientSize.y);
        _state.deltaMove = Vector2(_state.uvDeltaMove.x * _clientSize.x, _state.uvDeltaMove.y * _clientSize.y);

        if (_bounds.IsValid())
        {
            _state.position = _bounds.GetPosition(_state.position);
            _state.prevPosition = _bounds.GetPosition(_state.prevPosition);
            _state.deltaMove = _state.prevPosition - _state.position;
        }
    }
    else
    {
        _state.position = _state.uvPosition;
        _state.prevPosition = _state.uvPrevPosition;
        _state.deltaMove = _state.uvDeltaMove;
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボタンが押されているかを取得
 *  \param[in]  button  チェックするボタン
 *  \retval     true    押されている
 *  \retval     false   押されていない
 */
/* ------------------------------------------------------------------------- */
bool MouseServer::IsPressing(MouseButton button) const noexcept
{
    return IsPressingAny(button);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      指定したボタンが全て押されているかを取得
 *  \param[in]  buttonFlags     チェックするボタンフラグ
 *  \retval     true            押されている
 *  \retval     false           押されていない
 */
/* ------------------------------------------------------------------------- */
bool MouseServer::IsPressing(MouseButtonFlags buttonFlags) const noexcept
{
    return ((_state.buttonFlags & buttonFlags) == buttonFlags);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      指定したボタンが全て押されているかを取得
 *  \param[in]  buttonFlags     チェックするボタンフラグ
 *  \retval     true            押されている
 *  \retval     false           押されていない
 */
/* ------------------------------------------------------------------------- */
bool MouseServer::IsPressingAny(MouseButtonFlags buttonFlags) const noexcept
{
    return (_state.buttonFlags & buttonFlags).HasAny();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボタンが押された瞬間を取得
 *  \param[in]  button  チェックするボタン
 *  \retval     true    押された瞬間である
 *  \retval     false   押された瞬間ではない
 */
/* ------------------------------------------------------------------------- */
bool MouseServer::IsTriggered(MouseButton button) const noexcept
{
    return IsPressing(button) && !_state.prevButtonFlags.Has(button);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボタンが離された瞬間を取得
 *  \param[in]  button  チェックするボタン
 *  \retval     true    離された瞬間である
 *  \retval     false   離された瞬間ではない
 */
/* ------------------------------------------------------------------------- */
bool MouseServer::IsReleased(MouseButton button) const noexcept
{
    return !IsPressing(button) && _state.prevButtonFlags.Has(button);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスポインタの位置を取得
 *  \return     マウスポインタの位置
 */
/* ------------------------------------------------------------------------- */
const Vector2 &MouseServer::GetPosition() const noexcept
{
    return _state.position;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスポインタの位置をUV座標で取得
 *  \return     マウスポインタの位置
 */
/* ------------------------------------------------------------------------- */
const Vector2 &MouseServer::GetUVPosition() const noexcept
{
    return _state.uvPosition;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスポインタの移動量を取得
 *  \return     マウスポインタの移動量
 */
/* ------------------------------------------------------------------------- */
const Vector2 &MouseServer::GetDeltaMove() const noexcept
{
    return _state.deltaMove;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスポインタの移動量をUV座標で取得
 *  \return     マウスポインタの移動量
 */
/* ------------------------------------------------------------------------- */
const Vector2 &MouseServer::GetUVDeltaMove() const noexcept
{
    return _state.uvDeltaMove;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの入力範囲を設定
 *  \param[in]  bounds      入力範囲
 */
/* ------------------------------------------------------------------------- */
void MouseServer::SetBounds(const Rectangle &bounds) noexcept
{
    _bounds.Set(bounds);
    UpdatePosition();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの入力範囲を論理サイズ
 *  \param[in]  bounds      入力範囲
 *  \param[in]  logicalSize 論理サイズ
 */
/* ------------------------------------------------------------------------- */
void MouseServer::SetBounds(const Rectangle &bounds, const Vector2 &logicalSize) noexcept
{
    _bounds.SetWithLogicalSize(bounds, logicalSize);
    UpdatePosition();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルモードを設定
 *  \param[in]  cursorMode  設定するカーソルモード
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool MouseServer::SetCursorMode(CursorMode cursorMode) noexcept
{
    if (_delegate != nullptr)
    {
        return _delegate->SetCursorMode(cursorMode);
    }

    return false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルモードを取得
 *  \return     現在のカーソルモード
 */
/* ------------------------------------------------------------------------- */
CursorMode MouseServer::GetCursorMode() const noexcept
{
    if (_delegate != nullptr)
    {
        return _delegate->GetCursorMode();
    }

    return CursorMode::None;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルの表示モードを設定
 *  \param[in]  visibleMode 設定する表示モード
 */
/* ------------------------------------------------------------------------- */
void MouseServer::SetCursorVisibleMode(CursorVisibleMode visibleMode) noexcept
{
    if (_delegate != nullptr)
    {
        _delegate->SetCursorVisibleMode(visibleMode);
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルの表示モードを取得
 *  \return     現在のカーソルの表示モード
 */
/* ------------------------------------------------------------------------- */
CursorVisibleMode MouseServer::GetCursorVisibleMode() const noexcept
{
    if (_delegate != nullptr)
    {
        return _delegate->GetCursorVisibleMode();
    }

    return CursorVisibleMode::Invisible;
}
}    // namespace MGL::Input

// vim: et ts=4 sw=4 sts=4
