// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_mouse_delegate_win32.cc
 *  \brief      MGL Win32用マウスデリゲート
 *  \date       Since: March 31, 2021. 10:24:46 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/mouse/mgl_mouse_delegate_win32.h>
#if defined(MGL_MOUSE_DELEGATE_ENABLE_WIN32)

#include <Windows.h>

#include <mgl/system/mgl_system_application.h>
#include <mgl/platform/win32/mgl_win32_window.h>
#include <mgl/system/mgl_system_debug_macro.h>
#include <mgl/system/mgl_system_window.h>

namespace MGL::Input
{
namespace
{
constexpr float kCursorDisplayTime = 3000;      // カーソルの自動消去時間
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
Win32MouseDelegate::Win32MouseDelegate() noexcept
    : _cursorMode(CursorMode::Pointer)
    , _cursorVisibleMode(CursorVisibleMode::Visible)
    , _cursorDisplayTime(kCursorDisplayTime)
    , _isVisibleCursor(true)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウス入力状態の更新処理
 *  \param[out] state   更新するステート
 */
/* ------------------------------------------------------------------------- */
void Win32MouseDelegate::UpdateState(MouseState &state) noexcept
{
    const MGL::System::Window window;

    // ボタンの更新
    state.prevButtonFlags = state.buttonFlags;
    state.buttonFlags.Clear();
    if (window.IsFocused())
    {
        if ((GetAsyncKeyState(VK_LBUTTON) & 0x8000u) != 0)
        {
            state.buttonFlags |= static_cast<MouseButtonFlags>(MouseButton::Left);
        }
        if ((GetAsyncKeyState(VK_RBUTTON) & 0x8000u) != 0)
        {
            state.buttonFlags |= static_cast<MouseButtonFlags>(MouseButton::Right);
        }
        if ((GetAsyncKeyState(VK_MBUTTON) & 0x8000u) != 0)
        {
            state.buttonFlags |= static_cast<MouseButtonFlags>(MouseButton::Middle);
        }
    }

    // 位置と移動量の更新
    state.uvPrevPosition = state.uvPosition;
    if (window.IsFocused())
    {
        POINT point;
        if (GetCursorPos(&point))
        {
            WINDOWINFO info = {};
            if (GetWindowInfo(Win32::Window::GetInstance().GetWindowHandler(), &info))
            {
                point.x -= info.rcClient.left;
                point.y -= info.rcClient.top;

                state.uvPosition.x = static_cast<float>(point.x) / static_cast<float>(info.rcClient.right - info.rcClient.left);
                state.uvPosition.y = static_cast<float>(point.y) / static_cast<float>(info.rcClient.bottom - info.rcClient.top);
            }

            // 移動量モードの際の処理
            if (_cursorMode == CursorMode::DeltaMove)
            {
                FixCursorPosition();
                ClipCursor(&info.rcClient);
            }
            else
            {
                ClipCursor(nullptr);
            }
        }
    }
    else
    {
        state.uvPosition = MGL::Vector2();
    }
    state.uvDeltaMove = state.uvPosition - state.uvPrevPosition;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの座標確定後の処理
 *  \param[in]  state   更新されたステート
 */
/* ------------------------------------------------------------------------- */
void Win32MouseDelegate::PostUpdatePosition(const MouseState &state) noexcept
{
    // 移動量モードの場合は設定に関わらずカーソルを非表示にする
    if (_cursorMode == CursorMode::DeltaMove)
    {
        SetVisibleCursor(false);
    }
    // それ以外はモードに応じた表示状態を設定する
    else
    {
        switch (_cursorVisibleMode)
        {
            // 表示
            case CursorVisibleMode::Visible:
                SetVisibleCursor(true);
                break;

            // 自動消去
            case CursorVisibleMode::AutoInvisible:
                UpdateCursorAutoErase(state, System::Application().GetFrameElapsedTime());
                break;

            // 非表示
            case CursorVisibleMode::Invisible:
                SetVisibleCursor(false); 
                break;
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルモードを設定
 *  \param[in]  cursorMode  設定するカーソルモード
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Win32MouseDelegate::SetCursorMode(CursorMode cursorMode) noexcept
{
    switch (cursorMode)
    {
        // カーソルモードなし（不正）
        case CursorMode::None:
            return false;

        // ポインタモード
        case CursorMode::Pointer:
            _cursorMode = cursorMode;

            // 切り替えた際に非表示でなければ強制的に表示させる
            if (_cursorVisibleMode != CursorVisibleMode::Invisible)
            {
                _cursorDisplayTime = kCursorDisplayTime;
                SetVisibleCursor(true);
            }
            return true;

        // 移動量モード
        case CursorMode::DeltaMove:
            _cursorMode = cursorMode;
            return true;
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルモードを取得
 *  \return     現在のカーソルモード
 */
/* ------------------------------------------------------------------------- */
CursorMode Win32MouseDelegate::GetCursorMode() const noexcept
{
    return _cursorMode;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルの表示モードを設定
 *  \param[in]  visibleMode 設定する表示モード
 */
/* ------------------------------------------------------------------------- */
void Win32MouseDelegate::SetCursorVisibleMode(CursorVisibleMode visibleMode) noexcept
{
    _cursorVisibleMode = visibleMode;
    
    switch (visibleMode)
    {
        case CursorVisibleMode::Visible:
            SetVisibleCursor(true);
            break;
            
        case CursorVisibleMode::Invisible:
            SetVisibleCursor(false);
            break;
            
        default:
            break;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルの表示モードを取得
 *  \return     現在のカーソルの表示モード
 */
/* ------------------------------------------------------------------------- */
CursorVisibleMode Win32MouseDelegate::GetCursorVisibleMode() const noexcept
{
    return _cursorVisibleMode;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスカーソルの自動消去の更新処理
 *  \param[in]  state           マウスのステート
 *  \param[in]  elapsedTime     経過時間
 */
/* ------------------------------------------------------------------------- */
void Win32MouseDelegate::UpdateCursorAutoErase(const MouseState &state, float elapsedTime) noexcept
{
    // マウスに移動がなければ時間を経過させて，時間になったらカーソルを消去
    if (state.deltaMove.IsZero())
    {
        if (_cursorDisplayTime > 0.0f)
        {
            // 時間の経過はカーソルがウィンドウの内側にある場合のみ行う
            auto uvPosition = state.uvPosition;
            if ((uvPosition.x >= 0.0f) &&
                (uvPosition.y >= 0.0f) &&
                (uvPosition.x <= 1.0f) &&
                (uvPosition.y <= 1.0f))
            {
                _cursorDisplayTime -= elapsedTime;
                
                // 時間になったら消去
                if (_cursorDisplayTime <= 0.0f)
                {
                    SetVisibleCursor(false);
                }
            }
        }
    }
    // マウスが移動したらカーソルを表示して時間をリセット
    else
    {
        if (_cursorDisplayTime <= 0.0f)
        {
            SetVisibleCursor(true);
        }
        _cursorDisplayTime = kCursorDisplayTime;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスカーソルの表示・非表示の切り替え
 *  \param[in]  isVisible   表示フラグ
 *  \note
 *      ShowCursor()は内部カウンタを持っており，
 *      片方を複数回呼んだらもう片方も同じ回数呼び出さないと元の状態に戻らないため，
 *      この関数で重複呼び出しを行わないようにする．
 */
/* ------------------------------------------------------------------------- */
void Win32MouseDelegate::SetVisibleCursor(bool isVisible) noexcept
{
    if (isVisible && !_isVisibleCursor)
    {
        ShowCursor(TRUE);
    }
    else if (!isVisible && _isVisibleCursor)
    {
        ShowCursor(FALSE);
    }
    
    _isVisibleCursor = isVisible;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソル位置の固定
 */
/* ------------------------------------------------------------------------- */
void Win32MouseDelegate::FixCursorPosition() noexcept
{
    WINDOWINFO info = {};
    if (GetWindowInfo(Win32::Window::GetInstance().GetWindowHandler(), &info))
    {
        auto width = info.rcClient.right - info.rcClient.left;
        auto height = info.rcClient.bottom - info.rcClient.top;

        POINT point;
        point.x = info.rcClient.left + width / 2;
        point.y = info.rcClient.top + height / 2;

        SetCursorPos(point.x, point.y);
    }
}

}   // namespace MGL::Input

#endif  // MGL_MOUSE_DELEGATE_ENABLE_WIN32

// vim: et ts=4 sw=4 sts=4
