// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_mouse_delegate_macos.mm
 *  \brief      MGL macOS用マウスデリゲート
 *  \date       Since: February 22, 2021. 2:04:05 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/mouse/mgl_mouse_delegate_macos.h>
#if defined(MGL_MOUSE_DELEGATE_ENABLE_MACOS)

#include <AppKit/NSScreen.h>
#include <CoreGraphics/CoreGraphics.h>

#include <mgl/system/mgl_system_application.h>
#include <mgl/system/mgl_system_window.h>

namespace MGL::Input
{
namespace
{
constexpr float kCursorDisplayTime = 3000;      // カーソルの自動消去時間
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
MacOSMouseDelegate::MacOSMouseDelegate() noexcept
    : _eventInputMouse(Event::NotifyType::InputMouse, OnEventInputMouse, this)
    , _eventShouldClearInput(Event::NotifyType::ShouldClearInput, OnEventShouldClearInput, this)
    , _buttonFlags(0)
    , _firstFrameButtonFlags(0)
    , _nextFrameReleaseFlags(0)
    , _position()
    , _deltaMove()
    , _deltaWheel()
    , _cursorMode(CursorMode::Pointer)
    , _cursorVisibleMode(CursorVisibleMode::Visible)
    , _cursorDisplayTime(kCursorDisplayTime)
    , _isVisibleCursor(true)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウス入力状態の更新処理
 *  \param[out] state   更新するステート
 */
/* ------------------------------------------------------------------------- */
void MacOSMouseDelegate::UpdateState(MouseState &state) noexcept
{
    state.prevButtonFlags = state.buttonFlags;
    state.buttonFlags = _buttonFlags;
    
    _buttonFlags &= ~_nextFrameReleaseFlags;
    _firstFrameButtonFlags.Clear();
    _nextFrameReleaseFlags.Clear();
    
    state.uvPrevPosition = state.uvPosition;
    state.uvPosition = _position;
    state.uvDeltaMove = _deltaMove;
    state.deltaWheel = _deltaWheel;

    _deltaMove = Vector2();
    _deltaWheel = Vector2();
    
    // 移動量取得モードの際には必要に応じてカーソル位置を固定化する
    if (_cursorMode == CursorMode::DeltaMove)
    {
        auto *window = [NSApplication.sharedApplication windowWithWindowNumber: System::Window().GetIdentifier()];
        
        if (ShouldFixCursor(window))
        {
            CGAssociateMouseAndMouseCursorPosition(NO);
            MoveCursorToCenter(window);
        }
        else
        {
            CGAssociateMouseAndMouseCursorPosition(YES);
            state.uvDeltaMove = Vector2();
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの座標確定後の処理
 *  \param[in]  state   更新されたステート
 */
/* ------------------------------------------------------------------------- */
void MacOSMouseDelegate::PostUpdatePosition(const MouseState &state) noexcept
{
    // カーソルモードが移動量取得モードの際にはカーソルを消去する
    if (_cursorMode == CursorMode::DeltaMove)
    {
        SetVisibleCursor(false);
    }
    // それ以外の場合は表示モードに応じてカーソル表示を制御する
    else
    {
        switch (_cursorVisibleMode)
        {
            // Visibleの場合は常に表示（何もしなくていい）
            case CursorVisibleMode::Visible:
                break;
                
            // Invisibleの場合はカーソルがウィンドウの内側にある場合のみカーソルを非表示
            case CursorVisibleMode::Invisible:
                if ((state.uvPosition.x >= 0.0f) &&
                    (state.uvPosition.y >= 0.0f) &&
                    (state.uvPosition.x <= 1.0f) &&
                    (state.uvPosition.y <= 1.0f))
                {
                    SetVisibleCursor(false);
                }
                else
                {
                    SetVisibleCursor(true);
                }
                break;
            
            // AutoInvisibleの場合はマウスの移動量と経過時間に応じて変更する
            case CursorVisibleMode::AutoInvisible:
                UpdateCursorAutoErase(state, System::Application().GetFrameElapsedTime());
                break;
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウス入力イベントのコールバック関数
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       マウスイベントの引数
 */
/* ------------------------------------------------------------------------- */
void MacOSMouseDelegate::OnEventInputMouse(void *callbackArg, void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<MacOSMouseDelegate *>(callbackArg);
    auto *argument = static_cast<MouseEventArgument *>(notifyArg);
    
    switch (argument->type)
    {
        // ボタンダウン
        case MouseEventType::ButtonDown:
            thisPtr->UpdateButtonState(argument->button, true);
            break;
            
        // ボタンアップ
        case MouseEventType::ButtonUp:
            thisPtr->UpdateButtonState(argument->button, false);
            break;
            
        // カーソル移動
        case MouseEventType::MovePosition:
            thisPtr->UpdatePosition(argument->position);
            thisPtr->UpdateDeltaMove(argument->deltaMove);
            break;

        // ホイール操作
        case MouseEventType::DeltaWheel:
            thisPtr->UpdateDeltaWheel(argument->deltaMove);
            break;

        default:
            break;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      入力クリア要求イベント
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       未使用
 */
/* ------------------------------------------------------------------------- */
void MacOSMouseDelegate::OnEventShouldClearInput(void *callbackArg, [[maybe_unused]] void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<MacOSMouseDelegate *>(callbackArg);
    
    thisPtr->_buttonFlags.Clear();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルモードを設定
 *  \param[in]  cursorMode  設定するカーソルモード
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool MacOSMouseDelegate::SetCursorMode(CursorMode cursorMode) noexcept
{
    switch (cursorMode)
    {
        // なし（設定不可）
        case CursorMode::None:
            return false;
            
        // ポインタモード
        case CursorMode::Pointer:
            _cursorMode = cursorMode;
            CGAssociateMouseAndMouseCursorPosition(YES);
            return true;
            
        // 移動量モード
        case CursorMode::DeltaMove:
            _cursorMode = cursorMode;
            return true;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルモードを取得
 *  \return     現在のカーソルモード
 */
/* ------------------------------------------------------------------------- */
CursorMode MacOSMouseDelegate::GetCursorMode() const noexcept
{
    return _cursorMode;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルの表示モードを設定
 *  \param[in]  visibleMode 設定する表示モード
 */
/* ------------------------------------------------------------------------- */
void MacOSMouseDelegate::SetCursorVisibleMode(CursorVisibleMode visibleMode) noexcept
{
    // 変更がなければ何もしない
    if (_cursorVisibleMode == visibleMode)
    {
        return;
    }
    
    switch (visibleMode)
    {
        // 表示モード
        case CursorVisibleMode::Visible:
            SetVisibleCursor(true);
            break;
            
        // 非表示モード
        case CursorVisibleMode::Invisible:
            SetVisibleCursor(false);
            break;

        // 自動消去モード
        case CursorVisibleMode::AutoInvisible:
            if (_cursorVisibleMode == CursorVisibleMode::Visible)
            {
                SetVisibleCursor(true);
                _cursorDisplayTime = kCursorDisplayTime;
            }
            else
            {
                SetVisibleCursor(false);
                _cursorDisplayTime = 0.0f;
            }
            break;
    }

    _cursorVisibleMode = visibleMode;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルの表示モードを取得
 *  \return     現在のカーソルの表示モード
 */
/* ------------------------------------------------------------------------- */
CursorVisibleMode MacOSMouseDelegate::GetCursorVisibleMode() const noexcept
{
    return _cursorVisibleMode;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボタンの入力状態の更新
 *  \param[in]  button      押されたボタン
 *  \param[in]  isPressed   押されたかどうかのフラグ
 */
/* ------------------------------------------------------------------------- */
void MacOSMouseDelegate::UpdateButtonState(MouseButton button, bool isPressed) noexcept
{
    if (isPressed)
    {
        _buttonFlags.Set(button);
        _firstFrameButtonFlags.Set(button);
        _nextFrameReleaseFlags.Clear(button);
    }
    else
    {
        if (_firstFrameButtonFlags.Has(button))
        {
            _nextFrameReleaseFlags.Set(button);
        }
        else
        {
            _buttonFlags.Clear(button);
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルの位置の更新
 *  \param[in]  position    新しいカーソルの位置
 */
/* ------------------------------------------------------------------------- */
void MacOSMouseDelegate::UpdatePosition(const Vector2 &position) noexcept
{
    _position = position;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルの移動量を更新
 *  \param[in]  deltaMove    移動量
 */
/* ------------------------------------------------------------------------- */
void MacOSMouseDelegate::UpdateDeltaMove(const Vector2 &deltaMove) noexcept
{
    _deltaMove += deltaMove;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ホイールの移動量の更新
 *  \param[in]  deltaWheel  移動量
 */
/* ------------------------------------------------------------------------- */
void MacOSMouseDelegate::UpdateDeltaWheel(const Vector2 &deltaWheel) noexcept
{
    _deltaWheel += deltaWheel;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスカーソルの自動消去の更新処理
 *  \param[in]  elapsedTime     経過時間
 */
/* ------------------------------------------------------------------------- */
void MacOSMouseDelegate::UpdateCursorAutoErase(const MouseState &state, float elapsedTime) noexcept
{
    // マウスに移動がなければ時間を経過させて，時間になったらカーソルを消去
    if (state.deltaMove.IsZero())
    {
        if (_cursorDisplayTime > 0.0f)
        {
            // 時間の経過はカーソルがウィンドウの内側にある場合のみ行う
            auto uvPosition = state.uvPosition;
            if ((uvPosition.x >= 0.0f) &&
                (uvPosition.y >= 0.0f) &&
                (uvPosition.x <= 1.0f) &&
                (uvPosition.y <= 1.0f))
            {
                _cursorDisplayTime -= elapsedTime;
                
                // 時間になったら消去
                if (_cursorDisplayTime <= 0.0f)
                {
                    SetVisibleCursor(false);
                }
            }
        }
    }
    // マウスが移動したらカーソルを表示して時間をリセット
    else
    {
        SetVisibleCursor(true);
        _cursorDisplayTime = kCursorDisplayTime;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスカーソルの表示・非表示の切り替え
 *  \param[in]  isVisible   表示フラグ
 *  \note
 *      CGDisplayXxxxCursor()は内部カウンタを持っており，
 *      片方を複数回呼んだらもう片方も同じ回数呼び出さないと元の状態に戻らないため，
 *      この関数で重複呼び出しを行わないようにする．
 */
/* ------------------------------------------------------------------------- */
void MacOSMouseDelegate::SetVisibleCursor(bool isVisible) noexcept
{
    if (isVisible && !_isVisibleCursor)
    {
        CGDisplayShowCursor(kCGNullDirectDisplay);
    }
    else if (!isVisible && _isVisibleCursor)
    {
        CGDisplayHideCursor(kCGNullDirectDisplay);
    }
    
    _isVisibleCursor = isVisible;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスカーソルを画面中央に移動
 *  \param[in]  window  メインウィンドウ
 *  \note
 *      ウィンドウの座標系は左下原点，マウスカーソルの座標系は左上原点なので
 *      変換する必要がある．
 *      値そのものはどちらも擬似解像度準拠の値でOK．
 */
/* ------------------------------------------------------------------------- */
void MacOSMouseDelegate::MoveCursorToCenter(const NSWindow *window) noexcept
{
    // ウィンドウの中央を計算
    auto contentRect = [window contentRectForFrameRect: window.frame];
    auto center = CGPointMake(
                        contentRect.origin.x + contentRect.size.width * 0.5f,
                        contentRect.origin.y + contentRect.size.height * 0.5f);
    
    // ウィンドウのあるスクリーンの座標とサイズを取得
    auto screenFrame = window.screen.frame;

    // ウィンドウ中央を表すカーソル位置を算出
    auto cursorPosition = CGPointMake(
                            center.x - screenFrame.origin.x,
                            screenFrame.size.height - (center.y - screenFrame.origin.y));
    
    // マウスカーソルを移動
    auto displayId = [window.screen.deviceDescription[@"NSScreenNumber"] unsignedIntValue];
    CGDisplayMoveCursorToPoint(displayId, cursorPosition);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスカーソルを固定するかを取得
 *  \retval     true    固定する
 *  \retval     false   固定しない
 */
/* ------------------------------------------------------------------------- */
bool MacOSMouseDelegate::ShouldFixCursor(const NSWindow *window) const noexcept
{
    // ウィンドウが取得できていない状態なら固定しない
    if (window == nil)
    {
        return false;
    }
    
    // ウィンドウがキーウィンドウでなければ取得しない
    if (!window.isKeyWindow)
    {
        return false;
    }
    
    return true;
}

}   // namespace MGL::Input
#endif  // MGL_MOUSE_DELEGATE_ENABLE_MACOS
// vim: et ts=4 sw=4 sts=4
