// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_mouse_delegate_apple_gc.mm
 *  \brief      GameControllerフレームワークによるマウスデリゲート
 *  \date       Since: July 5, 2022. 7:00:57 JST.
 *  \author     Acerola
 *  \note
 *      色々と問題が多いので実装は見合わせ．
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/mouse/mgl_mouse_delegate_apple_gc.h>

#if defined(MGL_TARGET_MACOS) || defined(MGL_TARGET_IOS) || defined(MGL_TARGET_TVOS)

#include <mgl/event/mgl_event.h>

namespace MGL::Input
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      このデリゲートが利用可能かを取得
 *  \retval     true    利用可能
 *  \retval     false   利用不可
 */
/* ------------------------------------------------------------------------- */
bool AppleGCMouseDelegate::IsAvailable() noexcept
{
#if defined(MGL_TARGET_MACOS)
    // macOSは11.0以降
    if ([NSProcessInfo.processInfo isOperatingSystemAtLeastVersion:{11, 0, 0}])
    {
        return true;
    }
#elif defined(MGL_TARGET_IOS) || defined(MGL_TARGET_TVOS)
    // iOSとtvOSは14.0以降
    if ([NSProcessInfo.processInfo isOperatingSystemAtLeastVersion:{14, 0, 0}])
    {
        return true;
    }
#endif

    return false;

}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
AppleGCMouseDelegate::AppleGCMouseDelegate() noexcept
    : _currentInput(nil)
    , _becomeObserver(nil)
    , _stopObserver(nil)
    , _deltaX(0.0f)
    , _deltaY(0.0f)
{
    RegisterNotification();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デストラクタ
 */
/* ------------------------------------------------------------------------- */
AppleGCMouseDelegate::~AppleGCMouseDelegate() noexcept
{
    auto *defaultCenter = NSNotificationCenter.defaultCenter;
    if (_becomeObserver != nil)
    {
        [defaultCenter removeObserver:_becomeObserver];
    }
    
    if (_stopObserver != nil)
    {
        [defaultCenter removeObserver:_stopObserver];
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの通知登録
 */
/* ------------------------------------------------------------------------- */
void AppleGCMouseDelegate::RegisterNotification() noexcept
{
    if (@available(macOS 11.0, iOS 14.0, tvOS 14.0, *))
    {
        // マウスが有効化された際の通知
        _becomeObserver = [NSNotificationCenter.defaultCenter addObserverForName:GCMouseDidBecomeCurrentNotification
                                                                          object:nil
                                                                           queue:NSOperationQueue.mainQueue
                                                                      usingBlock:^(NSNotification * _Nonnull note)
        {
            (void)note;
            if (GCMouse.current != nil)
            {
                _currentInput = GCMouse.current.mouseInput;
                _currentInput.mouseMovedHandler = ^(GCMouseInput * _Nonnull mouse, float deltaX, float deltaY)
                {
                    (void)mouse;
                    OnMove(deltaX, deltaY);
                };
            }
        }];

        // マウスが無効化された際の通知
        _stopObserver = [NSNotificationCenter.defaultCenter addObserverForName:GCMouseDidStopBeingCurrentNotification
                                                                        object:nil
                                                                         queue:NSOperationQueue.mainQueue
                                                                    usingBlock:^(NSNotification * _Nonnull note)
        {
            (void)note;
            _currentInput = nil;
        }];
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスが移動した際の処理
 */
/* ------------------------------------------------------------------------- */
void AppleGCMouseDelegate::OnMove(float deltaX, float deltaY) noexcept
{
    _deltaX += deltaX;
    _deltaY += deltaY;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウス入力状態の更新処理
 *  \param[out] state   更新するステート
 */
/* ------------------------------------------------------------------------- */
void AppleGCMouseDelegate::UpdateState(MouseState &state) noexcept
{
    state.uvPosition = state.uvPrevPosition = MGL::Vector2();
    state.prevButtonFlags = state.buttonFlags;
    
    if (@available(macOS 11.0, iOS 14.0, tvOS 14.0, *))
    {
        if (_currentInput != nil)
        {
            state.deltaMove = MGL::Vector2(_deltaX, _deltaY);
            state.deltaWheel = MGL::Vector2(_currentInput.scroll.xAxis.value, _currentInput.scroll.yAxis.value);
            
            if (_currentInput.leftButton.isPressed)
            {
                state.buttonFlags |= MouseButton::Left;
            }
            if (_currentInput.rightButton.isPressed)
            {
                state.buttonFlags |= MouseButton::Right;
            }
            if (_currentInput.middleButton.isPressed)
            {
                state.buttonFlags |= MouseButton::Middle;
            }            
        }
        else
        {
            state.deltaMove = state.deltaWheel = MGL::Vector2();
            state.buttonFlags.Clear();
        }
    }
    
    _deltaX = _deltaY = 0;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの座標確定後の処理
 *  \param[in]  state   更新されたステート
 */
/* ------------------------------------------------------------------------- */
void AppleGCMouseDelegate::PostUpdatePosition([[maybe_unused]] const MouseState &state) noexcept
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルモードを設定
 *  \param[in]  cursorMode  設定するカーソルモード
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool AppleGCMouseDelegate::SetCursorMode([[maybe_unused]] CursorMode cursorMode) noexcept
{
    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルモードを取得
 *  \return     現在のカーソルモード
 */
/* ------------------------------------------------------------------------- */
CursorMode AppleGCMouseDelegate::GetCursorMode() const noexcept
{
    return CursorMode::DeltaMove;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルの表示モードを設定
 *  \param[in]  visibleMode 設定する表示モード
 */
/* ------------------------------------------------------------------------- */
void AppleGCMouseDelegate::SetCursorVisibleMode([[maybe_unused]] CursorVisibleMode visibleMode) noexcept
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カーソルの表示モードを取得
 *  \return     現在のカーソルの表示モード
 */
/* ------------------------------------------------------------------------- */
CursorVisibleMode AppleGCMouseDelegate::GetCursorVisibleMode() const noexcept
{
    return CursorVisibleMode::Invisible;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウスの接続状態を取得
 *  \retval     true    接続されている
 *  \retval     false   接続されていない
 */
/* ------------------------------------------------------------------------- */
bool AppleGCMouseDelegate::IsConnecting() const noexcept
{
    if (@available(macOS 11.0, iOS 14.0, tvOS 14.0, *))
    {
        return _currentInput != nullptr;
    }
    else
    {
        return false;
    }
}


}   // namespace MGL::Input

#endif  // defined(MGL_TARGET_MACOS) || defined(MGL_TARGET_IOS) || defined(MGL_TARGET_TVOS)

// vim: et ts=4 sw=4 sts=4
