// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_input_bounds.cc
 *  \brief      MGL 入力範囲
 *  \date       Since: January 1, 2021. 14:27:28 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/mgl_input_bounds.h>

namespace MGL::Input
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
InputBounds::InputBounds() noexcept
    : _scale(1.0f, 1.0f)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      範囲を設定
 *  \param[in]  bounds      有効範囲
 */
/* ------------------------------------------------------------------------- */
void InputBounds::Set(const Rectangle &bounds) noexcept
{
    _bounds = bounds;
    _scale.x = _scale.y = 1.0f;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      範囲を論理サイズ指定で設定
 *  \param[in]  bounds      有効範囲
 *  \param[in]  logicalSize 論理サイズ
 */
/* ------------------------------------------------------------------------- */
void InputBounds::SetWithLogicalSize(const Rectangle &bounds, const Vector2 &logicalSize) noexcept
{
    _bounds = bounds;
    if (IsValid())
    {
        _scale.x = logicalSize.x / _bounds.width;
        _scale.y = logicalSize.y / _bounds.height;
    }
    else
    {
        _scale.x = _scale.y = 1.0f;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      範囲をスケール指定で設定
 *  \param[in]  bounds      有効範囲
 *  \param[in]  scale       スケール
 */
/* ------------------------------------------------------------------------- */
void InputBounds::SetWithScale(const Rectangle &bounds, const Vector2 &scale) noexcept
{
    _bounds = bounds;
    _scale = scale;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      座標から範囲内の座標を取得
 *  \param[in]  position  座標
 *  \return     範囲を適用した座標
 */
/* ------------------------------------------------------------------------- */
Vector2 InputBounds::GetPosition(const Vector2 &position) const noexcept
{
    auto adjustedPosition(position - _bounds.GetPosition());
    adjustedPosition.x *= _scale.x;
    adjustedPosition.y *= _scale.y;

    return adjustedPosition;
}

}    // namespace MGL::Input

// vim: et ts=4 sw=4 sts=4
