// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_keyboard_server.cc
 *  \brief      MGL キーボードサーバ
 *  \date       Since: December 26, 2020. 21:23:31 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/keyboard/mgl_keyboard_server.h>

#include <mgl/mgl_environment.h>
#include <mgl/system/mgl_system_application.h>

namespace MGL::Input
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インスタンスの取得
 *  \return     インスタンスの参照
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<KeyboardServer> &KeyboardServer::GetInstanceRef() noexcept
{
    static STL::unique_ptr<KeyboardServer> sInstance = nullptr;
    return sInstance;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
KeyboardServer::KeyboardServer() noexcept
    : _eventPreFrameUpdate(Event::NotifyType::PreFrameUpdate, OnEventPreFrameUpdate, this)
    , _repeatTimerArray()
{
    _repeatTimerArray.fill(0.0f);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \param[in]  delegate    キーボード入力デリゲート
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool KeyboardServer::Initialize(STL::unique_ptr<KeyboardDelegate> &delegate) noexcept
{
    if (_delegate != nullptr)
    {
        return false;
    }

    _delegate = std::move(delegate);

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      指定したキーが押されているかを取得
 *  \param[in]  keycode     キーコード
 *  \retval     true        押されている
 *  \retval     false       押されていない
 */
/* ------------------------------------------------------------------------- */
bool KeyboardServer::IsPressing(Keycode keycode) const noexcept
{
    return _state[static_cast<size_t>(keycode)];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      指定したキーが押された瞬間を取得
 *  \param[in]  keycode     キーコード
 *  \retval     true        押された瞬間である
 *  \retval     false       押された瞬間ではない
 */
/* ------------------------------------------------------------------------- */
bool KeyboardServer::IsTriggered(Keycode keycode) const noexcept
{
    auto index = static_cast<size_t>(keycode);

    return _state[index] && !_prevState[index];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      指定したキーが離された瞬間を取得
 *  \param[in]  keycode     キーコード
 *  \retval     true        離された瞬間である
 *  \retval     false       離された瞬間ではない
 */
/* ------------------------------------------------------------------------- */
bool KeyboardServer::IsReleased(Keycode keycode) const noexcept
{
    auto index = static_cast<size_t>(keycode);

    return !_state[index] && _prevState[index];
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      指定したキーのリピート入力状態を取得
 *  \param[in]  keycode     キーコード
 *  \retval     true        リピート入力あり
 *  \retval     false       リピート入力なし
 */
/* ------------------------------------------------------------------------- */
bool KeyboardServer::IsARepeat(Keycode keycode) const noexcept
{
    auto index = static_cast<size_t>(keycode);

    return _repeatState[index];
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フレーム更新前のコールバック関数
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       未使用
 */
/* ------------------------------------------------------------------------- */
void KeyboardServer::OnEventPreFrameUpdate(void *callbackArg, [[maybe_unused]] void *notifyArg)
{
    auto *thisPtr = static_cast<KeyboardServer *>(callbackArg);

    if (thisPtr->_delegate != nullptr)
    {
        thisPtr->_prevState = thisPtr->_state;
        thisPtr->_state.reset();
        thisPtr->_delegate->UpdateState(thisPtr->_state);
        thisPtr->UpdateModifierKeys();
        thisPtr->UpdateRepeatInput();
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      修飾キーの更新処理
 */
/* ------------------------------------------------------------------------- */
void KeyboardServer::UpdateModifierKeys() noexcept
{
    // Ctrlキー
    if (IsPressing(Keycode::LeftControl) || IsPressing(Keycode::RightControl))
    {
        _state.set(static_cast<size_t>(Keycode::Control));
    }
    // Shiftキー
    if (IsPressing(Keycode::LeftShift) || IsPressing(Keycode::RightShift))
    {
        _state.set(static_cast<size_t>(Keycode::Shift));
    }
    // Option or Altキー
    if (IsPressing(Keycode::LeftAlt) || IsPressing(Keycode::RightAlt))
    {
        _state.set(static_cast<size_t>(Keycode::Alt));
    }
    // Command or Windowsキー
    if (IsPressing(Keycode::LeftGUI) || IsPressing(Keycode::RightGUI))
    {
        _state.set(static_cast<size_t>(Keycode::GUI));
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      キーリピートの更新処理
 */
/* ------------------------------------------------------------------------- */
void KeyboardServer::UpdateRepeatInput() noexcept
{
    const MGL::System::Application app;

    for (size_t i = 0; i < kKeycodeCount; i++)
    {
        auto keycode = static_cast<Keycode>(i);
        auto &timer = _repeatTimerArray[i];

        // 初回入力
        if (IsTriggered(keycode))
        {
            _repeatState[i] = true;
            timer = _repeatSetting.firstBlank;
        }
        // 継続入力
        else if (IsPressing(keycode))
        {
            // 経過時間を更新
            if (_repeatSetting.type == RepeatType::Frame)
            {
                timer -= 1.0f;
            }
            else if (_repeatSetting.type == RepeatType::Second)
            {
                timer -= app.GetFrameElapsedTime() / 1000.0f;
            }

            // 指定された時間が経過していたらリピート入力をオンにする
            if (timer <= 0.0f)
            {
                _repeatState[i] = true;
                timer = _repeatSetting.secondBlank;
            }
            else
            {
                _repeatState[i] = false;
            }
        }
        else
        {
            _repeatState[i] = false;
        }
    }
}
}    // namespace MGL::Input

// vim: et ts=4 sw=4 sts=4
