// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_keyboard_delegate_win32.cc
 *  \brief      MGL Win32用キーボードデリゲート
 *  \date       Since: March 30, 2021. 12:43:27 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/keyboard/mgl_keyboard_delegate_win32.h>
#if defined(MGL_KEYBOARD_DELEGATE_ENABLE_WIN32)

#include <Windows.h>

#include <mgl/system/mgl_system_window.h>

namespace MGL::Input
{
namespace
{
struct ConvertElement
{
    BYTE    win32Keycode;
    Keycode keycode;
};
constexpr ConvertElement kConvertTable[] =
{
    {'A',           Input::Keycode::KeyA},         //!< A
    {'B',           Input::Keycode::KeyB},         //!< B
    {'C',           Input::Keycode::KeyC},         //!< C
    {'D',           Input::Keycode::KeyD},         //!< D
    {'E',           Input::Keycode::KeyE},         //!< E
    {'F',           Input::Keycode::KeyF},         //!< F
    {'G',           Input::Keycode::KeyG},         //!< G
    {'H',           Input::Keycode::KeyH},         //!< H
    {'I',           Input::Keycode::KeyI},         //!< I
    {'J',           Input::Keycode::KeyJ},         //!< J
    {'K',           Input::Keycode::KeyK},         //!< K
    {'L',           Input::Keycode::KeyL},         //!< L
    {'M',           Input::Keycode::KeyM},         //!< M
    {'N',           Input::Keycode::KeyN},         //!< N
    {'O',           Input::Keycode::KeyO},         //!< O
    {'P',           Input::Keycode::KeyP},         //!< P
    {'Q',           Input::Keycode::KeyQ},         //!< Q
    {'R',           Input::Keycode::KeyR},         //!< R
    {'S',           Input::Keycode::KeyS},         //!< S
    {'T',           Input::Keycode::KeyT},         //!< T
    {'U',           Input::Keycode::KeyU},         //!< U
    {'V',           Input::Keycode::KeyV},         //!< V
    {'W',           Input::Keycode::KeyW},         //!< W
    {'X',           Input::Keycode::KeyX},         //!< X
    {'Y',           Input::Keycode::KeyY},         //!< Y
    {'Z',           Input::Keycode::KeyZ},         //!< Z
    {'1',           Input::Keycode::Key1},         //!< 1
    {'2',           Input::Keycode::Key2},         //!< 2
    {'3',           Input::Keycode::Key3},         //!< 3
    {'4',           Input::Keycode::Key4},         //!< 4
    {'5',           Input::Keycode::Key5},         //!< 5
    {'6',           Input::Keycode::Key6},         //!< 6
    {'7',           Input::Keycode::Key7},         //!< 7
    {'8',           Input::Keycode::Key8},         //!< 8
    {'9',           Input::Keycode::Key9},         //!< 9
    {'0',           Input::Keycode::Key0},         //!< 0
    {VK_RETURN,     Input::Keycode::Return},       //!< Return or Enter
    {VK_ESCAPE,     Input::Keycode::Escape},       //!< Escape
    {VK_BACK,       Input::Keycode::Backspace},    //!< Backspace
    {VK_TAB,        Input::Keycode::Tab},          //!< Tab
    {VK_SPACE,      Input::Keycode::Space},        //!< Space
    {VK_F1,         Input::Keycode::F1},           //!< F1
    {VK_F2,         Input::Keycode::F2},           //!< F2
    {VK_F3,         Input::Keycode::F3},           //!< F3
    {VK_F4,         Input::Keycode::F4},           //!< F4
    {VK_F5,         Input::Keycode::F5},           //!< F5
    {VK_F6,         Input::Keycode::F6},           //!< F6
    {VK_F7,         Input::Keycode::F7},           //!< F7
    {VK_F8,         Input::Keycode::F8},           //!< F8
    {VK_F9,         Input::Keycode::F9},           //!< F9
    {VK_F10,        Input::Keycode::F10},          //!< F10
    {VK_F11,        Input::Keycode::F11},          //!< F11
    {VK_F12,        Input::Keycode::F12},          //!< F12
    {VK_DELETE,     Input::Keycode::Delete},       //!< Delete
    {VK_RIGHT,      Input::Keycode::Right},        //!< →
    {VK_LEFT,       Input::Keycode::Left},         //!< ←
    {VK_DOWN,       Input::Keycode::Down},         //!< ↓
    {VK_UP,         Input::Keycode::Up},           //!< ↑
    {VK_LCONTROL,   Input::Keycode::LeftControl},  //!< 左Ctrl
    {VK_LSHIFT,     Input::Keycode::LeftShift},    //!< 左Shift
    {VK_LMENU,      Input::Keycode::LeftAlt},      //!< 左Option or 左Alt
    {VK_LWIN,       Input::Keycode::LeftGUI},      //!< 左Command or 左Windows
    {VK_RCONTROL,   Input::Keycode::RightControl}, //!< 右Ctrl
    {VK_RSHIFT,     Input::Keycode::RightShift},   //!< 右Shift
    {VK_RMENU,      Input::Keycode::RightAlt},     //!< 右Option or 右Alt
    {VK_RWIN,       Input::Keycode::RightGUI},     //!< 右Command or 右Windows
};
}   // namespace


/* ------------------------------------------------------------------------- */
/*!
 *  \brief       入力状態の更新処理
 *  \param[out]  state      入力状態の格納先
 */
/* ------------------------------------------------------------------------- */
void Win32KeyboardDelegate::UpdateState(KeycodeArray &state) noexcept
{
    if (MGL::System::Window().IsFocused())
    {
        for (const auto &element : kConvertTable)
        {
            state[static_cast<size_t>(element.keycode)] = (GetAsyncKeyState(element.win32Keycode) & 0x8000u) != 0;
        }
    }
    else
    {
        state.reset();
    }
}
}   // namespace MGL::Input

#endif  // MGL_KEYBOARD_DELEGATE_ENABLE_WIN32
// vim: et ts=4 sw=4 sts=4
