// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_keyboard_delegate_event.cc
 *  \brief      MGL イベントハンドリングによるキーボード入力デリゲート
 *  \date       Since: December 27, 2020. 18:08:12 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/keyboard/mgl_keyboard_delegate_event.h>
#if defined(MGL_KEYBOARD_DELEGATE_ENABLE_EVENT)

namespace MGL::Input
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
EventKeyboardDelegate::EventKeyboardDelegate() noexcept
    : _eventKeyDown(Event::NotifyType::InputKeyDown, OnEventKeyDown, this)
    , _eventKeyUp(Event::NotifyType::InputKeyUp, OnEventKeyUp, this)
    , _eventShouldClearInput(Event::NotifyType::ShouldClearInput, OnEventShouldClearInput, this)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief       入力状態の更新処理
 *  \param[out]  state      入力状態の格納先
 */
/* ------------------------------------------------------------------------- */
void EventKeyboardDelegate::UpdateState(KeycodeArray &state) noexcept
{
    state = _temporaryState;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      キーダウンイベントのコールバック関数
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       押されたキーのコード
 */
/* ------------------------------------------------------------------------- */
void EventKeyboardDelegate::OnEventKeyDown(void *callbackArg, void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<EventKeyboardDelegate *>(callbackArg);
    auto *keycode = static_cast<Keycode *>(notifyArg);

    thisPtr->SetState(*keycode, true);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      キーアップイベントのコールバック関数
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       離されたキーのコード
 */
/* ------------------------------------------------------------------------- */
void EventKeyboardDelegate::OnEventKeyUp(void *callbackArg, void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<EventKeyboardDelegate *>(callbackArg);
    auto *keycode = static_cast<Keycode *>(notifyArg);

    thisPtr->SetState(*keycode, false);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      入力クリア要求イベントのコールバック関数
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       未使用
 */
/* ------------------------------------------------------------------------- */
void EventKeyboardDelegate::OnEventShouldClearInput(void *callbackArg, [[maybe_unused]] void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<EventKeyboardDelegate *>(callbackArg);

    thisPtr->_temporaryState.reset();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      入力状態の設定
 *  \param[in]  keycode     設定するキーのコード
 *  \param[in]  isPressed   押されているかのフラグ
 */
/* ------------------------------------------------------------------------- */
void EventKeyboardDelegate::SetState(Keycode keycode, bool isPressed) noexcept
{
    auto index = static_cast<size_t>(keycode);
    _temporaryState[index] = isPressed;
}
}    // namespace MGL::Input

#endif    // MGL_KEYBOARD_DELEGATE_ENABLE_EVENT

// vim: et ts=4 sw=4 sts=4
