// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_keyboard_delegate_apple_gc.cc
 *  \brief      Apple GameControllerフレームワークによるキーボードデリゲート
 *  \date       Since: July 5, 2022. 1:24:30 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/keyboard/mgl_keyboard_delegate_apple_gc.h>

#if defined(MGL_TARGET_MACOS) || defined(MGL_TARGET_IOS) || defined(MGL_TARGET_TVOS)

#include <GameController/GameController.h>

#include <mgl/event/mgl_event.h>

namespace MGL::Input
{
namespace
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      MGLのキーコードからGCKeyboardのキーコードを取得
 *  \param[out] gcKeycode   GCKeyboardのキーコード
 *  \param[in]  keycode     MGLのキーコード
 *  \retval     true        取得成功
 *  \retval     false       取得失敗
 *  \note
 *      本当はテーブル化したかったけど，GCKeyCodeをstaticで定義すると
 *      対応バージョン以前のOSで実行できなくなってしまう．
 */
/* ------------------------------------------------------------------------- */
API_AVAILABLE(macos(11.0), ios(14.0), tvos(14.0))
constexpr bool GetGCKeyCode(GCKeyCode &gcKeycode, Keycode keycode) noexcept
{
    switch (keycode)
    {
        case Keycode::KeyA:         gcKeycode = GCKeyCodeKeyA;                 return true;     // A
        case Keycode::KeyB:         gcKeycode = GCKeyCodeKeyB;                 return true;     // B
        case Keycode::KeyC:         gcKeycode = GCKeyCodeKeyC;                 return true;     // C
        case Keycode::KeyD:         gcKeycode = GCKeyCodeKeyD;                 return true;     // D
        case Keycode::KeyE:         gcKeycode = GCKeyCodeKeyE;                 return true;     // E
        case Keycode::KeyF:         gcKeycode = GCKeyCodeKeyF;                 return true;     // F
        case Keycode::KeyG:         gcKeycode = GCKeyCodeKeyG;                 return true;     // G
        case Keycode::KeyH:         gcKeycode = GCKeyCodeKeyH;                 return true;     // H
        case Keycode::KeyI:         gcKeycode = GCKeyCodeKeyI;                 return true;     // I
        case Keycode::KeyJ:         gcKeycode = GCKeyCodeKeyJ;                 return true;     // J
        case Keycode::KeyK:         gcKeycode = GCKeyCodeKeyK;                 return true;     // K
        case Keycode::KeyL:         gcKeycode = GCKeyCodeKeyL;                 return true;     // L
        case Keycode::KeyM:         gcKeycode = GCKeyCodeKeyM;                 return true;     // M
        case Keycode::KeyN:         gcKeycode = GCKeyCodeKeyN;                 return true;     // N
        case Keycode::KeyO:         gcKeycode = GCKeyCodeKeyO;                 return true;     // O
        case Keycode::KeyP:         gcKeycode = GCKeyCodeKeyP;                 return true;     // P
        case Keycode::KeyQ:         gcKeycode = GCKeyCodeKeyQ;                 return true;     // Q
        case Keycode::KeyR:         gcKeycode = GCKeyCodeKeyR;                 return true;     // R
        case Keycode::KeyS:         gcKeycode = GCKeyCodeKeyS;                 return true;     // S
        case Keycode::KeyT:         gcKeycode = GCKeyCodeKeyT;                 return true;     // T
        case Keycode::KeyU:         gcKeycode = GCKeyCodeKeyU;                 return true;     // U
        case Keycode::KeyV:         gcKeycode = GCKeyCodeKeyV;                 return true;     // V
        case Keycode::KeyW:         gcKeycode = GCKeyCodeKeyW;                 return true;     // W
        case Keycode::KeyX:         gcKeycode = GCKeyCodeKeyX;                 return true;     // X
        case Keycode::KeyY:         gcKeycode = GCKeyCodeKeyY;                 return true;     // Y
        case Keycode::KeyZ:         gcKeycode = GCKeyCodeKeyZ;                 return true;     // Z
        case Keycode::Key1:         gcKeycode = GCKeyCodeOne;                  return true;     // 1
        case Keycode::Key2:         gcKeycode = GCKeyCodeTwo;                  return true;     // 2
        case Keycode::Key3:         gcKeycode = GCKeyCodeThree;                return true;     // 3
        case Keycode::Key4:         gcKeycode = GCKeyCodeFour;                 return true;     // 4
        case Keycode::Key5:         gcKeycode = GCKeyCodeFive;                 return true;     // 5
        case Keycode::Key6:         gcKeycode = GCKeyCodeSix;                  return true;     // 6
        case Keycode::Key7:         gcKeycode = GCKeyCodeSeven;                return true;     // 7
        case Keycode::Key8:         gcKeycode = GCKeyCodeEight;                return true;     // 8
        case Keycode::Key9:         gcKeycode = GCKeyCodeNine;                 return true;     // 9
        case Keycode::Key0:         gcKeycode = GCKeyCodeZero;                 return true;     // 0
        case Keycode::Return:       gcKeycode = GCKeyCodeReturnOrEnter;        return true;     // Return or Enter
        case Keycode::Escape:       gcKeycode = GCKeyCodeEscape;               return true;     // Escape
        case Keycode::Backspace:    gcKeycode = GCKeyCodeDeleteOrBackspace;    return true;     // Backspace
        case Keycode::Tab:          gcKeycode = GCKeyCodeTab;                  return true;     // Tab
        case Keycode::Space:        gcKeycode = GCKeyCodeSpacebar;             return true;     // Space
        case Keycode::F1:           gcKeycode = GCKeyCodeF1;                   return true;     // F1
        case Keycode::F2:           gcKeycode = GCKeyCodeF2;                   return true;     // F2
        case Keycode::F3:           gcKeycode = GCKeyCodeF3;                   return true;     // F3
        case Keycode::F4:           gcKeycode = GCKeyCodeF4;                   return true;     // F4
        case Keycode::F5:           gcKeycode = GCKeyCodeF5;                   return true;     // F5
        case Keycode::F6:           gcKeycode = GCKeyCodeF6;                   return true;     // F6
        case Keycode::F7:           gcKeycode = GCKeyCodeF7;                   return true;     // F7
        case Keycode::F8:           gcKeycode = GCKeyCodeF8;                   return true;     // F8
        case Keycode::F9:           gcKeycode = GCKeyCodeF9;                   return true;     // F9
        case Keycode::F10:          gcKeycode = GCKeyCodeF10;                  return true;     // F10
        case Keycode::F11:          gcKeycode = GCKeyCodeF11;                  return true;     // F11
        case Keycode::F12:          gcKeycode = GCKeyCodeF12;                  return true;     // F12
        case Keycode::Delete:       gcKeycode = GCKeyCodeDeleteForward;        return true;     // Delete
        case Keycode::Right:        gcKeycode = GCKeyCodeRightArrow;           return true;     // →
        case Keycode::Left:         gcKeycode = GCKeyCodeLeftArrow;            return true;     // ←
        case Keycode::Down:         gcKeycode = GCKeyCodeDownArrow;            return true;     // ↓
        case Keycode::Up:           gcKeycode = GCKeyCodeUpArrow;              return true;     // ↑
        case Keycode::LeftControl:  gcKeycode = GCKeyCodeLeftControl;          return true;     // 左Ctrl
        case Keycode::LeftShift:    gcKeycode = GCKeyCodeLeftShift;            return true;     // 左Shift
        case Keycode::LeftAlt:      gcKeycode = GCKeyCodeLeftAlt;              return true;     // 左Option or 左Alt
        case Keycode::LeftGUI:      gcKeycode = GCKeyCodeLeftGUI;              return true;     // 左Command or 左Windows
        case Keycode::RightControl: gcKeycode = GCKeyCodeRightControl;         return true;     // 右Ctrl
        case Keycode::RightShift:   gcKeycode = GCKeyCodeRightShift;           return true;     // 右Shift
        case Keycode::RightAlt:     gcKeycode = GCKeyCodeRightAlt;             return true;     // 右Option or 右Alt
        case Keycode::RightGUI:     gcKeycode = GCKeyCodeRightGUI;             return true;     // 右Command or 右Windows
            
        // 以下は対応するキー無し
        case Keycode::Control:
        case Keycode::Shift:
        case Keycode::Alt:
        case Keycode::GUI:
        case Keycode::Invalid:
            return false;
    }

    return false;
}
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      このデリゲートが利用可能かを取得
 *  \retval     true    利用可能
 *  \retval     false   利用不可
 */
/* ------------------------------------------------------------------------- */
bool AppleGCKeyboardDelegate::IsAvailable() noexcept
{
#if defined(MGL_TARGET_MACOS)
    // macOSは11.0以降
    if ([NSProcessInfo.processInfo isOperatingSystemAtLeastVersion:{11, 0, 0}])
    {
        return true;
    }
#elif defined(MGL_TARGET_IOS) || defined(MGL_TARGET_TVOS)
    // iOSとtvOSは14.0以降
    if ([NSProcessInfo.processInfo isOperatingSystemAtLeastVersion:{14, 0, 0}])
    {
        return true;
    }
#endif

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
AppleGCKeyboardDelegate::AppleGCKeyboardDelegate() noexcept
    : _isConnecting(false)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief       入力状態の更新処理
 *  \param[out]  state      入力状態の格納先
 */
/* ------------------------------------------------------------------------- */
void AppleGCKeyboardDelegate::UpdateState(KeycodeArray &state) noexcept
{
    state.reset();

    bool isConnecting = false;

    // ステートの更新処理
    if (@available(macOS 11.0, iOS 14.0, tvOS 14.0, *))
    {
        if (auto *keyboard = GCKeyboard.coalescedKeyboard; keyboard != nil)
        {
            if (auto *input = keyboard.keyboardInput; input != nil)
            {
                for (uint8_t i = 0; i < kKeycodeCount; i++)
                {
                    Keycode keycode{i};
                    GCKeyCode gcKeycode = 0;
                    if (GetGCKeyCode(gcKeycode, keycode))
                    {
                        if ([input buttonForKeyCode:gcKeycode].isPressed)
                        {
                            state[i] = true;
                        }
                    }
                }
                
                isConnecting = true;
            }
        }
    }
    
    // 接続状態の更新
    if (_isConnecting != isConnecting)
    {
        // 接続状態が変化した場合はイベントを発行する
        auto notifyType = isConnecting ? Event::NotifyType::ConnectKeyboard : Event::NotifyType::DisconnectKeyboard;
        Event::Notify(notifyType);
    }
    _isConnecting = isConnecting;
}
}   // namespace MGL::Input

#endif  // defined(MGL_TARGET_MACOS) || defined(MGL_TARGET_IOS) || defined(MGL_TARGET_TVOS)
// vim: et ts=4 sw=4 sts=4
