// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_gamepad_state.cc
 *  \brief      MGL ゲームパッドステート
 *  \date       Since: January 7, 2021. 12:39:51 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/gamepad/mgl_gamepad_state.h>

#include <mgl/event/mgl_event.h>
#include <mgl/input/gamepad/mgl_gamepad_server.h>
#include <mgl/system/mgl_system_application.h>
#include <mgl/system/mgl_system_debug_macro.h>

namespace MGL::Input
{
namespace
{
constexpr float kAnalogStickThreshold = 0.3f;    // アナログスティックの閾値

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ユニークIDの生成
 *  \return     新たに生成されたユニークID
 */
/* ------------------------------------------------------------------------- */
PadID MakeUniqueID() noexcept
{
    static PadID sUniqueID = PadID::UniqueIDStart;

    auto newID = sUniqueID;

    sUniqueID = PadID{static_cast<std::underlying_type_t<PadID>>(sUniqueID) + 1};
    if (sUniqueID >= PadID::UniqueIDEnd)
    {
        sUniqueID = PadID::UniqueIDStart;
    }

    return newID;
}
}    // namespace


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
PadState::PadState() noexcept
    : _repeatTimerArray()
{
    _repeatTimerArray.fill(0.0f);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      更新前の処理
 */
/* ------------------------------------------------------------------------- */
void PadState::PreUpdate() noexcept
{
    if (IsEnabled())
    {
        // 要求があればアクティベート解除
        if (_shouldDeactivate)
        {
            Deactivate();
            _shouldDeactivate = false;
        }

        _prevButtonFlags = _buttonFlags;
        _buttonFlags.Clear();
        _leftStick = _rightStick = Vector2(0.0f, 0.0f);
    }
    else
    {
        // 要求があればアクティベート
        if (_shouldActivate)
        {
            Activate();
            _shouldActivate = false;
            _buttonFlags.Clear();
            _prevButtonFlags.Clear();
            _leftStick = _rightStick = Vector2(0.0f, 0.0f);
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      更新後の処理
 *  \param[in]  repeatSetting   リピート入力設定
 */
/* ------------------------------------------------------------------------- */
void PadState::PostUpdate(const RepeatSetting &repeatSetting) noexcept
{
    // PadButton::Noneは常に入力なしにする
    _buttonFlags.Clear(PadButton::None);

    // 決定・キャンセルボタンの反映
    ApplyDecideCancelButton();

    // リピート入力の更新
    UpdateRepeatFlags(repeatSetting);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アクティベート処理
 */
/* ------------------------------------------------------------------------- */
void PadState::Activate() noexcept
{
    _identifier = MakeUniqueID();

    MGL_LIBRARY_TRACE("[MGL Gamepad] Connected: ID = %d: %s - %s: (%04X-%04X)",
                      _identifier,
                      _deviceInfo.vendorName.c_str(),
                      _deviceInfo.productName.c_str(),
                      _deviceInfo.vendorID,
                      _deviceInfo.productID);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アクティベート解除処理
 */
/* ------------------------------------------------------------------------- */
void PadState::Deactivate() noexcept
{
    MGL_LIBRARY_TRACE("[MGL Gamepad] Disconnected: ID = %d: %s - %s: (%04X-%04X)",
                      _identifier,
                      _deviceInfo.vendorName.c_str(),
                      _deviceInfo.productName.c_str(),
                      _deviceInfo.vendorID,
                      _deviceInfo.productID);

    if (HasEntry())
    {
        Event::Notify(Event::NotifyType::DisconnectGamepad, this);
        GamepadServer::GetInstance().Leave(*this);
    }

    _identifier = PadID::Invalid;
    _deviceInfo = PadDeviceInfo();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      決定とキャンセルのボタンを適用
 */
/* ------------------------------------------------------------------------- */
void PadState::ApplyDecideCancelButton() noexcept
{
    if (_decideButton != PadButton::None)
    {
        if (IsPressing(_decideButton))
        {
            SetButton(PadButton::Decide);
        }
    }
    if (_cancelButton != PadButton::None)
    {
        if (IsPressing(_cancelButton))
        {
            SetButton(PadButton::Cancel);
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アナログスティックからD-Padを適用する
 */
/* ------------------------------------------------------------------------- */
void PadState::ApplyLeftStickToDPad() noexcept
{
    constexpr struct
    {
        float angle{};
        PadButtonFlags flags;
    } kTable[] =
    {
        {-2.61799f, PadButtonFlags(PadButton::AnalogLeft)},
        {-2.09439f, PadButtonFlags(PadButton::AnalogLeft) | PadButton::AnalogDown},
        {-1.04719f, PadButtonFlags(PadButton::AnalogDown)},
        {-0.52359f, PadButtonFlags(PadButton::AnalogDown) | PadButton::AnalogRight},
        { 0.52359f, PadButtonFlags(PadButton::AnalogRight)},
        { 1.04719f, PadButtonFlags(PadButton::AnalogRight) | PadButton::AnalogUp},
        { 2.09439f, PadButtonFlags(PadButton::AnalogUp)},
        { 2.61799f, PadButtonFlags(PadButton::AnalogUp) | PadButton::AnalogLeft},
        { 3.2f,     PadButtonFlags(PadButton::AnalogLeft)}
    };


    if (_leftStick.Length() >= kAnalogStickThreshold)
    {
        const float angle = std::atan2(_leftStick.y, _leftStick.x);

        for (const auto &table : kTable)
        {
            if (table.angle >= angle)
            {
                SetButton(table.flags);
                break;
            }
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      リピート入力フラグを更新
 *  \param[in]  repeatSetting   リピート入力設定
 */
/* ------------------------------------------------------------------------- */
void PadState::UpdateRepeatFlags(const RepeatSetting &repeatSetting) noexcept
{
    const MGL::System::Application application;

    for (size_t i = 1; i < _repeatTimerArray.size(); i++)
    {
        auto button = PadButton(i);
        auto &timer = _repeatTimerArray[i];

        // 初回入力
        if (IsTriggered(button))
        {
            _repeatFlags.Set(button);
            timer = repeatSetting.firstBlank;
        }
        // 継続入力
        else if (IsPressing(button))
        {
            // 経過時間を更新
            if (repeatSetting.type == RepeatType::Frame)
            {
                timer -= 1.0f;
            }
            else if (repeatSetting.type == RepeatType::Second)
            {
                timer -= application.GetFrameElapsedTime() / 1000.0f;
            }

            // 指定された時間が経過していたらリピート入力をオンにする
            if (timer <= 0.0f)
            {
                _repeatFlags.Set(button);
                timer = repeatSetting.secondBlank;
            }
            else
            {
                _repeatFlags.Clear(button);
            }
        }
        else
        {
            _repeatFlags.Clear(button);
        }
    }
}

}    // namespace MGL::Input
// vim: et ts=4 sw=4 sts=4
