// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_gamepad_server.cc
 *  \brief      MGL ゲームパッドサーバ
 *  \date       Since: January 3, 2021. 15:11:07 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/gamepad/mgl_gamepad_server.h>

#include <mgl/initialize/mgl_initializer_delegate.h>
#include <mgl/system/mgl_system_window.h>


namespace MGL::Input
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インスタンスの取得
 *  \return     インスタンスの参照
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<GamepadServer> &GamepadServer::GetInstanceRef() noexcept
{
    static STL::unique_ptr<GamepadServer> sInstance = nullptr;
    return sInstance;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
GamepadServer::GamepadServer() noexcept
    : _eventPreFrameUpdate(Event::NotifyType::PreFrameUpdate, OnEventPreFrameUpdate, this)
    , _entryIndexes()
{
    _entryIndexes.fill(-1);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デリゲートの追加
 *  \param[in]  delegate    追加するデリゲート
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool GamepadServer::AddDelegate(STL::unique_ptr<GamepadDelegate> delegate) noexcept
{
    if (!delegate->Initialize())
    {
        return false;
    }

    _delegates.push_back(std::move(delegate));

    return true;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      フレーム更新前のコールバック関数
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       未使用
 */
/* ------------------------------------------------------------------------- */
void GamepadServer::OnEventPreFrameUpdate(void *callbackArg, [[maybe_unused]] void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<GamepadServer *>(callbackArg);

    const std::scoped_lock lock(thisPtr->_mutexPadStates);

    // 更新前の処理
    for (auto &state : thisPtr->_padStates)
    {
        state.PreUpdate();
    }

    // 登録されているデリゲートを用いてステートを更新
    if (System::Window().IsFocused())
    {
        for (auto &delegate : thisPtr->_delegates)
        {
            delegate->UpdateState();
        }
    }

    // ボタン入力更新後の処理
    for (auto &state : thisPtr->_padStates)
    {
        if (state.IsEnabled())
        {
            state.PostUpdate(thisPtr->_repeatSetting);
        }
    }

    // 高プライオリティのパッドがあるかをチェック
    bool existHighPriorityPad = false;
    for (auto &state : thisPtr->_padStates)
    {
        if (state.IsEnabled() && (state.GetPriority() == PadPriority::High))
        {
            existHighPriorityPad = true;
            break;
        }
    }

    // 高プライオリティのパッドが存在しており，今まで存在していなかった場合，低プライオリティのパッドのエントリーを解除する
    if (existHighPriorityPad && !thisPtr->_existsHighPriorityPad)
    {
        for (auto &state : thisPtr->_padStates)
        {
            if (state.HasEntry() && (state.GetPriority() == PadPriority::Low))
            {
                Event::Notify(Event::NotifyType::DisconnectGamepad, &state);
                thisPtr->Leave(state);
            }
        }
    }

    // 高プライオリティの存在フラグを更新
    thisPtr->_existsHighPriorityPad = existHighPriorityPad;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      未使用のパッドステートを要求
 *  \return     パッドステートのインスタンスのアドレス．空きがない場合はnullptr
 */
/* ------------------------------------------------------------------------- */
PadState *GamepadServer::RequestsFreePadState() noexcept
{
    const std::scoped_lock lock(_mutexPadStates);

    for (auto &state : _padStates)
    {
        if (!state.IsUsing())
        {
            state.SetUsing(true);
            return &state;
        }
    }

    return nullptr;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      パッドステートをサーバに返却
 *  \param[in]  returnState 返却するパッドステート
 */
/* ------------------------------------------------------------------------- */
void GamepadServer::ReturnPadState(PadState *returnState) noexcept
{
    const std::scoped_lock lock(_mutexPadStates);

    for (auto &state : _padStates)
    {
        if (state == *returnState)
        {
            state.SetUsing(false);
            break;
        }
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      空いているパッドステートの数を取得
 */
/* ------------------------------------------------------------------------- */
size_t GamepadServer::GetFreePadStateCount() const noexcept
{
    size_t count = 0;

    for (const auto &state : _padStates)
    {
        if (!state.IsUsing())
        {
            ++count;
        }
    }

    return count;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ゲームパッドのエントリー
 *  \param[in]  entryState  エントリーするパッドステート
 *  \param[in]  entry       エントリー番号．Autoを指定すると空いている番号を自動で設定
 *  \return     エントリーされた番号．失敗した場合はNoEntry
 */
/* ------------------------------------------------------------------------- */
PadEntry GamepadServer::Entry(const PadState &entryState, PadEntry entry) noexcept
{
    // 無効なステートを指定された場合は失敗
    if (!entryState.IsEnabled())
    {
        return PadEntry::NoEntry;
    }

    const std::scoped_lock lock(_mutexEntry);

    // Autoを指定した場合，空いているエントリー番号を検索する
    if (entry == PadEntry::Auto)
    {
        for (size_t i = 0; i < _entryIndexes.size(); ++i)
        {
            if (_entryIndexes[i] < 0)
            {
                entry = static_cast<PadEntry>(i);
                break;
            }
        }
    }

    // entryがエントリー番号でなければ失敗
    if (entry >= PadEntry::Reserve_Start)
    {
        return PadEntry::NoEntry;
    }

    // エントリー処理
    for (size_t stateIndex = 0; stateIndex < kGamepadMax; ++stateIndex)
    {
        auto &state = _padStates[stateIndex];
        if (state == entryState)
        {
            // エントリーするステートとエントリー番号に既にエントリーしているステートは解除する
            Leave(state);
            Leave(GetPadState(entry));

            // エントリー設定
            state.SetEntry(entry);
            SetEntryIndex(entry, static_cast<int32_t>(stateIndex));

            return entry;
        }
    }

    return PadEntry::NoEntry;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ゲームパッドのエントリーの解除
 *  \param[in]  leaveState          エントリーを解除するパッドステート
 */
/* ------------------------------------------------------------------------- */
void GamepadServer::Leave(const PadState &leaveState) noexcept
{
    const std::scoped_lock lock(_mutexEntry);

    // エントリーしていないステートは無視
    if (!leaveState.HasEntry())
    {
        return;
    }

    // エントリー解除
    for (auto &state : _padStates)
    {
        if (state == leaveState)
        {
            SetEntryIndex(state.GetEntry(), -1);
            state.SetEntry(PadEntry::NoEntry);

            break;
        }
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      全てのエントリーを解除
 */
/* ------------------------------------------------------------------------- */
void GamepadServer::LeaveAll() noexcept
{
    const std::scoped_lock lock(_mutexEntry);

    for (size_t entryIndex = 0; entryIndex < _entryIndexes.size(); ++entryIndex)
    {
        if (auto index = GetEntryIndex(static_cast<PadEntry>(entryIndex)); index)
        {
            Leave(_padStates[index.value()]);
        }
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ゲームパッドステートを取得
 *  \param[in]  entry   エントリー番号．Any, Auto, NoEntry指定可能
 *  \return     entryに対応したゲームパッドステート．見つからない場合は無効なステートが返る
 */
/* ------------------------------------------------------------------------- */
const PadState &GamepadServer::GetPadState(PadEntry entry) noexcept
{
    const std::scoped_lock lockPadState(_mutexPadStates);
    const std::scoped_lock lockEntry(_mutexEntry);

    // Any指定: エントリーの有無に関わらず入力のあるゲームパッドを返す
    if (entry == PadEntry::Any)
    {
        for (auto &state : _padStates)
        {
            if (_existsHighPriorityPad && (state.GetPriority() == PadPriority::Low))
            {
                continue;
            }

            if (state.HasInputButton())
            {
                return state;
            }
        }
    }
    // NoEntry指定: エントリーされていないゲームパッドのうち，入力のあるものを返す
    else if (entry == PadEntry::NoEntry)
    {
        for (auto &state : _padStates)
        {
            if (_existsHighPriorityPad && (state.GetPriority() == PadPriority::Low))
            {
                continue;
            }

            if (state.HasInputButton() && !state.HasEntry())
            {
                return state;
            }
        }
    }
    // Auto指定: エントリーされているゲームパッドがある場合はそのゲームパッドを返す．無い場合はNoEntryと同じ
    else if (entry == PadEntry::Auto)
    {
        for (size_t entryIndex = 0; entryIndex < _entryIndexes.size(); ++entryIndex)
        {
            if (const auto index = GetEntryIndex(static_cast<PadEntry>(entryIndex)); index)
            {
                return _padStates[index.value()];
            }
        }

        return GetPadState(PadEntry::NoEntry);
    }
    // 指定されたエントリー番号を返す
    else if (const auto index = GetEntryIndex(entry); index)
    {
        return _padStates[index.value()];
    }

    return _invalidPadState;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      エントリー番号から対応したゲームパッドステート配列のインデックスを返す
 *  \param[in]  entry   エントリー番号
 *  \return     対応したインデックス．見つからない場合はstd::nullopt
 */
/* ------------------------------------------------------------------------- */
std::optional<size_t> GamepadServer::GetEntryIndex(PadEntry entry) const noexcept
{
    if (entry >= PadEntry::Reserve_Start)
    {
        return std::nullopt;
    }

    const auto index = _entryIndexes[static_cast<size_t>(entry)];
    if (index < 0)
    {
        return std::nullopt;
    }

    return static_cast<size_t>(index);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      エントリー番号に対応したゲームパッドステート配列のインデックスを設定する
 *  \param[in]  entry   エントリー番号
 *  \param[in]  index   エントリー番号が示すゲームパッドステートのインデックス
 */
/* ------------------------------------------------------------------------- */
void GamepadServer::SetEntryIndex(PadEntry entry, int32_t index) noexcept
{
    if (entry >= PadEntry::Reserve_Start)
    {
        return;
    }

    _entryIndexes[static_cast<size_t>(entry)] = index;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      決定ボタンとキャンセルボタンの割り当て
 *  \param[in]  type            パッドの種類
 *  \param[in]  decideButton    決定ボタン
 *  \param[in]  cancelButton    キャンセルボタン
 */
/* ------------------------------------------------------------------------- */
void GamepadServer::SetDecideCancelButton(PadType type, PadButton decideButton, PadButton cancelButton) noexcept
{
    auto typeSettings = _typeSettings[type];

    typeSettings.decideButton = decideButton;
    typeSettings.cancelButton = cancelButton;

    // 既に接続されているHIDゲームパッドのステートにも設定する
    for (auto &state : _padStates)
    {
        if (state.IsEnabled())
        {
            if (state.GetType() == type)
            {
                state.SetDecideButton(typeSettings.decideButton);
                state.SetCancelButton(typeSettings.cancelButton);
            }
        }
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      決定ボタンの割り当てを取得
 *  \param[in]  type            パッドの種類
 *  \param[in]  defaultButton   設定が存在しない場合に割り当てるボタン
 *  \return     決定ボタンに割り当てられているボタン
 */
/* ------------------------------------------------------------------------- */
PadButton GamepadServer::GetDecideButton(PadType type, PadButton defaultButton) const noexcept
{
    auto it = _typeSettings.find(type);
    if (it == _typeSettings.end())
    {
        return defaultButton;
    }

    if (it->second.decideButton == PadButton::None)
    {
        return defaultButton;
    }

    return it->second.decideButton;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      キャンセルボタンの割り当てを取得
 *  \param[in]  type            パッドの種類
 *  \param[in]  defaultButton   設定が存在しない場合に割り当てるボタン
 *  \return     キャンセルボタンに割り当てられているボタン
 */
/* ------------------------------------------------------------------------- */
PadButton GamepadServer::GetCancelButton(PadType type, PadButton defaultButton) const noexcept
{
    const auto it = _typeSettings.find(type);
    if (it == _typeSettings.end())
    {
        return defaultButton;
    }

    if (it->second.cancelButton == PadButton::None)
    {
        return defaultButton;
    }

    return it->second.cancelButton;
}
}    // namespace MGL::Input
// vim: et ts=4 sw=4 sts=4
