// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_gamepad_delegate_xinput.cc
 *  \brief      MGL XInputゲームパッドデリゲート
 *  \date       Since: April 1, 2021. 15:27:27 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/gamepad/mgl_gamepad_delegate_xinput.h>
#if defined(MGL_GAMEPAD_DELEGATE_ENABLE_XINPUT)

#include <Windows.h>
#include <Xinput.h>

#include <mgl/input/gamepad/mgl_gamepad_server.h>
#include <mgl/system/mgl_system_window.h>

#pragma comment(lib, "xinput.lib")

namespace MGL::Input
{
namespace
{
    constexpr uint8_t kTriggerThreshold = 0x7F;     //!< LRトリガーをボタンとして認識させる閾値

    // NOTE:
    //  スティックの遊びは一応設定してあるけど，古いXBox360コントローラは
    //  ニュートラル時のブレがかなり大きいのであんまり意味がない．
    constexpr int16_t kStickDeadzonePlus = 2000;    //!< スティックの遊び（＋方向）
    constexpr int16_t kStickDeadzoneMinus = -2000;  //!< スティックの遊び（－方向）
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool XInputGamepadDelegate::Initialize() noexcept
{
    // ゲームパッドサーバに必要数の未使用パッドがなければ失敗
    if (_server.GetFreePadStateCount() < _padArray.size())
    {
        return false;
    }

    // ゲームパッドサーバから空いているステートを取得してプレイヤーインデックスを設定
    for (size_t index = 0; index < _padArray.size(); ++index)
    {
        _padArray[index] = _server.RequestsFreePadState();
        _padArray[index]->SetPlayerIndex(static_cast<int32_t>(index) + 1);
    }

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      パッドステートの更新処理
 */
/* ------------------------------------------------------------------------- */
void XInputGamepadDelegate::UpdateState() noexcept
{
    for (DWORD index = 0; index < XUSER_MAX_COUNT; ++index)
    {
        XINPUT_STATE state = {};
        auto *pad = _padArray[static_cast<size_t>(index)];

        auto result = XInputGetState(index, &state);
        if (result == ERROR_SUCCESS)
        {
            // 新規に検出された場合は接続処理を行う
            if (!pad->IsEnabled())
            {
                PadDeviceInfo info;
                info.productName = "Generic XInput Gamepad";
                pad->RequestsActivate(PadType::XInput, PadPriority::High, info);
            }
            // ウィンドウがフォーカスされている場合のみ適用する
            else if (System::Window().IsFocused())
            {
                // ボタン
                pad->SetButton(ConvertButton(state.Gamepad.wButtons));

                // 左右スティック
                pad->SetLeftStick(Vector2(
                    ConvertStickValue(state.Gamepad.sThumbLX),
                    ConvertStickValue(state.Gamepad.sThumbLY))
                );
                pad->SetRightStick(Vector2(
                    ConvertStickValue(state.Gamepad.sThumbRX),
                    ConvertStickValue(state.Gamepad.sThumbRY))
                );

                // アナログD-Pad
                pad->ApplyLeftStickToDPad();

                // 左右トリガー
                if (state.Gamepad.bLeftTrigger >= kTriggerThreshold)
                {
                    pad->SetButton(PadButton::L2);
                }
                if (state.Gamepad.bRightTrigger >= kTriggerThreshold)
                {
                    pad->SetButton(PadButton::R2);
                }

                // 決定とキャンセル
                if (pad->IsPressing(PadButton::XInputA))
                {
                    pad->SetButton(PadButton::Decide);
                }
                if (pad->IsPressing(PadButton::XInputB))
                {
                    pad->SetButton(PadButton::Cancel);
                }
            }
        }
        else
        {
            // 有効なステートが検出できない場合は切断処理を行う
            if (pad->IsEnabled())
            {
                pad->RequestsDeactivate();
            }
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      XInputのボタンからMGLのボタンに変換
 *  \param[in]  xiButton    XInputのボタン
 *  \return     MGLのボタンフラグ
 */
/* ------------------------------------------------------------------------- */
PadButtonFlags XInputGamepadDelegate::ConvertButton(uint32_t xiButton) noexcept
{
    constexpr struct
    {
        uint32_t xiButton;
        PadButton mglButton;
    } kConvertTable[] =
    {
        {XINPUT_GAMEPAD_DPAD_UP,        PadButton::Up},             // 上
        {XINPUT_GAMEPAD_DPAD_DOWN,      PadButton::Down},           // 下
        {XINPUT_GAMEPAD_DPAD_LEFT,      PadButton::Left},           // 左
        {XINPUT_GAMEPAD_DPAD_RIGHT,     PadButton::Right},          // 右
        {XINPUT_GAMEPAD_START,          PadButton::XInputMenu},     // Menu/Start
        {XINPUT_GAMEPAD_BACK,           PadButton::XInputView},     // View/Back
        {XINPUT_GAMEPAD_LEFT_THUMB,     PadButton::L3},             // 左スティック押し込み
        {XINPUT_GAMEPAD_RIGHT_THUMB,    PadButton::R3},             // 右スティック押し込み
        {XINPUT_GAMEPAD_LEFT_SHOULDER,  PadButton::L1},             // LB
        {XINPUT_GAMEPAD_RIGHT_SHOULDER, PadButton::R1},             // RB
        {XINPUT_GAMEPAD_A,              PadButton::XInputA},        // A
        {XINPUT_GAMEPAD_B,              PadButton::XInputB},        // B
        {XINPUT_GAMEPAD_X,              PadButton::XInputX},        // X
        {XINPUT_GAMEPAD_Y,              PadButton::XInputY}         // Y
    };

    PadButtonFlags button;

    for (const auto &info : kConvertTable)
    {
        if ((xiButton & info.xiButton) != 0)
        {
            button |= static_cast<PadButtonFlags>(info.mglButton);
        }
    }

    return button;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スティックの値を変換
 *  \param[in]  value   スティックの値
 *  \return     0から1の値
 */
/* ------------------------------------------------------------------------- */
float XInputGamepadDelegate::ConvertStickValue(int16_t value) noexcept
{
    if (value > kStickDeadzonePlus)
    {
        return static_cast<float>(value) / 32767;
    }
    else if (value < kStickDeadzoneMinus)
    {
        return static_cast<float>(value) / 32768;
    }

    return 0.0f;
}
}   // namespace MGL::Input

#endif  // MGL_GAMEPAD_DELEGATE_ENABLE_XINPUT
// vim: et ts=4 sw=4 sts=4
