// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_gamepad_delegate_iokit_hid.mm
 *  \brief      MGL IOKitによるHIDゲームパッドデリゲート
 *  \date       Since: January 9, 2021. 17:35:04 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/gamepad/mgl_gamepad_delegate_iokit_hid.h>
#if defined(MGL_GAMEPAD_DELEGATE_ENABLE_IOKIT_HID)

#import <Foundation/Foundation.h>
#import <GameController/GameController.h>

#include <mgl/input/gamepad/iokit_hid_driver/mgl_iokit_hid_generic_driver.h>
#include <mgl/input/gamepad/mgl_gamepad_server.h>
#include <mgl/stl/mgl_stl_string.h>

// GCフレームワークのゲームパッドと共存させるためのフラグ。
// macOS 26.2現在、APIの不具合と思われる挙動により機能しなくなったため、いつでも切り替えられるようにしておく。
//#define MGL_ENABLE_COEXIST_GC_GAMEPAD

namespace MGL::Input
{
namespace
{
constexpr IOKitHID::MakeGamepadDriverFunction kMakeDriverFunctionArray[] =
{
    IOKitHID::MakeGamepadDriver<IOKitHID::GenericGamepadDriver>    //!< 標準ドライバ
};
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
IOKitHIDGamepadDelegate::IOKitHIDGamepadDelegate(GamepadServer &server) noexcept
    : GamepadDelegate(server)
    , _hidManager(nullptr)
    , _priority(PadPriority::Low)
    , _hidStateList()
    , _mutex()
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デストラクタ
 */
/* ------------------------------------------------------------------------- */
IOKitHIDGamepadDelegate::~IOKitHIDGamepadDelegate() noexcept
{
    // HIDマネージャをクローズ
    if (_hidManager != nullptr)
    {
        IOHIDManagerClose(_hidManager, kIOHIDManagerOptionNone);
        CFRelease(_hidManager);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool IOKitHIDGamepadDelegate::Initialize() noexcept
{
    // HIDマネージャを生成
    _hidManager = IOHIDManagerCreate(kCFAllocatorDefault, kIOHIDManagerOptionNone);
    if (_hidManager == nullptr)
    {
        return false;
    }
    
    // このクラスで扱うデバイスのマッチングテーブルを設定
    NSArray *deviceMatchingArray = @[
        @{@kIOHIDDeviceUsagePageKey: @(kHIDPage_GenericDesktop), @kIOHIDDeviceUsageKey: @(kHIDUsage_GD_GamePad)},
        @{@kIOHIDDeviceUsagePageKey: @(kHIDPage_GenericDesktop), @kIOHIDDeviceUsageKey: @(kHIDUsage_GD_Joystick)},
    ];
    IOHIDManagerSetDeviceMatchingMultiple(_hidManager, (__bridge CFArrayRef)deviceMatchingArray);
    
    // 接続・切断コールバック関数を設定
    IOHIDManagerRegisterDeviceMatchingCallback(_hidManager, CallbackDeviceMatching, this);
    IOHIDManagerRegisterDeviceRemovalCallback(_hidManager, CallbackDeviceRemoval, this);
    
    // HIDマネージャを動作させるスレッドを指定
    IOHIDManagerScheduleWithRunLoop(_hidManager, CFRunLoopGetCurrent(), kCFRunLoopCommonModes);
    
    // HIDマネージャをオープン
    auto result = IOHIDManagerOpen(_hidManager, kIOHIDOptionsTypeNone);
    if (result != kIOReturnSuccess)
    {
        return false;
    }
    
#if defined(MGL_ENABLE_COEXIST_GC_GAMEPAD)
    // macOS 11.0移行はMFiゲームパッドとの併用が可能なのでプライオリティをHighに設定
    if ([NSProcessInfo.processInfo isOperatingSystemAtLeastVersion: {11, 0, 0}])
    {
        _priority = PadPriority::High;
    }
#endif

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      更新処理
 */
/* ------------------------------------------------------------------------- */
void IOKitHIDGamepadDelegate::UpdateState() noexcept
{
    std::lock_guard<std::mutex> lock(_mutex);
    
    for (auto &state : _hidStateList)
    {
        state->driver->UpdateState(*(state->state), state->device);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デバイス接続時のコールバック関数
 *  \param[in]  inContext           このクラスのアドレス
 *  \param[in]  inResult            マッチングの結果
 *  \param[in]  inSender            この通知を送ったHIDマネージャ
 *  \param[in]  inIOHIDDeviceRef    新たに検出されたデバイス
 */
/* ------------------------------------------------------------------------- */
void IOKitHIDGamepadDelegate::CallbackDeviceMatching(void *inContext, IOReturn inResult, [[maybe_unused]] void *inSender, IOHIDDeviceRef inIOHIDDeviceRef) noexcept
{
    // 認識に失敗しているデバイスは処理しない
    if ((inResult != kIOReturnSuccess) || (inIOHIDDeviceRef == nullptr))
    {
        return;
    }

    auto *thisPtr = static_cast<IOKitHIDGamepadDelegate *>(inContext);

#if defined(MGL_ENABLE_COEXIST_GC_GAMEPAD)
    // GameControllerフレームワークがサポートしているデバイスはこちらでは扱わない（macOS 11以降）
    if (@available(macOS 11.0, *))
    {
        if ([NSProcessInfo.processInfo isOperatingSystemAtLeastVersion: {11, 0, 0}])
        {
            if ([GCController supportsHIDDevice:inIOHIDDeviceRef])
            {
                return;
            }
        }
    }
#endif

    // デバイスを追加
    thisPtr->AddDevice(inIOHIDDeviceRef);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デバイス切断時のコールバック関数
 *  \param[in]  inContext           このクラスのアドレス
 *  \param[in]  inResult            マッチングの結果
 *  \param[in]  inSender            この通知を送ったHIDマネージャ
 *  \param[in]  inIOHIDDeviceRef    切断が検出されたデバイス
 */
/* ------------------------------------------------------------------------- */
void IOKitHIDGamepadDelegate::CallbackDeviceRemoval(void *inContext, IOReturn inResult, [[maybe_unused]] void *inSender, IOHIDDeviceRef inIOHIDDeviceRef) noexcept
{
    // 認識に失敗しているデバイスは処理しない
    if ((inResult != kIOReturnSuccess) || (inIOHIDDeviceRef == nullptr))
    {
        return;
    }
    
    // デバイスを削除
    auto *thisPtr = static_cast<IOKitHIDGamepadDelegate *>(inContext);
    thisPtr->RemoveDevice(inIOHIDDeviceRef);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デバイスの追加
 *  \param[in]  device  追加するデバイス
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool IOKitHIDGamepadDelegate::AddDevice(IOHIDDeviceRef device) noexcept
{
    auto state = STL::make_unique<IOKitHIDState>();
    
    // デバイスを設定
    state->device = device;

    // ベンダIDとプロダクトIDを取得して対応するドライバを生成
    int32_t vendorID = 0;
    int32_t productID = 0;
    GetDevicePropertyValue(vendorID, device, CFSTR(kIOHIDVendorIDKey));
    GetDevicePropertyValue(productID, device, CFSTR(kIOHIDProductIDKey));
    for (auto makeDriver : kMakeDriverFunctionArray)
    {
        state->driver = makeDriver(vendorID, productID);
        if (state->driver != nullptr)
        {
            if (!state->driver->Initialize(device))
            {
                return false;
            }
            break;
        }
    }
    if (state->driver == nullptr)
    {
        return false;
    }

    // ゲームパッドサーバから未使用のパッドステートを要求
    state->state = _server.RequestsFreePadState();
    if (state->state == nullptr)
    {
        return false;
    }
    
    // デバイス情報を取得
    PadDeviceInfo deviceInfo;
    deviceInfo.vendorID = static_cast<uint16_t>(vendorID);
    deviceInfo.productID = static_cast<uint16_t>(productID);
    GetDevicePropertyString(deviceInfo.vendorName, device, CFSTR(kIOHIDManufacturerKey));
    GetDevicePropertyString(deviceInfo.productName, device, CFSTR(kIOHIDProductKey));

    // アクティベートを要求
    state->state->RequestsActivate(PadType::GenericHID, _priority, deviceInfo);
    state->state->SetDecideButton(_server.GetDecideButton(PadType::GenericHID, PadButton::Button01));
    state->state->SetCancelButton(_server.GetCancelButton(PadType::GenericHID, PadButton::Button02));

    // ステートをリストに追加
    std::lock_guard<std::mutex> lock(_mutex);
    _hidStateList.push_back(std::move(state));
    
    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デバイスの削除
 *  \param[in]  device  削除するデバイス
 */
/* ------------------------------------------------------------------------- */
void IOKitHIDGamepadDelegate::RemoveDevice(IOHIDDeviceRef device) noexcept
{
    std::lock_guard<std::mutex> lock(_mutex);

    for (auto it = _hidStateList.begin(); it != _hidStateList.end(); ++it)
    {
        if ((*it)->device == device)
        {
            auto *state = (*it)->state;
            state->RequestsDeactivate();
            _server.ReturnPadState(state);
            _hidStateList.erase(it);
            break;
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デバイスから数値のプロパティを取得
 *  \param[out] value   取得した数値の格納先．失敗した場合は書き換えない
 *  \param[in]  device  プロパティを取得するデバイス
 *  \param[in]  key     取得するプロパティのキー
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool IOKitHIDGamepadDelegate::GetDevicePropertyValue(int32_t &value, const IOHIDDeviceRef device, const CFStringRef key) const noexcept
{
    // デイバスからプロパティを取得
    CFTypeRef typeRef = IOHIDDeviceGetProperty(device, key);
    if (typeRef == nullptr)
    {
        return false;
    }

    // 取得した値が数値でなければ失敗
    if (CFGetTypeID(typeRef) != CFNumberGetTypeID())
    {
        return false;
    }

    // 数値を取得して返す
    return CFNumberGetValue((CFNumberRef)typeRef, kCFNumberSInt32Type, &value);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デバイスから文字列のプロパティを取得
 *  \param[out] string  取得した文字列の格納先．失敗した場合は書き換えない
 *  \param[in]  device  プロパティを取得するデバイス
 *  \param[in]  key     取得するプロパティのキー
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool IOKitHIDGamepadDelegate::GetDevicePropertyString(STL::string &string, const IOHIDDeviceRef device, const CFStringRef key) const noexcept
{
    // デイバスからプロパティを取得
    CFTypeRef typeRef = IOHIDDeviceGetProperty(device, key);
    if (typeRef == nullptr)
    {
        return false;
    }

    // 取得した値が数値でなければ失敗
    if (CFGetTypeID(typeRef) != CFStringGetTypeID())
    {
        return false;
    }

    // 文字列を格納
    NSString *nsString = (__bridge NSString *)typeRef;
    string = nsString.UTF8String;

    return true;
}

}   // namespace MGL::Input
#endif  // MGL_GAMEPAD_DELEGATE_ENABLE_IOKIT_HID
// vim: et ts=4 sw=4 sts=4
