// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_gamepad_delegate_directinput.cc
 *  \brief      MGL DirectInput用デリゲート
 *  \date       Since: April 1, 2021. 19:10:35 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/gamepad/mgl_gamepad_delegate_directinput.h>
#if defined(MGL_GAMEPAD_DELEGATE_ENABLE_DIRECTINPUT)

#include <mgl/input/gamepad/mgl_gamepad_server.h>
#include <mgl/system/mgl_system_window.h>
#include <mgl/system/mgl_system_debug_macro.h>
#include <mgl/platform/win32/mgl_win32_window.h>
#include <mgl/text/mgl_text_converter.h>

#pragma comment(lib, "dinput8.lib")
#pragma comment(lib, "dxguid.lib")

namespace MGL::Input
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
DirectInputGamepadDelegate::DirectInputGamepadDelegate(GamepadServer &server) noexcept
    : GamepadDelegate(server)
    , _eventDeviceArrival(Event::NotifyType::DeviceArrival, OnEventDeviceArrival, this)
    , _eventDeviceRemove(Event::NotifyType::DeviceRemove, OnEventDeviceRemove, this)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool DirectInputGamepadDelegate::Initialize() noexcept
{
    if (auto hr = DirectInput8Create(GetModuleHandle(nullptr), DIRECTINPUT_VERSION, IID_IDirectInput8, reinterpret_cast<LPVOID *>(& _dinput.p), nullptr); FAILED(hr))
    {
        MGL_WARNING("[MGL DirectInput] Failed to initialize DirectInput8.");
        return false;
    }

    // ゲームコントローラを列挙
    auto hr = _dinput->EnumDevices(
        DI8DEVCLASS_GAMECTRL,
        [](const DIDEVICEINSTANCE *deviceInstance, void *ref)
        {
            auto *thisPtr = static_cast<DirectInputGamepadDelegate *>(ref);

            // デバイスを登録
            if (!thisPtr->AddDevice(deviceInstance))
            {
                return DIENUM_STOP;
            }

            return DIENUM_CONTINUE;
        },
        this,
        DIEDFL_ATTACHEDONLY
    );

    return SUCCEEDED(hr);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      更新処理
 */
/* ------------------------------------------------------------------------- */
void DirectInputGamepadDelegate::UpdateState() noexcept
{
    const bool isFocused = System::Window().IsFocused();

    for (auto &pad : _padArray)
    {
        pad.state->PreUpdate();

        if (isFocused)
        {
            bool isActive = false;
            if (SUCCEEDED(pad.device->Poll()))
            {
                isActive = true;
            }
            else
            {
                if (SUCCEEDED(pad.device->Acquire()))
                {
                    if (SUCCEEDED(pad.device->Poll()))
                    {
                        isActive = true;
                    }
                }
            }

            if (isActive)
            {
                DIJOYSTATE2 diState;
                if (SUCCEEDED(pad.device->GetDeviceState(sizeof(DIJOYSTATE2), &diState)))
                {
                    ApplyState(pad.state, diState);
                }
            }
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デバイスの追加
 *  \param[in]  deviceInstance  追加するデバイスのインスタンス
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool DirectInputGamepadDelegate::AddDevice(const DIDEVICEINSTANCE *deviceInstance) noexcept
{
    DirectInputGamepad pad = {};

    // デバイスの生成
    if (auto hr = _dinput->CreateDevice(deviceInstance->guidInstance, &pad.device.p, nullptr); FAILED(hr))
    {
        return false;
    }

    // 空きステートを取得
    pad.state = _server.RequestsFreePadState();
    if (pad.state == nullptr)
    {
        return false;
    }

    // GUIDを保持
    pad.guid = deviceInstance->guidInstance;

    // デバイスの初期設定
    auto& window = Win32::Window::GetInstance();
    pad.device->SetDataFormat(&c_dfDIJoystick2);
    pad.device->SetCooperativeLevel(window.GetWindowHandler(), DISCL_FOREGROUND | DISCL_EXCLUSIVE);

    // 使用準備
    pad.device->Acquire();
    PadDeviceInfo info;
    Text::ToUTF8(info.productName, deviceInstance->tszProductName, MAX_PATH, Text::Encoding::UTF16LE);
    pad.state->RequestsActivate(PadType::DirectInput, PadPriority::Low, info);

    // 追加
    _padArray.push_back(pad);

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ステートを適用
 *  \param[out] state       ステートの適用先
 *  \param[in]  diState     DirectInputのステート
 */
/* ------------------------------------------------------------------------- */
void DirectInputGamepadDelegate::ApplyState(PadState *state, const DIJOYSTATE2 &diState) noexcept
{
    // ボタン
    for (size_t i = 0; i < kGamepadButtonMax; ++i)
    {
        if ((diState.rgbButtons[i] & 0x80u) != 0)
        {
            auto button = PadButton(static_cast<uint8_t>(PadButton::Button01) + static_cast<uint8_t>(i));
            state->SetButton(button);
        }
    }

    // POVに有効値が入っていなければ軸の状態から上下左右を取得
    const uint32_t pov = diState.rgdwPOV[0];
    if (pov > 36000)
    {
        // 左右
        if (diState.lX >= 0x9FFF)
        {
            state->SetButton(PadButton::Right);
        }
        else if (diState.lX <= 0x5FFF)
        {
            state->SetButton(PadButton::Left);
        }

        // 上下
        if (diState.lY >= 0x9FFF)
        {
            state->SetButton(PadButton::Down);
        }
        else if (diState.lY <= 0x5FFF)
        {
            state->SetButton(PadButton::Up);
        }
    }
    // POVに有効値があればそれをもとに上下左右を取得
    else
    {
        // 上
        if ((pov <= 6000) || (pov >= 30000))
        {
            state->SetButton(PadButton::Up);
        }
        // 右
        if ((pov >= 3000) && (pov <= 15000))
        {
            state->SetButton(PadButton::Right);
        }
        // 下
        if ((pov >= 12000) && (pov <= 24000))
        {
            state->SetButton(PadButton::Down);
        }
        // 左
        if ((pov >= 21000) && (pov <= 33000))
        {
            state->SetButton(PadButton::Left);
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デバイス接続イベントの処理
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       未使用（接続されたデバイスの情報が入っている）
 */
/* ------------------------------------------------------------------------- */
void DirectInputGamepadDelegate::OnEventDeviceArrival(void *callbackArg, [[maybe_unused]] void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<DirectInputGamepadDelegate *>(callbackArg);

    if (thisPtr->_dinput == nullptr)
    {
        return;
    }

    // ゲームコントローラを列挙
    thisPtr->_dinput->EnumDevices(
        DI8DEVCLASS_GAMECTRL,
        [](const DIDEVICEINSTANCE *deviceInstance, void *ref)
        {
            auto *thisPtr = static_cast<DirectInputGamepadDelegate *>(ref);

            // 既に登録されているデバイスは飛ばす
            for (const auto &pad : thisPtr->_padArray)
            {
                if (IsEqualGUID(pad.guid, deviceInstance->guidInstance))
                {
                    return DIENUM_CONTINUE;
                }
            }

            // 新たに接続されたデバイスを登録
            if (!thisPtr->AddDevice(deviceInstance))
            {
                return DIENUM_STOP;
            }

            return DIENUM_CONTINUE;
        },
        thisPtr,
        DIEDFL_ATTACHEDONLY
    );
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デバイス接続イベントの処理
 *  \param[in]  callbackArg     このクラスのアドレス
 *  \param[in]  notifyArg       未使用（切断されたデバイスの情報が入っている）
 */
/* ------------------------------------------------------------------------- */
void DirectInputGamepadDelegate::OnEventDeviceRemove(void *callbackArg, [[maybe_unused]] void *notifyArg) noexcept
{
    auto *thisPtr = static_cast<DirectInputGamepadDelegate *>(callbackArg);

    if (thisPtr->_dinput == nullptr)
    {
        return;
    }

    // 無効になっているパッドを検出して，切断・削除を行う
    for (auto it = thisPtr->_padArray.begin(); it != thisPtr->_padArray.end();)
    {
        if (FAILED(it->device->Poll()))
        {
            if (FAILED((it->device->Acquire())))
            {
                it->state->RequestsDeactivate();          
                thisPtr->_server.ReturnPadState(it->state);
            }
        }

        if (it->state->IsEnabled())
        {
            ++it;
        }
        else
        {
            it = thisPtr->_padArray.erase(it);
        }
    }
}

}   // namespace MGL::Input

#endif  // MGL_GAMEPAD_DELEGATE_ENABLE_DIRECTINPUT

// vim: et ts=4 sw=4 sts=4
