// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_gamepad_delegate_apple_gc.mm
 *  \brief      MGL GameControllerフレームワークデリゲート
 *  \date       Since: September 24, 2022. 16:33:18 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/gamepad/mgl_gamepad_delegate_apple_gc.h>
#if defined(MGL_GAMEPAD_DELEGATE_ENABLE_APPLE_GC)

#include <mgl/input/gamepad/mgl_gamepad_server.h>

namespace MGL::Input
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
AppleGCGamepadDelegate::AppleGCGamepadDelegate(GamepadServer &server) noexcept
    : GamepadDelegate(server)
    , _connectObserver(nil)
    , _disconnectObserver(nil)
    , _gamepads()
    , _availableFlags()
    , _mutex()
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デストラクタ
 */
/* ------------------------------------------------------------------------- */
AppleGCGamepadDelegate::~AppleGCGamepadDelegate() noexcept
{
    auto defaultCenter = NSNotificationCenter.defaultCenter;

    if (_connectObserver != nil)
    {
        [defaultCenter removeObserver:_connectObserver];
    }
    if (_disconnectObserver != nil)
    {
        [defaultCenter removeObserver:_disconnectObserver];
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool AppleGCGamepadDelegate::Initialize() noexcept
{
    // 利用可能なパッドステートの数をチェック
    if (_server.GetFreePadStateCount() < kGamepadMax)
    {
        return false;
    }

    // サーバから空いているパッドステートをを取得
    for (size_t i = 0; i < kGamepadMax; i++)
    {
        _gamepads[i].state = _server.RequestsFreePadState();
    }
    
    
    auto defaultCenter = NSNotificationCenter.defaultCenter;
    
    // 接続コールバックを登録
    _connectObserver = [defaultCenter addObserverForName:GCControllerDidConnectNotification
                                                  object:nil
                                                   queue:nil
                                              usingBlock:^(NSNotification *note)
    {
        OnConnect(static_cast<GCController *>(note.object));
    }];

    // 切断コールバックを登録
    _disconnectObserver = [defaultCenter addObserverForName:GCControllerDidDisconnectNotification
                                                     object:nil
                                                      queue:nil
                                                 usingBlock:^(NSNotification *note)
    {
        OnDisconnect(static_cast<GCController *>(note.object));
    }];

    // 利用可能な機能のフラグを初期化
    InitializeAvailableFlags();

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      更新処理
 */
/* ------------------------------------------------------------------------- */
void AppleGCGamepadDelegate::UpdateState() noexcept
{
    std::lock_guard<std::mutex> lock(_mutex);

    for (auto &gamepad : _gamepads)
    {
        if (gamepad.state->IsEnabled())
        {
            switch (gamepad.state->GetType())
            {
                // MFi拡張
                case PadType::MFiExtended:
                    UpdateMFiExtendedPadState(*(gamepad.state), gamepad.controller, gamepad.isRequestedPause);
                    if (!IsAvailable(AvailableFlag::MFiReadMenuButton))
                    {
                        gamepad.isRequestedPause = false;
                    }
                    break;

                // AppleRemote/SiriRemote
                case PadType::MFiMicro:
                    UpdateMFiMicroPadState(*(gamepad.state), gamepad.controller, gamepad.isRequestedPause);
                    if (!IsAvailable(AvailableFlag::MFiReadMenuButton))
                    {
                        gamepad.isRequestedPause = false;
                    }
                    break;

                // XboxOne
                case PadType::XboxOne:
                    UpdateXboxOnePadState(*(gamepad.state), gamepad.controller);
                    break;

                // DualShock4
                case PadType::DualShock4:
                    UpdateDualShock4PadState(*(gamepad.state), gamepad.controller);
                    break;

                // DualSense
                case PadType::DualSense:
                    UpdateDualSensePadState(*(gamepad.state), gamepad.controller);
                    break;

                default:
                    break;
            }
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief  各機能が有効かどうかのフラグを初期化
 */
/* ------------------------------------------------------------------------- */
void AppleGCGamepadDelegate::InitializeAvailableFlags() noexcept
{
    // 機能の有効フラグと有効なOSのバージョンを関連付けるためのテーブル
    struct AvailableParameter
    {
        AvailableFlag flag;                  // 関連付ける有効フラグ
        NSOperatingSystemVersion version;   // 利用可能なOSのバージョン
    };
    const AvailableParameter availableParameters[] =
    {
#if defined(MGL_TARGET_MACOS)
        // macOS
        {AvailableFlag::MFiThumbstickButton,    {10, 14, 1}},   // MFiゲームパッドのサムスティックボタン（L3/R3）の読み取り
        {AvailableFlag::MFiReadMenuButton,      {10, 15, 0}},   // MFiゲームパッドのメニューボタンの直接読み取り
        {AvailableFlag::MFiOptionsButton,       {10, 15, 0}},   // MFiゲームパッドのOptionsボタンの読み取り
        {AvailableFlag::XboxOneGamepad,         {11,  0, 0}},   // XboxOneコントローラの読み取り
        {AvailableFlag::DualShock4Gamepad,      {11,  0, 0}},   // DualShock4の読み取り
        {AvailableFlag::DualSenseGamepad,       {11,  3, 0}},   // DualSenseの読み取り
        {AvailableFlag::XboxShareButton,        {12,  0, 0}},   // XboxOneのシェアボタン
#elif defined(MGL_TARGET_IOS)
        // iOS
        {AvailableFlag::MFiThumbstickButton,    {12,  1, 0}},   // MFiゲームパッドのサムスティックボタン（L3/R3）の読み取り
        {AvailableFlag::MFiReadMenuButton,      {13,  0, 0}},   // MFiゲームパッドのメニューボタンの直接読み取り
        {AvailableFlag::MFiOptionsButton,       {13,  0, 0}},   // MFiゲームパッドのOptionsボタンの読み取り
        {AvailableFlag::XboxOneGamepad,         {14,  0, 0}},   // XboxOneコントローラの読み取り
        {AvailableFlag::DualShock4Gamepad,      {14,  0, 0}},   // DualShock4の読み取り
        {AvailableFlag::DualSenseGamepad,       {14,  5, 0}},   // DualSenseの読み取り
        {AvailableFlag::XboxShareButton,        {15,  0, 0}},   // XboxOneのシェアボタン
#elif defined(MTL_TARGET_TVOS)
        // tvOS
        {AvailableFlag::MFiThumbstickButton,    {12,  1, 0}},   // MFiゲームパッドのサムスティックボタン（L3/R3）の読み取り
        {AvailableFlag::MFiReadMenuButton,      {13,  0, 0}},   // MFiゲームパッドのメニューボタンの直接読み取り
        {AvailableFlag::MFiOptionsButton,       {13,  0, 0}},   // MFiゲームパッドのOptionsボタンの読み取り
        {AvailableFlag::XboxOneGamepad,         {14,  0, 0}},   // XboxOneコントローラの読み取り
        {AvailableFlag::DualShock4Gamepad,      {14,  0, 0}},   // DualShock4の読み取り
        {AvailableFlag::DualSenseGamepad,       {14,  5, 0}},   // DualSenseの読み取り
        {AvailableFlag::XboxShareButton,        {15,  0, 0}},   // XboxOneのシェアボタン
#endif
    };

    _availableFlags.fill(false);
    
    for (auto &parameter : availableParameters)
    {
        if ([NSProcessInfo.processInfo isOperatingSystemAtLeastVersion:parameter.version])
        {
            auto index = static_cast<size_t>(parameter.flag);
            _availableFlags[index] = true;
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      パッドの種類を取得
 *  \param[in]  controller  コントローラクラス
 *  \return     引数に対応するパッドの種類
 */
/* ------------------------------------------------------------------------- */
PadType AppleGCGamepadDelegate::GetPadType(GCController *controller) const noexcept
{
    // Xbox One
    if (@available(macOS 11.0, iOS 14.0, tvOS 14.0, *))
    {
        if (IsAvailable(AvailableFlag::XboxOneGamepad))
        {
            if ([controller.extendedGamepad isKindOfClass:[GCXboxGamepad class]])
            {
                return PadType::XboxOne;
            }
        }
    }

    // DualShock4
    if (@available(macOS 11.0, iOS 14.0, tvOS 14.0, *))
    {
        if (IsAvailable(AvailableFlag::DualShock4Gamepad))
        {
            if ([controller.extendedGamepad isKindOfClass:[GCDualShockGamepad class]])
            {
                return PadType::DualShock4;
            }
        }
    }

    // DualSense
    if (@available(macOS 11.3, iOS 14.5, tvOS 14.5, *))
    {
        if (IsAvailable(AvailableFlag::DualSenseGamepad))
        {
            if ([controller.extendedGamepad isKindOfClass:[GCDualSenseGamepad class]])
            {
                return PadType::DualSense;
            }
        }
    }

    // 上記以外はMFiExtendedかMFiMicro
    if (controller.extendedGamepad != nil)
    {
        return PadType::MFiExtended;
    }
    else if (controller.microGamepad != nil)
    {
        return PadType::MFiMicro;
    }
    
    return PadType::Disable;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      接続コールバック
 *  \param[in]  controller  接続されたコントローラ
 */
/* ------------------------------------------------------------------------- */
void AppleGCGamepadDelegate::OnConnect(GCController *controller) noexcept
{
    // パッドの種類を判別
    PadType type = GetPadType(controller);
    if (type == PadType::Disable)
    {
        return;     // 判別できないものは追加しない
    }
    
    std::lock_guard<std::mutex> lock(_mutex);

    for (auto &gamepad : _gamepads)
    {
        if (gamepad.state->IsEnabled())
        {
            continue;
        }
        
        PadDeviceInfo deviceInfo;
        deviceInfo.vendorName = [controller.vendorName UTF8String];

        // タイプ毎の設定
        switch (type)
        {
            // XboxOne
            case PadType::XboxOne:
                deviceInfo.productName = "Xbox Controller";
                gamepad.state->SetDecideButton(_server.GetDecideButton(type, PadButton::XInputA));
                gamepad.state->SetCancelButton(_server.GetCancelButton(type, PadButton::XInputB));
                break;
                
            // DualShock4
            case PadType::DualShock4:
                deviceInfo.productName = "DualShock 4";
                gamepad.state->SetDecideButton(_server.GetDecideButton(type, PadButton::DSCross));
                gamepad.state->SetCancelButton(_server.GetCancelButton(type, PadButton::DSCircle));
                break;
                
            // DualSense
            case PadType::DualSense:
                deviceInfo.productName = "DualSense";
                gamepad.state->SetDecideButton(_server.GetDecideButton(type, PadButton::DSCross));
                gamepad.state->SetCancelButton(_server.GetCancelButton(type, PadButton::DSCircle));
                break;

            // MFi拡張
            case PadType::MFiExtended:
                deviceInfo.productName = "Generic MFi Gamepad";
                gamepad.state->SetDecideButton(_server.GetDecideButton(type, PadButton::MFiA));
                gamepad.state->SetCancelButton(_server.GetCancelButton(type, PadButton::MFiB));
                break;
                
            default:
                return;
        }

        // メニューボタンの直接読み取りに対応していない場合，コールバック関数を登録する
        if (!IsAvailable(AvailableFlag::MFiReadMenuButton))
        {
            controller.controllerPausedHandler = ^(GCController *controller)
            {
                for (auto &gamepad : _gamepads)
                {
                    if ([gamepad.controller isEqual:controller])
                    {
                        gamepad.isRequestedPause = true;
                    }
                }
            };
        }

        gamepad.controller = controller;
        gamepad.state->RequestsActivate(type, PadPriority::High, deviceInfo);
        
        break;
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      切断コールバック
 *  \param[in]  controller  切断されたコントローラ
 */
/* ------------------------------------------------------------------------- */
void AppleGCGamepadDelegate::OnDisconnect(GCController *controller) noexcept
{
    std::lock_guard<std::mutex> lock(_mutex);

    for (auto &gamepad : _gamepads)
    {
        if ([controller isEqual:gamepad.controller])
        {
            gamepad.state->RequestsDeactivate();
            gamepad.controller = nil;
            break;
        }
    }

    controller.controllerPausedHandler = nil;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      MFi拡張ゲームパッドの更新処理
 *  \param[out] state               更新するパッドステート
 *  \param[in]  controller          コントローラクラス
 *  \param[in]  isRequestedPause    ポーズ要求フラグ（古いバージョンでのみ使用）
 */
/* ------------------------------------------------------------------------- */
void AppleGCGamepadDelegate::UpdateMFiExtendedPadState(PadState &state, GCController *controller, bool isRequestedPause) noexcept
{
    auto *gamepad = controller.extendedGamepad;
    if (gamepad == nil)
    {
        return;
    }
    
    // D-Padの上下左右
    if (gamepad.dpad.up.pressed)
    {
        state.SetButton(PadButton::Up);
    }
    if (gamepad.dpad.down.pressed)
    {
        state.SetButton(PadButton::Down);
    }
    if (gamepad.dpad.left.pressed)
    {
        state.SetButton(PadButton::Left);
    }
    if (gamepad.dpad.right.pressed)
    {
        state.SetButton(PadButton::Right);
    }

    // ABXY
    if (gamepad.buttonA.pressed)
    {
        state.SetButton(PadButton::MFiA);
    }
    if (gamepad.buttonB.pressed)
    {
        state.SetButton(PadButton::MFiB);
    }
    if (gamepad.buttonX.pressed)
    {
        state.SetButton(PadButton::MFiX);
    }
    if (gamepad.buttonY.pressed)
    {
        state.SetButton(PadButton::MFiY);
    }

    // L1/R1
    if (gamepad.leftShoulder.pressed)
    {
        state.SetButton(PadButton::L1);
    }
    if (gamepad.rightShoulder.pressed)
    {
        state.SetButton(PadButton::R1);
    }
    
    // L2/R2
    if (gamepad.leftTrigger.pressed)
    {
        state.SetButton(PadButton::L2);
    }
    if (gamepad.rightTrigger.pressed)
    {
        state.SetButton(PadButton::R2);
    }
    
    // L3/R3
    if (@available(macOS 10.14.1, iOS 12.1, tvOS 12.1, *))
    { 
        if (IsAvailable(AvailableFlag::MFiThumbstickButton))
        {
            if (gamepad.leftThumbstickButton.pressed)
            {
                state.SetButton(PadButton::L3);
            }
            if (gamepad.rightThumbstickButton.pressed)
            {
                state.SetButton(PadButton::R3);
            }
        }
    }
    
    // 左右スティック
    state.GetLeftStick() = MGL::Vector2(gamepad.leftThumbstick.xAxis.value, gamepad.leftThumbstick.yAxis.value);
    state.GetRightStick() = MGL::Vector2(gamepad.rightThumbstick.xAxis.value, gamepad.rightThumbstick.yAxis.value);
    state.ApplyLeftStickToDPad();

    // メニューボタン
    if (IsAvailable(AvailableFlag::MFiReadMenuButton))
    {
        if (@available(macOS 10.15, iOS 13.0, tvOS 13.0, *))
        {
            if (gamepad.buttonMenu.pressed)
            {
                state.SetButton(PadButton::PrimaryMenu);
            }
        }
    }
    else if (isRequestedPause)
    {
        state.SetButton(PadButton::PrimaryMenu);
    }
    
    // Optionsボタン
    if (IsAvailable(AvailableFlag::MFiOptionsButton))
    {
        if (@available(macOS 10.15, iOS 13.0, tvOS 13.0, *))
        {
            if (gamepad.buttonOptions.pressed)
            {
                state.SetButton(PadButton::SecondaryMenu);
            }
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      AppleRemote/SiriRemoteの更新処理
 *  \param[out] state               更新するパッドステート
 *  \param[in]  controller          コントローラクラス
 *  \param[in]  isRequestedPause    ポーズ要求フラグ（古いバージョンでのみ使用）
 */
/* ------------------------------------------------------------------------- */
void AppleGCGamepadDelegate::UpdateMFiMicroPadState(PadState &state, GCController *controller, bool isRequestedPause) noexcept
{
    auto *gamepad = controller.microGamepad;
    if (gamepad == nil)
    {
        return;
    }
        
    // 左スティック（タッチサーフェス）
    if (state.IsPortraitMode())
    {
        // 縦持ち
        state.SetLeftStick(Vector2(gamepad.dpad.xAxis.value, gamepad.dpad.yAxis.value));
    }
    else
    {
        // 横持ち
        state.SetLeftStick(Vector2(gamepad.dpad.yAxis.value * -1.0f, gamepad.dpad.xAxis.value));
    }
    
    // AボタンとXボタン
    if (gamepad.buttonA.pressed)
    {
        state.SetButton(PadButton::MFiA);
    }
    if (gamepad.buttonX.pressed)
    {
        state.SetButton(PadButton::MFiX);
    }

    // タッチサーフェスのタッチ
    if ((gamepad.dpad.xAxis.value != 0.0f) && (gamepad.dpad.yAxis.value != 0.0f))
    {
        state.SetButton(PadButton::MFiTouchSurface);
    }

    // メニューボタン
    if (IsAvailable(AvailableFlag::MFiReadMenuButton))
    {
        if (@available(macOS 10.15, iOS 13.0, tvOS 13.0, *))
        {
            if (gamepad.buttonMenu.pressed)
            {
                state.SetButton(PadButton::PrimaryMenu);
            }
        }
    }
    else if (isRequestedPause)
    {
        state.SetButton(PadButton::PrimaryMenu);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      XboxOneゲームパッドの更新処理
 *  \param[out] state       更新するパッドステート
 *  \param[in]  controller  コントローラクラス
 */
/* ------------------------------------------------------------------------- */
void AppleGCGamepadDelegate::UpdateXboxOnePadState(PadState &state, GCController *controller) noexcept
{
    if (@available(macOS 11.0, iOS 14.0, tvOS 14.0, *))
    {
        // GCExtendedGamepadをGCXBoxGamepadにキャスト
        GCXboxGamepad *gamepad = nil;
        if ([controller.extendedGamepad isKindOfClass:[GCXboxGamepad class]])
        {
            gamepad = static_cast<GCXboxGamepad *>(controller.extendedGamepad);
        }
        else
        {
            return;
        }
        
        // D-Padの上下左右
        if (gamepad.dpad.up.pressed)
        {
            state.SetButton(PadButton::Up);
        }
        if (gamepad.dpad.down.pressed)
        {
            state.SetButton(PadButton::Down);
        }
        if (gamepad.dpad.left.pressed)
        {
            state.SetButton(PadButton::Left);
        }
        if (gamepad.dpad.right.pressed)
        {
            state.SetButton(PadButton::Right);
        }

        // ABXY
        if (gamepad.buttonA.pressed)
        {
            state.SetButton(PadButton::XInputA);
        }
        if (gamepad.buttonB.pressed)
        {
            state.SetButton(PadButton::XInputB);
        }
        if (gamepad.buttonX.pressed)
        {
            state.SetButton(PadButton::XInputX);
        }
        if (gamepad.buttonY.pressed)
        {
            state.SetButton(PadButton::XInputY);
        }

        // L1/R1
        if (gamepad.leftShoulder.pressed)
        {
            state.SetButton(PadButton::L1);
        }
        if (gamepad.rightShoulder.pressed)
        {
            state.SetButton(PadButton::R1);
        }
        
        // L2/R2
        if (gamepad.leftTrigger.pressed)
        {
            state.SetButton(PadButton::L2);
        }
        if (gamepad.rightTrigger.pressed)
        {
            state.SetButton(PadButton::R2);
        }
        
        // L3/R3
        if (gamepad.leftThumbstickButton.pressed)
        {
            state.SetButton(PadButton::L3);
        }
        if (gamepad.rightThumbstickButton.pressed)
        {
            state.SetButton(PadButton::R3);
        }

        // 左右スティック
        state.GetLeftStick() = MGL::Vector2(gamepad.leftThumbstick.xAxis.value, gamepad.leftThumbstick.yAxis.value);
        state.GetRightStick() = MGL::Vector2(gamepad.rightThumbstick.xAxis.value, gamepad.rightThumbstick.yAxis.value);
        state.ApplyLeftStickToDPad();

        // Homeボタン
        if (gamepad.buttonHome.pressed)
        {
            state.SetButton(PadButton::Home);
        }

        // プライマリメニューボタン
        if (gamepad.buttonMenu.pressed)
        {
            state.SetButton(PadButton::XInputMenu);
        }

        // セカンダリメニューボタン
        if (gamepad.buttonOptions.pressed)
        {
            state.SetButton(PadButton::XInputView);
        }
        
        // Share
        if (@available(macOS 12.0, iOS 15.0, tvOS 15.0, *))
        {
            if (IsAvailable(AvailableFlag::XboxShareButton))
            {
                if (gamepad.buttonShare.pressed)
                {
                    state.SetButton(PadButton::XInputShare);
                }
            }
        }
        
        // 背面パドル
        if (gamepad.paddleButton1.pressed)
        {
            state.SetButton(PadButton::XInputPaddle1);
        }
        if (gamepad.paddleButton2.pressed)
        {
            state.SetButton(PadButton::XInputPaddle2);
        }
        if (gamepad.paddleButton3.pressed)
        {
            state.SetButton(PadButton::XInputPaddle3);
        }
        if (gamepad.paddleButton4.pressed)
        {
            state.SetButton(PadButton::XInputPaddle4);
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      DualShock4ゲームパッドの更新処理
 *  \param[out] state       更新するパッドステート
 *  \param[in]  controller  コントローラクラス
 */
/* ------------------------------------------------------------------------- */
void AppleGCGamepadDelegate::UpdateDualShock4PadState(PadState &state, GCController *controller) noexcept
{
    if (@available(macOS 11.0, iOS 14.0, tvOS 14.0, *))
    {
        // GCExtendedGamepadをGCDualShockGamepadにキャスト
        GCDualShockGamepad *gamepad = nil;
        if ([controller.extendedGamepad isKindOfClass:[GCDualShockGamepad class]])
        {
            gamepad = static_cast<GCDualShockGamepad *>(controller.extendedGamepad);
        }
        else
        {
            return;
        }

        // D-Padの上下左右
        if (gamepad.dpad.up.pressed)
        {
            state.SetButton(PadButton::Up);
        }
        if (gamepad.dpad.down.pressed)
        {
            state.SetButton(PadButton::Down);
        }
        if (gamepad.dpad.left.pressed)
        {
            state.SetButton(PadButton::Left);
        }
        if (gamepad.dpad.right.pressed)
        {
            state.SetButton(PadButton::Right);
        }

        // ABXY
        if (gamepad.buttonA.pressed)
        {
            state.SetButton(PadButton::DSCross);
        }
        if (gamepad.buttonB.pressed)
        {
            state.SetButton(PadButton::DSCircle);
        }
        if (gamepad.buttonX.pressed)
        {
            state.SetButton(PadButton::DSSquare);
        }
        if (gamepad.buttonY.pressed)
        {
            state.SetButton(PadButton::DSTriangle);
        }

        // L1/R1
        if (gamepad.leftShoulder.pressed)
        {
            state.SetButton(PadButton::L1);
        }
        if (gamepad.rightShoulder.pressed)
        {
            state.SetButton(PadButton::R1);
        }
        
        // L2/R2
        if (gamepad.leftTrigger.pressed)
        {
            state.SetButton(PadButton::L2);
        }
        if (gamepad.rightTrigger.pressed)
        {
            state.SetButton(PadButton::R2);
        }
        
        // L3/R3
        if (gamepad.leftThumbstickButton.pressed)
        {
            state.SetButton(PadButton::L3);
        }
        if (gamepad.rightThumbstickButton.pressed)
        {
            state.SetButton(PadButton::R3);
        }

        // 左右スティック
        state.GetLeftStick() = MGL::Vector2(gamepad.leftThumbstick.xAxis.value, gamepad.leftThumbstick.yAxis.value);
        state.GetRightStick() = MGL::Vector2(gamepad.rightThumbstick.xAxis.value, gamepad.rightThumbstick.yAxis.value);
        state.ApplyLeftStickToDPad();
        
        // Homeボタン
        if (gamepad.buttonHome.pressed)
        {
            state.SetButton(PadButton::Home);
        }

        // プライマリメニューボタン
        if (gamepad.buttonMenu.pressed)
        {
            state.SetButton(PadButton::DSOptions);
        }

        // セカンダリメニューボタン
        if (gamepad.buttonOptions.pressed)
        {
            state.SetButton(PadButton::DSShare);
        }
        
        // タッチパッドボタン
        if (gamepad.touchpadButton.pressed)
        {
            state.SetButton(PadButton::DSTouchPad);
        }
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      DualSenseゲームパッドの更新処理
 *  \param[out] state       更新するパッドステート
 *  \param[in]  controller  コントローラクラス
 */
/* ------------------------------------------------------------------------- */
void AppleGCGamepadDelegate::UpdateDualSensePadState(PadState &state, GCController *controller) noexcept
{
    if (@available(macOS 11.3, iOS 14.5, tvOS 14.5, *))
    {
        // GCExtendedGamepadをGCDualSenseGamepadにキャスト
        GCDualSenseGamepad *gamepad = nil;
        if ([controller.extendedGamepad isKindOfClass:[GCDualSenseGamepad class]])
        {
            gamepad = static_cast<GCDualSenseGamepad *>(controller.extendedGamepad);
        }
        else
        {
            return;
        }

        // D-Padの上下左右
        if (gamepad.dpad.up.pressed)
        {
            state.SetButton(PadButton::Up);
        }
        if (gamepad.dpad.down.pressed)
        {
            state.SetButton(PadButton::Down);
        }
        if (gamepad.dpad.left.pressed)
        {
            state.SetButton(PadButton::Left);
        }
        if (gamepad.dpad.right.pressed)
        {
            state.SetButton(PadButton::Right);
        }

        // ABXY
        if (gamepad.buttonA.pressed)
        {
            state.SetButton(PadButton::DSCross);
        }
        if (gamepad.buttonB.pressed)
        {
            state.SetButton(PadButton::DSCircle);
        }
        if (gamepad.buttonX.pressed)
        {
            state.SetButton(PadButton::DSSquare);
        }
        if (gamepad.buttonY.pressed)
        {
            state.SetButton(PadButton::DSTriangle);
        }

        // L1/R1
        if (gamepad.leftShoulder.pressed)
        {
            state.SetButton(PadButton::L1);
        }
        if (gamepad.rightShoulder.pressed)
        {
            state.SetButton(PadButton::R1);
        }
        
        // L2/R2
        if (gamepad.leftTrigger.pressed)
        {
            state.SetButton(PadButton::L2);
        }
        if (gamepad.rightTrigger.pressed)
        {
            state.SetButton(PadButton::R2);
        }
        
        // L3/R3
        if (gamepad.leftThumbstickButton.pressed)
        {
            state.SetButton(PadButton::L3);
        }
        if (gamepad.rightThumbstickButton.pressed)
        {
            state.SetButton(PadButton::R3);
        }

        // 左右スティック
        state.GetLeftStick() = MGL::Vector2(gamepad.leftThumbstick.xAxis.value, gamepad.leftThumbstick.yAxis.value);
        state.GetRightStick() = MGL::Vector2(gamepad.rightThumbstick.xAxis.value, gamepad.rightThumbstick.yAxis.value);
        state.ApplyLeftStickToDPad();

        // Homeボタン
        if (gamepad.buttonHome.pressed)
        {
            state.SetButton(PadButton::Home);
        }

        // プライマリメニューボタン
        if (gamepad.buttonMenu.pressed)
        {
            state.SetButton(PadButton::DSOptions);
        }

        // セカンダリメニューボタン
        if (gamepad.buttonOptions.pressed)
        {
            state.SetButton(PadButton::DSShare);
        }
        
        // タッチパッドボタン
        if (gamepad.touchpadButton.pressed)
        {
            state.SetButton(PadButton::DSTouchPad);
        }
    }
}

}   // namespace MGL::Input
#endif  // MGL_GAMEPAD_DELEGATE_ENABLE_APPLE_GC

// vim: et ts=4 sw=4 sts=4
