// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_iokit_hid_generic_driver.cc
 *  \brief      MGL IOKit HID 標準ゲームパッドドライバ
 *  \date       Since: January 11, 2021. 18:04:36 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/gamepad/iokit_hid_driver/mgl_iokit_hid_generic_driver.h>
#if defined(MGL_GAMEPAD_DELEGATE_ENABLE_IOKIT_HID)

#include <mgl/input/gamepad/mgl_gamepad_state.h>

namespace MGL::Input::IOKitHID
{
namespace
{
// ボタンとusageを関連付けるテーブル
constexpr PadButton kButtonTable[] =
{
    PadButton::Button01,
    PadButton::Button02,
    PadButton::Button03,
    PadButton::Button04,
    PadButton::Button05,
    PadButton::Button06,
    PadButton::Button07,
    PadButton::Button08,
    PadButton::Button09,
    PadButton::Button10,
    PadButton::Button11,
    PadButton::Button12,
    PadButton::Button13,
    PadButton::Button14,
    PadButton::Button15,
    PadButton::Button16,
    PadButton::Button17,
    PadButton::Button18,
    PadButton::Button19,
    PadButton::Button20,
    PadButton::Button21,
};

// ボタンの最大認識数
constexpr size_t kButtonMax = sizeof(kButtonTable) / sizeof(PadButton);
}    // namespace


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      初期化処理
 *  \param[in]  device  デバイス
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
bool GenericGamepadDriver::Initialize(IOHIDDeviceRef device) noexcept
{
    const auto *elemArray = IOHIDDeviceCopyMatchingElements(device, nullptr, 0);
    auto elemCount = CFArrayGetCount(elemArray);

    for (CFIndex elemIndex = 0; elemIndex < elemCount; ++elemIndex)
    {
        auto *elemRef = (IOHIDElementRef)CFArrayGetValueAtIndex(elemArray, elemIndex);

        if (IOHIDElementGetUsagePage(elemRef) == kHIDPage_GenericDesktop)
        {
            // 各々の軸の情報を初期化
            switch (IOHIDElementGetUsage(elemRef))
            {
                case kHIDUsage_GD_X: _axisX.SetParameter(elemRef); break;
                case kHIDUsage_GD_Y: _axisY.SetParameter(elemRef); break;
                case kHIDUsage_GD_Z: _axisZ.SetParameter(elemRef); break;
                case kHIDUsage_GD_Rz: _axisRz.SetParameter(elemRef); break;

                case kHIDUsage_GD_DPadUp:
                    _hasDPad = true;
                    break;

                case kHIDUsage_GD_Hatswitch:
                    _hasHatswitch = true;
                    break;

                default:
                    break;
            }
        }
    }

    CFRelease(elemArray);

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ゲームパッドステートの更新処理
 *  \param[in,out]  state   ゲームパッドステート
 *  \param[in]      device  デバイス
 *  \retval         true    成功
 *  \retval         false   失敗
 */
/* ------------------------------------------------------------------------- */
bool GenericGamepadDriver::UpdateState(MGL::Input::PadState &state, IOHIDDeviceRef device) noexcept
{
    // デバイスが持つ全ての要素とその数を取得
    const auto *elemArray = IOHIDDeviceCopyMatchingElements(device, nullptr, 0);
    auto elemCount = CFArrayGetCount(elemArray);

    for (CFIndex elemIndex = 0; elemIndex < elemCount; ++elemIndex)
    {
        auto *elemRef = (IOHIDElementRef)CFArrayGetValueAtIndex(elemArray, elemIndex);    // 要素
        const uint32_t usagePage = IOHIDElementGetUsagePage(elemRef);                     // Usage Page
        const uint32_t usage = IOHIDElementGetUsage(elemRef);                             // Usage

        // 値を取得
        IOHIDValueRef valueRef = nullptr;
        if (IOHIDDeviceGetValue(device, elemRef, &valueRef) != kIOReturnSuccess)
        {
            continue;
        }
        auto value = static_cast<int32_t>(IOHIDValueGetIntegerValue(valueRef));

        // UsagePageがGenericDesktopの要素を読み取る
        if (usagePage == kHIDPage_GenericDesktop)
        {
            switch (usage)
            {
                // X軸: 左スティックのX軸 or D-Padの左右
                case kHIDUsage_GD_X:
                    if (!_hasDPad || !_hasHatswitch)
                    {
                        if (_axisX.IsPositive(value))
                        {
                            state.SetButton(PadButton::Right);
                        }
                        else if (_axisX.IsNegative(value))
                        {
                            state.SetButton(PadButton::Left);
                        }
                    }
                    else
                    {
                        state.GetLeftStick().x = _axisX.GetValue(value);
                    }
                    break;

                // Y軸: 左スティックのY軸 or D-Padの上下
                case kHIDUsage_GD_Y:
                    if (!_hasDPad || !_hasHatswitch)
                    {
                        if (_axisY.IsPositive(value))
                        {
                            state.SetButton(PadButton::Down);
                        }
                        else if (_axisY.IsNegative(value))
                        {
                            state.SetButton(PadButton::Up);
                        }
                    }
                    else
                    {
                        state.GetLeftStick().y = _axisX.GetValue(value);
                    }
                    break;

                // Z軸: 右スティックのX軸
                case kHIDUsage_GD_Z:
                    state.GetRightStick().x = _axisZ.GetValue(value);
                    break;

                // Rz軸: 右スティックのY軸
                case kHIDUsage_GD_Rz:
                    state.GetRightStick().y = _axisRz.GetValue(value);
                    break;

                // D-Padの上
                case kHIDUsage_GD_DPadUp:
                    if (value > 0)
                    {
                        state.SetButton(PadButton::Up);
                    }
                    break;

                // D-Padの下
                case kHIDUsage_GD_DPadDown:
                    if (value > 0)
                    {
                        state.SetButton(PadButton::Down);
                    }
                    break;

                // D-Padの左
                case kHIDUsage_GD_DPadLeft:
                    if (value > 0)
                    {
                        state.SetButton(PadButton::Left);
                    }
                    break;

                // D-Padの右
                case kHIDUsage_GD_DPadRight:
                    if (value > 0)
                    {
                        state.SetButton(PadButton::Right);
                    }
                    break;

                // ハットスイッチ
                case kHIDUsage_GD_Hatswitch:
                    switch (value)
                    {
                        case 0: state.SetButton(PadButton::Up); break;
                        case 1: state.SetButton(PadButton::Up | PadButton::Right); break;
                        case 2: state.SetButton(PadButton::Right); break;
                        case 3: state.SetButton(PadButton::Down | PadButton::Right); break;
                        case 4: state.SetButton(PadButton::Down); break;
                        case 5: state.SetButton(PadButton::Down | PadButton::Left); break;
                        case 6: state.SetButton(PadButton::Left); break;
                        case 7: state.SetButton(PadButton::Up | PadButton::Left); break;
                        default: break;
                    }
                    break;

                default:
                    break;
            }
        }
        // UsagePageがButtonの場合はボタン入力
        else if (usagePage == kHIDPage_Button)
        {
            // usageに1から始まるボタン番号が割り当てられており，入力があればvalueに値が入る
            // ボタンが感圧ボタンなどのアナログ入力の場合，valueには0と1以外の値が入るが，標準ドライバでは扱わないことにする
            if ((usage >= 1) && (usage <= kButtonMax))
            {
                if (value > 0)
                {
                    auto buttonIndex = static_cast<size_t>(usage - 1);
                    state.SetButton(kButtonTable[buttonIndex]);
                }
            }
        }
    }

    // 要素配列にはもうアクセスしないので解放
    CFRelease(elemArray);

    return true;
}
}    // namespace MGL::Input::IOKitHID

#endif    // MGL_GAMEPAD_DELEGATE_ENABLE_IOKIT_HID

// vim: et ts=4 sw=4 sts=4
