// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_iokit_hid_axis.cc
 *  \brief      MGL IOKit HID用軸パラメータクラス
 *  \date       Since: January 14, 2021. 1:23:24 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/input/gamepad/iokit_hid_driver/mgl_iokit_hid_axis.h>
#if defined(MGL_GAMEPAD_DELEGATE_ENABLE_IOKIT_HID)

#include <algorithm>
#include <cmath>

namespace MGL::Input::IOKitHID
{
namespace
{
constexpr float kCenterThresholdCoeff = 0.4f;    // 軸の中央部分の遊び係数
constexpr float kNormalizedThreshold = 0.01f;    // 正規化後の値の中央部分の遊び
}    // namespace


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      軸パラメータの設定
 *  \param[in]  elemRef     デバイスから取得した軸の要素
 */
/* ------------------------------------------------------------------------- */
void Axis::SetParameter(IOHIDElementRef elemRef) noexcept
{
    _min = static_cast<float>(IOHIDElementGetLogicalMin(elemRef));
    _max = static_cast<float>(IOHIDElementGetLogicalMax(elemRef));
    _center = _max - ((std::abs(_min) + std::abs(_max)) / 2.0f);
    _threshold = std::abs(_max - _center) * kCenterThresholdCoeff;

    _isAvailable = true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      軸の値を取得
 *  \param[in]  value   デバイスから取得した値
 *  \return     -1.0f 〜 1.0fに正規化した軸の値
 */
/* ------------------------------------------------------------------------- */
float Axis::GetValue(int32_t value) const noexcept
{
    if (!_isAvailable)
    {
        return 0.0f;
    }

    const float axisValue = static_cast<float>(value) / (_max - _min) * 2.0f - 1.0f;

    if (std::abs(axisValue) < kNormalizedThreshold)
    {
        return 0.0f;
    }

    return std::clamp(axisValue, -1.0f, 1.0f);
}

}    // namespace MGL::Input::IOKitHID

#endif    // MGL_GAMEPAD_DELEGATE_ENABLE_IOKIT_HID
          // vim: et ts=4 sw=4 sts=4
