// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_initializer_win32.cc
 *  \brief      MGL Win32用イニシャライザクラス
 *  \date       Since: March 27, 2021. 3:33:53 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/initialize/mgl_initializer_win32.h>
#if defined(MGL_TARGET_WIN32)

#include <mgl/render/d3d11/mgl_d3d11_defs.h>
#include <mgl/render/d3d11/mgl_d3d11_renderer_2d.h>
#include <mgl/render/d3d11/mgl_d3d11_texture_generator.h>
#include <mgl/render/d3d11/mgl_d3d11_wic_texture_loader.h>
#include <mgl/render/mgl_render.h>

#include <mgl/audio/renderer/mgl_audio_renderer_xaudio2.h>

#include <mgl/file/delegate/mgl_file_delegate_win32.h>
#include <mgl/file/mgl_file.h>

#include <mgl/input/keyboard/mgl_keyboard_delegate_win32.h>
#include <mgl/input/mouse/mgl_mouse_delegate_win32.h>
#include <mgl/input/gamepad/mgl_gamepad_delegate_xinput.h>
#include <mgl/input/gamepad/mgl_gamepad_delegate_directinput.h>
#include <mgl/input/gamepad/mgl_gamepad_server.h>

#include <mgl/system/application/mgl_application_module_win32.h>
#include <mgl/system/chrono/mgl_chrono_module_win32.h>
#include <mgl/system/debug/mgl_debug_module_win32.h>
#include <mgl/system/locale/mgl_locale_module_win32.h>
#include <mgl/system/window/mgl_window_module_win32.h>
#include <mgl/system/mgl_system_debug_macro.h>

#include <mgl/text/mgl_text_converter.h>

#include <Windows.h>
#include <cstdlib>

namespace MGL::Win32
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
Initializer::Initializer() noexcept
    : _rendererType()
    , _isEnabledXInputGamepad(true)
    , _isEnabledDirectInputGamepad(true)
{
#if defined(MGL_RENDERER_ENABLE_D3D11)
    _rendererType = Render::D3D11::kRendererTypeDirect3D11; // NOLINT(cppcoreguidelines-prefer-member-initializer)  Note: 分岐方法を再考したほうが良さそう
#endif

    SetAudioInitializeMode(Audio::InitializeMode::Sample44k2ch);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      2Dレンダラを生成
 *  \return     2Dレンダラ
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Render::Renderer2DDelegate> Initializer::MakeRenderer2D() const noexcept
{
#if defined(MGL_RENDERER_ENABLE_D3D11)
    if (_rendererType == Render::D3D11::kRendererTypeDirect3D11)
    {
        return STL::make_unique<Render::D3D11::Renderer2D>();
    }
#endif

    return nullptr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャジェネレータを生成
 *  \return     テクスチャジェネレータ
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Render::TextureGenerator> Initializer::MakeTextureGenerator() const noexcept
{
#if defined(MGL_RENDERER_ENABLE_D3D11)
    if (_rendererType == Render::D3D11::kRendererTypeDirect3D11)
    {
        return STL::make_unique<Render::D3D11::TextureGenerator>();
    }
#endif

    return nullptr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャローダーを生成
 */
/* ------------------------------------------------------------------------- */
void Initializer::MakeTextureLoader(Render::TextureStorage &textureStorage) const noexcept
{
    // WICによるテクスチャローダーを追加
    textureStorage.RegisterLoader(Render::D3D11::WICTextureLoader::kLoaderKey, STL::make_unique<Render::D3D11::WICTextureLoader>());
    textureStorage.SetDefaultLoader(Render::D3D11::WICTextureLoader::kLoaderKey);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      オーディオレンダラの生成
 *  \return     オーディオレンダラ
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Audio::Renderer> Initializer::MakeAudioRenderer() const noexcept
{
#if defined(MGL_AUDIO_RENDERER_ENABLE_XAUDIO2)
    return STL::make_unique<Audio::XAudio2Renderer>();
#else
    return nullptr;
#endif
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルシステム初期化後の処理
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
File::Result Initializer::DidInitializeFileSystem() noexcept
{
    try
    {
        File::ThrowingUtility utility;
        
        // Win32デリゲートを追加してデフォルトに設定
        utility.AddDelegate<File::Win32Delegate>(File::Win32Delegate::kDelegateKey);
        utility.SetDefaultDelegate(File::Win32Delegate::kDelegateKey);

        // カレントディレクトリを各種ディレクトリとしてマウント
        auto currentDirectory = GetWorkDirectory();
        utility.Mount("cwd", currentDirectory.c_str(), File::MountAccessType::Writable);
        utility.Mount("resource", currentDirectory.c_str(), File::MountAccessType::ReadOnly);
        utility.Mount("user", currentDirectory.c_str(), File::MountAccessType::Writable);
        utility.Mount("temp", currentDirectory.c_str(), File::MountAccessType::Writable);

        // 実行ディレクトリをマウント
        auto execDirectory = GetExecDirectory();
        utility.Mount("exec", execDirectory.c_str(), File::MountAccessType::Writable);

        return File::Result::Succeeded();
    }
    catch (const File::Exception &fileException)
    {
        MGL_ERROR(fileException.what());
        return {fileException.GetError()};
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      カレントディレクトリを取得
 *  \return     カレントディレクトリ
 */
/* ------------------------------------------------------------------------- */
STL::string Initializer::GetWorkDirectory() noexcept
{
    // カレントディレクトリを取得
    auto bufferLength = (static_cast<size_t>(GetCurrentDirectory(0, nullptr)) + 1) * sizeof(wchar_t);
    auto wCurrDir = STL::make_unique<wchar_t []>(bufferLength);
    GetCurrentDirectory(static_cast<DWORD>(bufferLength), wCurrDir.get());

    // UTF-8に変換
    STL::string utf8String;
    Text::ToUTF8(utf8String, wCurrDir.get(), bufferLength, Text::Encoding::UTF16LE);

    // パスの区切り文字を'/'に変換
    size_t i = 0;
    while (utf8String[i] != '\0')
    {
        if (utf8String[i] == '\\')
        {
            utf8String[i] = '/';
        }

        i++;
    }

    return utf8String;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実行ファイルのあるディレクトリを取得
 *  \return     実行ファイルのあるディレクトリ
 */
/* ------------------------------------------------------------------------- */
STL::string Initializer::GetExecDirectory() noexcept
{
    constexpr DWORD kBufferLength = 1024;

    // 実行中のプロセスのパスを取得
    wchar_t wCurrDir[kBufferLength];
    GetModuleFileName(nullptr, wCurrDir, kBufferLength);

    // UTF-8に変換
    STL::string utf8String;
    Text::ToUTF8(utf8String, wCurrDir, kBufferLength, Text::Encoding::UTF16LE);

    // パスの区切り文字を'/'に変換
    size_t i = 0;
    size_t lastSeparatorPoint = 0;
    while (utf8String[i] != '\0')
    {
        if (utf8String[i] == '\\')
        {
            utf8String[i] = '/';
            lastSeparatorPoint = i;
        }

        i++;
    }

    // 欲しいのはディレクトリなので，最後の区切り文字以降を捨てる
    utf8String[lastSeparatorPoint] = '\0';

    return utf8String;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      キーボード入力デリゲートの生成
 *  \return     キーボード入力デリゲート
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Input::KeyboardDelegate> Initializer::MakeKeyboardDelegate() const noexcept
{
    return STL::make_unique<Input::Win32KeyboardDelegate>();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウス入力デリゲートの生成
 *  \return     マウス入力デリゲート
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Input::MouseDelegate> Initializer::MakeMouseDelegate() const noexcept
{
    return STL::make_unique<Input::Win32MouseDelegate>();
}


/* ------------------------------------------------------------------------- */
/*!
    *  \brief      ゲームパッドサーバ初期化後の処理
    *  \param[in]  server      ゲームパッドサーバ
    *  \retval     true        成功
    *  \retval     false       失敗
    */
/* ------------------------------------------------------------------------- */
bool Initializer::DidInitializeGamepadServer(Input::GamepadServer &server) const noexcept
{
    if (_isEnabledXInputGamepad)
    {
        server.AddDelegate(STL::make_unique<Input::XInputGamepadDelegate>(server));
    }

    if (_isEnabledDirectInputGamepad)
    {
        server.AddDelegate(STL::make_unique<Input::DirectInputGamepadDelegate>(server));
    }

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アプリケーションモジュールを生成
 *  \return     アプリケーションモジュール
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<System::ApplicationModuleInterface> Initializer::MakeApplicationModule() const noexcept
{
    return STL::make_unique<System::Win32ApplicationModule>();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      日付と時刻のモジュールを生成
 *  \return     日付と時刻のモジュール
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<System::ChronoModuleInterface> Initializer::MakeChronoModule() const noexcept
{
    return STL::make_unique<System::Win32ChronoModule>();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デバッグモジュールを生成
 *  \return     デバッグモジュール
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<System::DebugModuleInterface> Initializer::MakeDebugModule() const noexcept
{
    return STL::make_unique<System::Win32DebugModule>(GetLogLevel());
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ロケール情報モジュールを生成
 *  \return     ロケール情報モジュール
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<System::LocaleModuleInterface> Initializer::MakeLocaleModule() const noexcept
{
    return STL::make_unique<System::Win32LocaleModule>();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ウィンドウモジュールを生成
 *  \return     ウィンドウモジュール
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<System::WindowModuleInterface> Initializer::MakeWindowModule() const noexcept
{
    return STL::make_unique<System::Win32WindowModule>();
}

}   // namespace MGL::Win32

#endif  // MGL_TARGET_WIN32

// vim: et ts=4 sw=4 sts=4
