// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_initializer_macos.mm
 *  \brief      MGL macOS用イニシャライザクラス
 *  \date       Since: March 18, 2021. 15:31:50 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/initialize/mgl_initializer_macos.h>
#if defined(MGL_TARGET_MACOS)

#include <Foundation/Foundation.h>

#include <mgl/audio/renderer/mgl_audio_renderer_apple_coreaudio.h>

#include <mgl/file/delegate/mgl_file_delegate_nsfile.h>
#include <mgl/file/delegate/mgl_file_delegate_posix.h>
#include <mgl/file/mgl_file.h>

#include <mgl/input/gamepad/mgl_gamepad_delegate_iokit_hid.h>
#include <mgl/input/gamepad/mgl_gamepad_delegate_apple_gc.h>
#include <mgl/input/gamepad/mgl_gamepad_server.h>
#include <mgl/input/keyboard/mgl_keyboard_delegate_event.h>
#include <mgl/input/keyboard/mgl_keyboard_delegate_apple_gc.h>
#include <mgl/input/mouse/mgl_mouse_delegate_macos.h>

#include <mgl/render/metal/mgl_metal_defs.h>
#include <mgl/render/metal/mgl_metal_texture_generator.h>
#include <mgl/render/metal/mgl_metalkit_texture_loader.h>
#include <mgl/render/mgl_render.h>

#include <mgl/system/application/mgl_application_module_macos.h>
#include <mgl/system/chrono/mgl_chrono_module_apple.h>
#include <mgl/system/debug/mgl_debug_module_apple.h>
#include <mgl/system/locale/mgl_locale_module_apple.h>
#include <mgl/system/window/mgl_window_module_macos.h>

#include <mgl/system/mgl_system_debug_macro.h>

namespace MGL::Apple::macOS
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 */
/* ------------------------------------------------------------------------- */
Initializer::Initializer() noexcept
    : _isEnabledIOKitGamepadDelegate(true)
    , _isEnabledGCGamepadDelegate(true)
    , _isEnabledGCKeyboardDelegate(false)
{
    SetAudioInitializeMode(Audio::InitializeMode::Sample44k2ch);
    SetLogLevel(System::LogLevel::AppTrace);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      2Dレンダラを生成
 *  \return     2Dレンダラ
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Render::Renderer2DDelegate> Initializer::MakeRenderer2D() const noexcept
{
    return STL::make_unique<Render::Metal::Renderer2D>();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャジェネレータを生成
 *  \return     テクスチャジェネレータ
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Render::TextureGenerator> Initializer::MakeTextureGenerator() const noexcept
{
    return STL::make_unique<Render::Metal::TextureGenerator>();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      テクスチャローダを生成
 */
/* ------------------------------------------------------------------------- */
void Initializer::MakeTextureLoader(Render::TextureStorage &textureStorage) const noexcept
{
    // MetalKitによるテクスチャローダーを追加
    textureStorage.RegisterLoader(Render::Metal::MetalKitTextureLoader::kLoaderKey, STL::make_unique<Render::Metal::MetalKitTextureLoader>());
    textureStorage.SetDefaultLoader(Render::Metal::MetalKitTextureLoader::kLoaderKey);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      オーディオレンダラの生成
 *  \return     オーディオレンダラ
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Audio::Renderer> Initializer::MakeAudioRenderer() const noexcept
{
    return STL::make_unique<Audio::AppleCoreAudioRenderer>();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルシステム初期化後の処理
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
File::Result Initializer::DidInitializeFileSystem() noexcept
{
    try
    {
        File::ThrowingUtility utility;
        
        // POSIXデリゲートを追加してデフォルトに設定
        utility.AddDelegate<File::POSIXDelegate>(File::POSIXDelegate::kDelegateKey);
        utility.SetDefaultDelegate(File::POSIXDelegate::kDelegateKey);
        
        auto mainBundle = NSBundle.mainBundle;
        
        // リソースディレクトリをマウント
        utility.Mount("resource", mainBundle.resourcePath.UTF8String, File::MountAccessType::ReadOnly);
        
        // ユーザーディレクトリをマウント
        auto *appSupportPath = NSSearchPathForDirectoriesInDomains(NSApplicationSupportDirectory, NSUserDomainMask, YES);
        if (appSupportPath.count >= 1)
        {
            // Application SupportディレクトリにバンドルIDのディレクトリを生成する
            auto userPath = STL::string(appSupportPath.firstObject.UTF8String) + "/" + STL::string(mainBundle.bundleIdentifier.UTF8String);
            utility.MakeDirectory(userPath.c_str());
            
            // 生成したディレクトリにマウント
            utility.Mount("user", userPath.c_str(), File::MountAccessType::Writable);
        }
        
        // 実行ファイルのあるディレクトリをマウント
        utility.Mount("exec", mainBundle.executablePath.stringByDeletingLastPathComponent.UTF8String, File::MountAccessType::Writable);

        // テンポラリディレクトリをマウント
        utility.Mount("temp", NSTemporaryDirectory().UTF8String, File::MountAccessType::Writable);
        
        // 作業ディレクトリをマウント
        utility.Mount("cwd", NSFileManager.defaultManager.currentDirectoryPath.UTF8String, File::MountAccessType::Writable);

        return File::Result::Succeeded();
    }
    catch (const MGL::File::Exception &fileException)
    {
        MGL_ERROR(fileException.what());
        return {fileException.GetError()};
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      アプリケーションモジュールの生成
 *  \return     アプリケーションモジュール
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<System::ApplicationModuleInterface> Initializer::MakeApplicationModule() const noexcept
{
    return STL::make_unique<System::MacOSApplicationModule>();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      日付と時刻のモジュールを生成
 *  \return     日付と時刻のモジュール
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<System::ChronoModuleInterface> Initializer::MakeChronoModule() const noexcept
{
    return STL::make_unique<System::AppleChronoModule>();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デバッグモジュールを生成
 *  \return     デバッグモジュール
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<System::DebugModuleInterface> Initializer::MakeDebugModule() const noexcept
{
    return STL::make_unique<System::AppleDebugModule>(GetLogLevel());
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ロケール情報モジュールを生成
 *  \return     ロケール情報モジュール
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<System::LocaleModuleInterface> Initializer::MakeLocaleModule() const noexcept
{
    return STL::make_unique<System::AppleLocaleModule>();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ウィンドウモジュールを生成
 *  \return     ウィンドウモジュール
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<System::WindowModuleInterface> Initializer::MakeWindowModule() const noexcept
{
    return STL::make_unique<System::MacOSWindowModule>();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      キーボード入力デリゲートの生成
 *  \return     キーボード入力デリゲート
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Input::KeyboardDelegate> Initializer::MakeKeyboardDelegate() const noexcept
{
    // OSが対応していればGCKeyboardデリゲートを使用する
    if (_isEnabledGCKeyboardDelegate)
    {
        if (Input::AppleGCKeyboardDelegate::IsAvailable())
        {
            return STL::make_unique<Input::AppleGCKeyboardDelegate>();
        }
    }

    // GCKeyboardが使えない or 使わない場合はイベントキーボードデリゲートを使用
    return STL::make_unique<Input::EventKeyboardDelegate>();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウス入力デリゲートの生成
 *  \return     マウス入力デリゲート
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Input::MouseDelegate> Initializer::MakeMouseDelegate() const noexcept
{
    return STL::make_unique<Input::MacOSMouseDelegate>();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ゲームパッドサーバ初期化後の処理
 *  \param[in]  server      ゲームパッドサーバ
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
bool Initializer::DidInitializeGamepadServer(Input::GamepadServer &server) const noexcept
{
    // GameControllerフレームワークによるゲームパッドデリゲートを追加
    if (_isEnabledGCGamepadDelegate)
    {
        if (!server.AddDelegate(STL::make_unique<Input::AppleGCGamepadDelegate>(server)))
        {
            MGL_WARNING("[MGL Gamepad] Failed to initialize GC gamepad delegate.");
        }
    }
    
    // IOKitによるゲームパッドデリゲートを追加
    if (_isEnabledIOKitGamepadDelegate)
    {
        if (!server.AddDelegate(STL::make_unique<Input::IOKitHIDGamepadDelegate>(server)))
        {
            MGL_WARNING("[MGL Gamepad] Failed to initialize HID gamepad delegate.");
        }
    }
    
    return true;
}
}   // namespace MGL::Apple::macOS
#endif  // MGL_TARGET_MACOS

// vim: et ts=4 sw=4 sts=4
