// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_initialize.cc
 *  \brief      MGL 初期化と終了
 *  \date       Since: November 28, 2020. 19:14:31 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/initialize/mgl_initialize.h>

#include <mgl/audio/mgl_audio_player.h>
#include <mgl/file/mgl_file_mounter.h>
#include <mgl/input/gamepad/mgl_gamepad_server.h>
#include <mgl/input/keyboard/mgl_keyboard_server.h>
#include <mgl/input/mouse/mgl_mouse_server.h>
#include <mgl/input/touch/mgl_touch_server.h>
#include <mgl/memory/mgl_memory.h>
#include <mgl/render/font/mgl_font_storage.h>
#include <mgl/system/mgl_system_application.h>
#include <mgl/system/mgl_system_debug_macro.h>
#include <mgl/system/mgl_system_locale.h>
#include <mgl/system/mgl_system_module_set.h>

namespace MGL
{
namespace
{
bool s_IsInitialized = false;            // 初期化済みフラグ
Event::Handle s_EventDidInitRenderer;    // レンダラ初期化用イベント

/* ------------------------------------------------------------------------- */
/*!
*  \brief      レンダラの初期化が後回しにされた場合に呼び出される処理
*  \param[in]  callbackArg     未使用
*  \param[in]  notifyArg       未使用
*/
/* ------------------------------------------------------------------------- */
void OnEventDidInitializableRenderer([[maybe_unused]] void *callbackArg, [[maybe_unused]] void *notifyArg)
{
    if (Render::RendererSet::GetInstance().InitializeRenderer())
    {
        s_IsInitialized = true;
        Render::FontStorage::CreateInstance();
        Event::Notify(Event::NotifyType::AppInitialize);
        s_EventDidInitRenderer.Unregister();
    }
    else
    {
        MGL_ERROR("[MGL Renderer] Failed to initialize.");
        System::Application().Quit();
    }
}
}    // namespace


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      MGL初期化処理
 *  \param[in]  initializer     イニシャライザクラス
 *  \param[in]  makeAppDelegate アプリケーションデリゲート生成関数
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
bool Initialize(InitializerDelegate &initializer, MakeAppDelegateFunction makeAppDelegate) noexcept
{
    // 既に初期化済みであれば再度実行しない
    if (s_IsInitialized)
    {
        return true;
    }

    // シングルトン解放クラスを初期化
    if (!SingletonFinalizer::Initialize())
    {
        return false;
    }

    // 有効なアロケータが設定されていなければデフォルトのアロケータを構築する
    if (!Memory::IsAvailableAllocator())
    {
        Memory::SetAllocator(nullptr);
    }

    // イベントを初期化
    Event::Notifier::CreateInstance();

    // システムモジュールを生成
    auto &systemModuleSet = System::ModuleSet::CreateInstance();
    systemModuleSet.Initialize(initializer);

    // 使用言語がイニシャライザで指定されている場合はそれを使用する
    if (auto language = initializer.GetLanguage(); language != System::Language::Unknown)
    {
        System::Locale().SetLanguage(language);
    }

    // ファイル関連の初期化
    File::Mounter::CreateInstance();
    if (auto result = initializer.DidInitializeFileSystem(); result.HasError())
    {
        // NoOperationの場合は不要とみなしてエラー扱いにしない
        if (result.GetError() != File::Error::NoOperation)
        {
            MGL_ERROR("[MGL File] Failed to initialize: %d", result.GetErrorCode());
            return false;
        }
    }

    // オーディオを初期化
    auto &audioPlayer = Audio::Player::CreateInstance();
    if (auto audioInitMode = initializer.GetAudioInitializeMode(); audioInitMode != Audio::InitializeMode::None)
    {
        if (auto audioRenderer = initializer.MakeAudioRenderer(); audioRenderer != nullptr)
        {
            if (!audioPlayer.Initialize(audioRenderer, audioInitMode))
            {
                MGL_ERROR("[MGL Audio] Failed to initialize.");
                return false;
            }
        }
    }

    // 各種入力管理サーバを生成
    auto &keyboardServer = Input::KeyboardServer::CreateInstance();
    if (auto keyboardDelegate = initializer.MakeKeyboardDelegate(); keyboardDelegate != nullptr)
    {
        keyboardServer.Initialize(keyboardDelegate);
    }
    auto &mouseServer = Input::MouseServer::CreateInstance();
    if (auto mouseDelegate = initializer.MakeMouseDelegate(); mouseDelegate != nullptr)
    {
        mouseServer.Initialize(mouseDelegate);
    }
    auto &touchServer = Input::TouchServer::CreateInstance();
    if (auto touchDelegate = initializer.MakeTouchDelegate(); touchDelegate != nullptr)
    {
        touchServer.Initialize(touchDelegate);
    }
    Input::GamepadServer::CreateInstance();
    initializer.DidInitializeGamepadServer(Input::GamepadServer::GetInstance());

    // レンダラを生成
    auto &rendererSet = Render::RendererSet::CreateInstance();
    if (rendererSet.Initialize(initializer))
    {
        if (rendererSet.CanInitializeRenderer())
        {
            if (!rendererSet.InitializeRenderer())
            {
                MGL_ERROR("[MGL Renderer] Failed to initialize.");
                return false;
            }

            Render::FontStorage::CreateInstance();
        }
        else
        {
            // レンダラが初期化できない状況であればイベントを登録
            s_EventDidInitRenderer = Event::Handle(Event::NotifyType::DidInitializableRenderer, OnEventDidInitializableRenderer, nullptr);
        }
    }

    // アプリケーションデリゲートを生成
    if (makeAppDelegate != nullptr)
    {
        STL::unique_ptr<ApplicationDelegate> appDelegate;
        makeAppDelegate(appDelegate);

        // アプリケーションモジュールにデリゲートを登録
        System::ModuleSet::GetInstance().GetApplicationModule().SetApplicationDelegate(std::move(appDelegate));
    }

    // レンダラの初期化が後回しにされていなければアプリケーションの初期化を呼び出す
    if (!s_EventDidInitRenderer.IsValid())
    {
        s_IsInitialized = true;

        // アプリケーションの初期化を実行
        Event::Notify(Event::NotifyType::AppInitialize);
    }

    return true;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      終了処理の呼び出し
 */
/* ------------------------------------------------------------------------- */
void Terminate() noexcept
{
    Event::Notify(Event::NotifyType::AppExit);

    SingletonFinalizer::Finalize();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      MGLが初期化済みかどうかを取得
 *  \retval     true    初期化済み
 *  \retval     false   初期化されていない
 */
/* ------------------------------------------------------------------------- */
bool IsInitialized() noexcept
{
    return s_IsInitialized;
}
}    // namespace MGL
// vim: et ts=4 sw=4 sts=4

