// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_utility.cc
 *  \brief      MGL ファイルユーティリティ
 *  \date       Since: September 29, 2022. 15:40:31 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/file/mgl_file_utility.h>

#include <mgl/file/mgl_file_accessor.h>

namespace MGL::File
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルのサイズを取得
 *  \param[in]  path    サイズを取得するファイルのパス
 *  \return     ファイルサイズ
 */
/* ------------------------------------------------------------------------- */
size_t Utility::GetSize(const PathView &path) noexcept
{
    return Accessor::GetSize(_result, path);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ディレクトリを作成する
 *  \param[in]  path    作成するディレクトリのパス
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
const Result &Utility::MakeDirectory(const PathView &path) noexcept
{
    _result = Accessor::MakeDirectory(path);
    return _result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルの移動・リネーム
 *  \param[in]  sourcePath  移動元のパス
 *  \param[in]  destPath    移動先のパス
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
const Result &Utility::Move(const PathView &sourcePath, const PathView &destPath) noexcept
{
    _result = Accessor::Move(sourcePath, destPath);
    return _result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルの削除
 *  \param[in]  path    削除するファイルのパス
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
const Result &Utility::Remove(const PathView &path) noexcept
{
    _result = Accessor::Remove(path);
    return _result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルのコピー
 *  \param[in]  sourcePath  コピー元のパス
 *  \param[in]  destPath    コピー先のパス
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
const Result &Utility::Copy(const PathView &sourcePath, const PathView &destPath) noexcept
{
    _result = Accessor::Copy(sourcePath, destPath);
    return _result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルの存在をチェック
 *  \param[in]  path        チェックするファイルのパス
 *  \retval     true        存在する
 *  \retval     false       存在しない
 *  \note       失敗時には戻り値にいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
bool Utility::Exists(const PathView &path) noexcept
{
    return Accessor::Exists(_result, path);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      システム標準のファイルであるかをチェック
 *  \param[in]  path    パス
 *  \retval     true    システム標準のファイルである
 *  \retval     false   システム標準のファイルではない
 *  \note       失敗時には戻り値にいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
bool Utility::IsSystemNativeFile(const PathView &path) noexcept
{
    return Accessor::IsSystemNativeFile(_result, path);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウント
 *  \param[in]  mountName   マウント名
 *  \param[in]  path        マウント先のパス
 *  \param[in]  accessType  アクセスタイプ
 *  \param[in]  delegateKey デリゲートキー
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
const Result &Utility::Mount(const PathView &mountName, const PathView &path, MountAccessType accessType, DelegateKey delegateKey) noexcept
{
    _result = Accessor::Mount(mountName, path, accessType, delegateKey);
    return _result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウント解除
 *  \param[in]  mountName   マウント名
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
const Result &Utility::Unmount(const PathView &mountName) noexcept
{
    _result = Accessor::Unmount(mountName);
    return _result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      再マウント
 *  \param[in]  mountName   マウント名
 *  \param[in]  path        マウント先のパス
 *  \param[in]  accessType  アクセスタイプ
 *  \param[in]  delegateKey デリゲートキー
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
const Result &Utility::Remount(const PathView &mountName, const PathView &path, MountAccessType accessType, DelegateKey delegateKey) noexcept
{
    _result = Accessor::Remount(mountName, path, accessType, delegateKey);
    return _result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウントされているかを取得
 *  \param[in]  mountName   マウント名
 *  \retval     true        マウントされている
 *  \retval     false       マウントされていない
 */
/* ------------------------------------------------------------------------- */
bool Utility::IsMounted(const PathView &mountName) noexcept
{
    return Accessor::IsMounted(mountName);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウントパスからシステム標準のパスに変換
 *  \param[in]  path                パス
 *  \return     システム標準のパス．失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
STL::string Utility::GetSystemNativePath(const PathView &path) noexcept
{
    return Accessor::GetSystemNativePath(_result, path);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デリゲートの削除
 *  \param[in]  key     削除するデリゲートキー
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
const Result &Utility::RemoveDelegate(DelegateKey key) noexcept
{
    _result = Accessor::RemoveDelegate(key);
    return _result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デフォルトのデリゲートを設定
 *  \param[in]  key     デフォルトのデリゲートキー
 *  \note       この関数は例外を発生させず，File::SetDefaultDelegate()と等価である
 */
/* ------------------------------------------------------------------------- */
void Utility::SetDefaultDelegate(DelegateKey key) noexcept
{
    Accessor::SetDefaultDelegate(key);
}
}    // namespace MGL::File

// vim: et ts=4 sw=4 sts=4
