// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_throwing_utility.cc
 *  \brief      MGL 例外を発生させるファイルユーティリティ
 *  \date       Since: September 29, 2022. 15:47:34 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/file/mgl_file_throwing_utility.h>

#include <mgl/file/mgl_file_accessor.h>

namespace MGL::File
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルのサイズを取得
 *  \param[in]  path                    サイズを取得するファイルのパス
 *  \return     ファイルサイズ
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
size_t ThrowingUtility::GetSize(const PathView &path)
{
    const auto size = Accessor::GetSize(_result, path);

    if (_result.HasError())
    {
        throw Exception(_result.GetError(), Text::Format("Failed to get size: {}", path).c_str());
    }

    return size;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ディレクトリを作成する
 *  \param[in]  path                    作成するディレクトリのパス
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
void ThrowingUtility::MakeDirectory(const PathView &path)
{
    _result = Accessor::MakeDirectory(path);
    if (_result.HasError())
    {
        throw Exception(_result.GetError(), Text::Format("Failed to make directory: {}", path).c_str());
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルの移動・リネーム
 *  \param[in]  sourcePath              移動元のパス
 *  \param[in]  destPath                移動先のパス
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
void ThrowingUtility::Move(const PathView &sourcePath, const PathView &destPath)
{
    _result = Accessor::Move(sourcePath, destPath);
    if (_result.HasError())
    {
        throw Exception(_result.GetError(), Text::Format("Failed to move file: {} -> {}", sourcePath, destPath).c_str());
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルの削除
 *  \param[in]  path                    削除するファイルのパス
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
void ThrowingUtility::Remove(const PathView &path)
{
    _result = Accessor::Remove(path);
    if (_result.HasError())
    {
        throw Exception(_result.GetError(), Text::Format("Failed to remove file: {}", path).c_str());
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルのコピー
 *  \param[in]  sourcePath              コピー元のパス
 *  \param[in]  destPath                コピー先のパス
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
void ThrowingUtility::Copy(const PathView &sourcePath, const PathView &destPath)
{
    _result = Accessor::Copy(sourcePath, destPath);
    if (_result.HasError())
    {
        throw Exception(_result.GetError(), Text::Format("Failed to copy file: {} -> {}", sourcePath, destPath).c_str());
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルの存在をチェック
 *  \param[in]  path                    チェックするファイルのパス
 *  \retval     true                    存在する
 *  \retval     false                   存在しない
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
bool ThrowingUtility::Exists(const PathView &path)
{
    auto exists = Accessor::Exists(_result, path);
    if (_result.HasError())
    {
        throw Exception(_result.GetError(), Text::Format("Failed to verify file existence: {}", path).c_str());
    }

    return exists;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      システム標準のファイルであるかをチェック
 *  \param[in]  path                    パス
 *  \retval     true                    システム標準のファイルである
 *  \retval     false                   システム標準のファイルではない
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
bool ThrowingUtility::IsSystemNativeFile(const PathView &path)
{
    const auto isSystemNativeFile = Accessor::IsSystemNativeFile(_result, path);
    if (_result.HasError())
    {
        throw Exception(_result.GetError(), Text::Format("Failed to verify the native file: {}", path).c_str());
    }

    return isSystemNativeFile;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウント
 *  \param[in]  mountName               マウント名
 *  \param[in]  path                    マウント先のパス
 *  \param[in]  accessType              アクセスタイプ
 *  \param[in]  delegateKey             デリゲートキー
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
void ThrowingUtility::Mount(const PathView &mountName,
                            const PathView &path,
                            MountAccessType accessType,
                            DelegateKey delegateKey)
{
    _result = Accessor::Mount(mountName, path, accessType, delegateKey);
    if (_result.HasError())
    {
        throw Exception(_result.GetError(), Text::Format("Failed to mouth: {}, path: {}", mountName, path).c_str());
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウント解除
 *  \param[in]  mountName               マウント名
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
void ThrowingUtility::Unmount(const PathView &mountName)
{
    _result = Accessor::Unmount(mountName);
    if (_result.HasError())
    {
        throw Exception(_result.GetError(), Text::Format("Failed to unmouth: {}", mountName).c_str());
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      再マウント
 *  \param[in]  mountName               マウント名
 *  \param[in]  path                    マウント先のパス
 *  \param[in]  accessType              アクセスタイプ
 *  \param[in]  delegateKey             デリゲートキー
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
void ThrowingUtility::Remount(const PathView &mountName,
                              const PathView &path,
                              MountAccessType accessType,
                              DelegateKey delegateKey)
{
    _result = Accessor::Remount(mountName, path, accessType, delegateKey);
    if (_result.HasError())
    {
        throw Exception(_result.GetError(), Text::Format("Failed to remount: {}, path: {}", mountName, path).c_str());
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウントされているかを取得
 *  \param[in]  mountName   マウント名
 *  \retval     true        マウントされている
 *  \retval     false       マウントされていない
 *  \note       この関数は例外を発生させず、File::IsMounted()と等価である。
 */
/* ------------------------------------------------------------------------- */
bool ThrowingUtility::IsMounted(const PathView &mountName) noexcept
{
    return Accessor::IsMounted(mountName);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウントパスからシステム標準のパスに変換
 *  \param[in]  path                    パス
 *  \return     システム標準のパス
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
STL::string ThrowingUtility::GetSystemNativePath(const PathView &path)
{
    auto filesystemPath = Accessor::GetSystemNativePath(_result, path);

    if (_result.HasError())
    {
        throw Exception(_result.GetError(), Text::Format("Failed to convert to the system native path: {}", path).c_str());
    }

    return filesystemPath;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デリゲートの削除
 *  \param[in]  key                     削除するデリゲートキー
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 *  \note       存在しないデリゲートキーに対しては何もしない
 */
/* ------------------------------------------------------------------------- */
void ThrowingUtility::RemoveDelegate(DelegateKey key)
{
    if (_result = Accessor::RemoveDelegate(key); _result.HasError())
    {
        throw Exception(
            _result.GetError(),
            Text::Format("Failed to remove delegate: {}", static_cast<uint32_t>(key)).c_str());
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デフォルトのデリゲートを設定
 *  \param[in]  key     デフォルトのデリゲートキー
 *  \note       この関数は例外を発生させず、File::SetDefaultDelegate()と等価である。
 */
/* ------------------------------------------------------------------------- */
void ThrowingUtility::SetDefaultDelegate(DelegateKey key) noexcept
{
    Accessor::SetDefaultDelegate(key);
}
}    // namespace MGL::File

// vim: et ts=4 sw=4 sts=4
