// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_throwing_handle.cc
 *  \brief      MGL 例外を発生させるファイルハンドル
 *  \date       Since: September 29, 2022. 16:07:16 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/file/mgl_file_throwing_handle.h>

#include <mgl/file/mgl_file_accessor.h>
#include <mgl/file/mgl_file_exception.h>
#include <mgl/text/mgl_text_format.h>

namespace MGL::File
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  path                    オープンするファイルのパス
 *  \param[in]  mode                    オープンモード
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
ThrowingHandle::ThrowingHandle(const PathView &path, OpenMode mode)
    : ThrowingHandle()
{
    Open(path, mode);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルをオープン
 *  \param[in]  path                    オープンするファイルのパス
 *  \param[in]  mode                    オープンモード
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
void ThrowingHandle::Open(const PathView &path, OpenMode mode)
{
    _work = Accessor::Open(_result, path, mode);

    if (_result.HasError())
    {
        throw Exception(_result.GetError(), Text::Format("Failed to open {}", path).c_str());
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルをクローズ
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
void ThrowingHandle::Close()
{
    if (IsOpen())
    {
        _result = Accessor::Close(_work);
        if (_result.HasError())
        {
            throw Exception(_result.GetError(), "Failed to close file.");
        }
    }
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルの読み込み
 *  \param[out] buffer                  読み込み先のバッファ
 *  \param[in]  size                    読み込むサイズ
 *  \return     実際に読み込んだサイズ
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
size_t ThrowingHandle::Read(void *buffer, size_t size)
{
    const auto readSize = Accessor::Read(_work, _result, buffer, size);

    if (_result.HasError())
    {
        throw Exception(_result.GetError(), "Failed to read file.");
    }

    return readSize;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルに書き込み
 *  \param[out] buffer                  書き込むデータを格納したバッファ
 *  \param[in]  size                    書き込むサイズ
 *  \return     実際に書き込んだサイズ
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
size_t ThrowingHandle::Write(const void *buffer, size_t size)
{
    const auto writeSize = Accessor::Write(_work, _result, buffer, size);

    if (_result.HasError())
    {
        throw Exception(_result.GetError(), "Failed to write file.");
    }

    return writeSize;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ストリーム位置を設定
 *  \param[in]  seekType                シークタイプ
 *  \param[in]  offset                  オフセット
 *  \return     設定後のストリーム位置
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
size_t ThrowingHandle::Seek(SeekType seekType, int32_t offset)
{
    const auto newOffset = Accessor::Seek(_work, _result, seekType, offset);

    if (_result.HasError())
    {
        throw Exception(_result.GetError(), "Failed to seek offset.");
    }

    return newOffset;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ストリーム位置を設定
 *  \param[in]  seekType                シークタイプ
 *  \param[in]  offset                  オフセット
 *  \return     設定後のストリーム位置
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
size_t ThrowingHandle::Seek(SeekType seekType, size_t offset)
{
    return Seek(seekType, static_cast<int32_t>(offset));
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ストリーム位置をスキップ
 *  \param[in]  size                    スキップするサイズ
 *  \return     スキップ後のストリーム位置
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
size_t ThrowingHandle::Skip(size_t size)
{
    return Seek(SeekType::Current, static_cast<int32_t>(size));
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ストリーム位置を取得
 *  \return     現在のストリーム位置
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
size_t ThrowingHandle::GetOffset()
{
    const auto offset = Accessor::GetOffset(_work, _result);

    if (_result.HasError())
    {
        throw Exception(_result.GetError(), "Failed to get offset.");
    }

    return offset;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルストリームが終端に達しているかを取得
 *  \retval     true                    終端に達している
 *  \retval     false                   終端に達していない
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
bool ThrowingHandle::IsEOF()
{
    const auto isEOF = Accessor::IsEOF(_work, _result);

    if (_result.HasError())
    {
        throw Exception(_result.GetError(), "Failed to get eof marker.");
    }

    return isEOF;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      オープンしているファイルのサイズを取得
 *  \return     ファイルサイズ
 *  \exception  MGL::File::Exception    エラーコードを保持した例外
 */
/* ------------------------------------------------------------------------- */
size_t ThrowingHandle::GetSize()
{
    const auto size = Accessor::GetSize(_work, _result);

    if (_result.HasError())
    {
        throw Exception(_result.GetError(), "Failed to get size.");
    }

    return size;
}
}    // namespace MGL::File
// vim: et ts=4 sw=4 sts=4
