// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_path_view.cc
 *  \brief      文字列の参照のみを行うファイルパスクラス
 *  \date       Since: July 30, 2023. 19:50:11 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/file/mgl_file_path_view.h>

#include <cstring>
#include <filesystem>

namespace MGL::File
{
namespace
{
//! デフォルトのアロケートサイズ
constexpr size_t kDefaultReserveSize = 512 - 32;
}    // namespace

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      要素の追加
 *  \param[in]  element     追加する要素
 *  \return     追加後のパス
 */
/* ------------------------------------------------------------------------- */
Path PathView::Append(const char *element) const noexcept
{
    Path path(kDefaultReserveSize);
    path.Concat(_path);
    return path.Append(element);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字列の追加
 *  \param[in]  string  追加する文字列
 *  \return     追加後のパス
 */
/* ------------------------------------------------------------------------- */
Path PathView::Concat(const char *string) const noexcept
{
    Path path(kDefaultReserveSize);
    path.Concat(_path);
    return path.Concat(string);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字の追加
 *  \param[in]  character   追加する文字
 *  \return     追加後のパス
 */
/* ------------------------------------------------------------------------- */
Path PathView::Concat(char character) const noexcept
{
    Path path(kDefaultReserveSize);
    path.Concat(_path);
    return path.Concat(character);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      パスの長さを取得
 *  \return     パスの長さ（終端記号を除くchar型の要素数）
 */
/* ------------------------------------------------------------------------- */
size_t PathView::GetLength() const noexcept
{
    if (_path == nullptr)
    {
        return 0;
    }

    return strlen(_path);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      パスが空であるかを取得
 *  \retval     true    空である
 *  \retval     false   空ではない
 */
/* ------------------------------------------------------------------------- */
bool PathView::IsEmpty() const noexcept
{
    if (_path == nullptr)
    {
        return true;
    }
    else if (_path[0] == '\0')
    {
        return true;
    }

    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ルートパスの取得
 *  \return     ルートパス．取得できない場合は空のパス
 */
/* ------------------------------------------------------------------------- */
Path PathView::GetRoot() const noexcept
{
    if (_path == nullptr)
    {
        return {};
    }

    Path path(kDefaultReserveSize);

    for (size_t i = 0;; i++)
    {
        auto c = _path[i];

        if ((c == '\0') || (c == '/'))
        {
            break;
        }

        path += c;
    }

    return path;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウント名の取得
 *  \return     マウント名．取得できない場合は空のパス
 */
/* ------------------------------------------------------------------------- */
Path PathView::GetMountName() const noexcept
{
    if (_path == nullptr)
    {
        return {};
    }

    if (_path[0] != '$')
    {
        return {};
    }

    Path path(kDefaultReserveSize);

    for (size_t i = 1;; i++)
    {
        auto c = _path[i];

        if ((c == '\0') || (c == '/'))
        {
            break;
        }

        path += c;
    }

    return path;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウント名からの相対パスを取得
 *  \return     相対パス．取得できない場合は空のパス
 */
/* ------------------------------------------------------------------------- */
Path PathView::GetRelativePath() const noexcept
{
    if (_path == nullptr)
    {
        return {};
    }

    if (_path[0] != '$')
    {
        return {_path};
    }

    size_t position = SIZE_MAX;
    for (size_t i = 1; _path[i] != '\0'; i++)
    {
        if (_path[i] == '/')
        {
            position = i + 1;
            break;
        }
    }

    if (position == SIZE_MAX)
    {
        return {};
    }

    Path path(kDefaultReserveSize);
    path.Concat(&_path[position]);

    return path;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      親ディレクトリのパスを取得
 *  \return     親ディレクトリのパス．取得できない場合は空のパス
 */
/* ------------------------------------------------------------------------- */
Path PathView::GetParent() const noexcept
{
    // 空のパスに対しては何もしない
    if (_path == nullptr)
    {
        return {};
    }

    // 最後の区切り文字の位置を取得
    size_t position = SIZE_MAX;
    for (auto i = static_cast<int32_t>(GetLength()); i >= 0; i--)
    {
        auto index = static_cast<size_t>(i);

        if (_path[index] == '/')
        {
            position = index;
            break;
        }
    }

    // 最後の区切り文字の位置が見つからない場合は失敗
    if (position == SIZE_MAX)
    {
        return {};
    }

    // 先頭から最後の区切り文字の前までをコピー
    Path path(kDefaultReserveSize);
    for (size_t i = 0; i < position; i++)
    {
        path += _path[i];
    }

    return path;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイル名の取得
 *  \return     ファイル名．取得できない場合は空のパス
 */
/* ------------------------------------------------------------------------- */
Path PathView::GetFilename() const noexcept
{
    // 空のパスに対しては何もしない
    if (_path == nullptr)
    {
        return {};
    }

    // 最後の区切り文字の位置を取得
    size_t position = SIZE_MAX;
    for (auto i = static_cast<int32_t>(GetLength()); i >= 0; i--)
    {
        auto index = static_cast<size_t>(i);

        if (_path[index] == '/')
        {
            position = index;
            break;
        }
    }

    // 最後の区切り文字の位置が見つからない場合はそのまま返す
    if (position == SIZE_MAX)
    {
        // パスがマウント名のみの場合は失敗とする
        if (_path[0] == '$')
        {
            return {};
        }

        return {_path};
    }

    // 最後の区切り文字から最後までをコピー
    Path path(kDefaultReserveSize);
    path.Concat(&_path[position + 1]);

    return path;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      拡張子を除いたファイル名を取得
 *  \return     拡張子を除いたファイル名．取得できない場合は空のパス
 */
/* ------------------------------------------------------------------------- */
Path PathView::GetStem() const noexcept
{
    // 空のパスに対しては何もしない
    if (_path == nullptr)
    {
        return {};
    }

    auto length = GetLength();

    // 最後の区切り文字の位置を取得
    size_t startPosition = SIZE_MAX;
    for (auto i = static_cast<int32_t>(length); i >= 0; i--)
    {
        auto index = static_cast<size_t>(i);

        if (_path[index] == '/')
        {
            startPosition = index + 1;
            break;
        }
    }

    // 最後の区切り文字の位置が見つからない場合は先頭から
    if (startPosition == SIZE_MAX)
    {
        // パスがマウント名のみの場合は失敗とする
        if (_path[0] == '$')
        {
            return {};
        }

        startPosition = 0;
    }

    // 最後の'.'の位置を取得
    size_t endPosition = SIZE_MAX;
    for (size_t i = length; i != 0; i--)
    {
        if (_path[i] == '/')
        {
            break;
        }
        else if (_path[i] == '.')
        {
            endPosition = i;
            break;
        }
    }

    // 最後の'.'が見つからなかったり，名前が'.'で始まるファイルの場合は最後までをコピーする
    if ((endPosition == SIZE_MAX) || (startPosition == endPosition))
    {
        endPosition = length;
    }

    // 開始点から終了点までをコピー
    Path path(kDefaultReserveSize);
    for (size_t i = startPosition; i < endPosition; i++)
    {
        path += _path[i];
    }

    return path;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      拡張子を取得
 *  \return     拡張子．取得できない場合は空のパス
 */
/* ------------------------------------------------------------------------- */
Path PathView::GetExtension() const noexcept
{
    // 空のパスに対しては何もしない
    if (_path == nullptr)
    {
        return {};
    }

    auto length = GetLength();

    // 最後の'.'の位置を取得
    size_t position = SIZE_MAX;
    for (auto i = static_cast<int32_t>(length); i >= 0; i--)
    {
        auto index = static_cast<size_t>(i);

        if (_path[index] == '/')
        {
            break;
        }
        else if (_path[index] == '.')
        {
            position = index;
            break;
        }
    }

    // 最後の区切り文字の位置が見つからない場合は失敗
    if (position == SIZE_MAX)
    {
        return {};
    }

    // パスの区切り文字の直後に'.'がある場合は拡張子ではない
    if (position >= 1)
    {
        if (_path[position - 1] == '/')
        {
            return {};
        }
    }

    // 最後の区切り文字から最後までをコピー
    Path path(kDefaultReserveSize);
    path.Concat(&_path[position]);

    return path;
}
}    // namespace MGL::File
// vim: et ts=4 sw=4 sts=4
