// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_path.cc
 *  \brief      MGL ファイルパスクラス
 *  \date       Since: July 26, 2023. 15:55:02 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/file/mgl_file_path.h>
#include <mgl/file/mgl_file_path_view.h>

namespace MGL::File
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  path    設定するパスの文字列
 */
/* ------------------------------------------------------------------------- */
Path::Path(const char *path) noexcept
    : _path(path)
{
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  reserveSize     予約サイズ
 */
/* ------------------------------------------------------------------------- */
Path::Path(size_t reserveSize) noexcept
{
    if (reserveSize != 0)
    {
        _path.reserve(reserveSize);
    }
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  path            設定するパスの文字列
 *  \param[in]  reserveSize     予約サイズ
 */
/* ------------------------------------------------------------------------- */
Path::Path(const char *path, size_t reserveSize) noexcept
{
    _path.reserve(reserveSize);

    _path += path;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      要素の追加
 *  \param[in]  element     追加する要素
 *  \return     追加後のパス
 */
/* ------------------------------------------------------------------------- */
Path &Path::Append(const char *element) noexcept
{
    if (_path.back() != '/')
    {
        _path += '/';
    }

    _path += element;

    return *this;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字列の追加
 *  \param[in]  string  追加する文字列
 *  \return     追加後のパス
 */
/* ------------------------------------------------------------------------- */
Path &Path::Concat(const char *string) noexcept
{
    _path += string;

    return *this;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      文字の追加
 *  \param[in]  character   追加する文字
 *  \return     追加後のパス
 */
/* ------------------------------------------------------------------------- */
Path &Path::Concat(char character) noexcept
{
    _path += character;

    return *this;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ルートパスの取得
 *  \return     ルートパス．取得できない場合は空のパス
 */
/* ------------------------------------------------------------------------- */
Path Path::GetRoot() const noexcept
{
    return PathView(_path.c_str()).GetRoot();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウント名の取得
 *  \return     マウント名．取得できない場合は空のパス
 */
/* ------------------------------------------------------------------------- */
Path Path::GetMountName() const noexcept
{
    return PathView(_path.c_str()).GetMountName();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ルートまたはマウントからの相対パスを取得
 *  \return     相対パス．取得できない場合は空のパス
 */
/* ------------------------------------------------------------------------- */
Path Path::GetRelativePath() const noexcept
{
    return PathView(_path.c_str()).GetRelativePath();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      親ディレクトリのパスを取得
 *  \return     親ディレクトリのパス．取得できない場合は空のパス
 */
/* ------------------------------------------------------------------------- */
Path Path::GetParent() const noexcept
{
    return PathView(_path.c_str()).GetParent();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイル名の取得
 *  \return     ファイル名．取得できない場合は空のパス
 */
/* ------------------------------------------------------------------------- */
Path Path::GetFilename() const noexcept
{
    return PathView(_path.c_str()).GetFilename();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      拡張子を除いたファイル名を取得
 *  \return     拡張子を除いたファイル名．取得できない場合は空のパス
 */
/* ------------------------------------------------------------------------- */
Path Path::GetStem() const noexcept
{
    return PathView(_path.c_str()).GetStem();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      拡張子を取得
 *  \return     拡張子．取得できない場合は空のパス
 */
/* ------------------------------------------------------------------------- */
Path Path::GetExtension() const noexcept
{
    return PathView(_path.c_str()).GetExtension();
}
}    // namespace MGL::File

// vim: et ts=4 sw=4 sts=4
