// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_mounter.cc
 *  \brief      MGL ファイルマウンタ
 *  \date       Since: January 20, 2021. 10:25:17 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/file/mgl_file_mounter.h>

namespace MGL::File
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      インスタンスの取得
 *  \return     インスタンスの参照
 */
/* ------------------------------------------------------------------------- */
STL::unique_ptr<Mounter> &Mounter::GetInstanceRef() noexcept
{
    static STL::unique_ptr<Mounter> sInstance = nullptr;
    return sInstance;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウント
 *  \param[in]  mountName   マウント名
 *  \param[in]  path        マウント先のパス
 *  \param[in]  accessType  アクセスタイプ
 *  \param[in]  delegateKey デリゲートキー
 *  \return     エラー発生時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result Mounter::Mount(const PathView &mountName, const PathView &path, MountAccessType accessType, DelegateKey delegateKey) noexcept
{
    const std::scoped_lock lock(_mountMutex);

    // 不正な引数のチェック
    if ((mountName == nullptr) || (path == nullptr))
    {
        return Error::InvalidArgument;
    }

    // デフォルトのデリゲートキーが設定されている場合，引数でkDefaultDelegateKeyを受けた際にそのキーで代替する
    if ((_defaultDelegateKey != kDefaultDelegateKey) && (delegateKey == kDefaultDelegateKey))
    {
        delegateKey = _defaultDelegateKey;
    }

    // 既にマウントされている場合は失敗
    const auto it = _mountProperties.find(mountName.GetCString());
    if (it != _mountProperties.end())
    {
        return Error::AlreadyMounted;
    }

    // デリゲートを取得
    auto delegate = GetDelegate(delegateKey);
    if (delegate == nullptr)
    {
        return Error::DelegateNotExist;
    }

    // デリゲートが書き込みに対応しておらず，書き込み可能なアクセスタイプでマウントしている場合は失敗
    if ((accessType == MountAccessType::Writable) && !delegate->IsWritable())
    {
        return Error::MountNotWritable;
    }

    // マウント情報を生成して設定
    auto property = STL::make_shared<MountProperty>();
    property->name.Set(mountName);
    property->path.Set(path);
    property->accessType = accessType;
    property->delegate = std::move(delegate);

    // ディレクトリをマウントした場合，パスの末尾には'/'を付与する
    if (property->delegate->IsManagedSystemNativeFile())
    {
        property->path /= "";
    }

    // デリゲートのマウント処理を呼ぶ
    if (auto result = property->delegate->Mount(property->mountWork, property->path, property->accessType); result.HasError())
    {
        return result;
    }

    _mountProperties.emplace(mountName, std::move(property));

    return Result::Succeeded();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウント解除
 *  \param[in]  mountName   マウント名
 *  \return     エラー発生時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result Mounter::Unmount(const PathView &mountName) noexcept
{
    const std::scoped_lock lock(_mountMutex);

    const auto it = _mountProperties.find(mountName.GetCString());
    if (it == _mountProperties.end())
    {
        return Error::NotMounted;
    }

    it->second->delegate->Unmount(it->second->mountWork);

    _mountProperties.erase(it);

    return Result::Succeeded();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウント情報を取得
 *  \param[in]  mountName   マウント名
 *  \return     マウント情報．見つからない場合はnullptr
 */
/* ------------------------------------------------------------------------- */
Mounter::SharedMountProperty Mounter::Get(const PathView &mountName) noexcept
{
    const std::scoped_lock lock(_mountMutex);

    auto it = _mountProperties.find(mountName.GetCString());
    if (it == _mountProperties.end())
    {
        return nullptr;
    }

    return it->second;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウントされているかを取得
 *  \param[in]  mountName   マウント名
 *  \retval     true        マウントされている
 *  \retval     false       マウントされていない
 */
/* ------------------------------------------------------------------------- */
bool Mounter::IsMounted(const PathView &mountName) noexcept
{
    const std::scoped_lock lock(_mountMutex);

    const auto it = _mountProperties.find(mountName.GetCString());
    return it != _mountProperties.end();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デリゲートを追加
 *  \param[in]  key         デリゲートキー
 *  \param[in]  delegate    追加するデリゲート
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result Mounter::AddDelegate(DelegateKey key, const SharedDelegate &delegate) noexcept
{
    // kDefaultDelegateKeyで予約されたキーは使用できない
    if (key == kDefaultDelegateKey)
    {
        return Error::InvalidArgument;
    }

    const std::scoped_lock lock(_delegateMutex);

    // 既に存在している場合は失敗
    if (_delegates.find(key) != _delegates.end())
    {
        return Error::DelegateAlreadyExist;
    }

    _delegates.emplace(key, delegate);

    return Result::Succeeded();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デリゲートを削除
 *  \param[in]  key         デリゲートキー
 *  \return     失敗時にはいずれかのエラーが設定される．存在しないデリゲートキーに対しては何もしない
 */
/* ------------------------------------------------------------------------- */
Result Mounter::RemoveDelegate(DelegateKey key) noexcept
{
    const std::scoped_lock lock(_delegateMutex);

    // デリゲートを検索して削除
    const auto it = _delegates.find(key);
    if (it == _delegates.end())
    {
        return Result::Succeeded();
    }

    _delegates.erase(it);

    // 削除したデリゲートがデフォルトに設定されていた場合，デフォルト設定を解除する
    if (key == _defaultDelegateKey)
    {
        _defaultDelegateKey = kDefaultDelegateKey;
    }

    return Result::Succeeded();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デリゲートを取得
 *  \param[in]  key         デリゲートキー
 *  \return     キーに対応したデリゲート．失敗時にはnullptr
 */
/* ------------------------------------------------------------------------- */
SharedDelegate Mounter::GetDelegate(DelegateKey key) noexcept
{
    // デフォルトのデリゲートキーが設定されている場合，引数でkDefaultDelegateKeyを受けた際にそのキーで代替する
    if ((_defaultDelegateKey != kDefaultDelegateKey) && (key == kDefaultDelegateKey))
    {
        key = _defaultDelegateKey;
    }

    const std::scoped_lock lock(_delegateMutex);

    const auto it = _delegates.find(key);
    if (it == _delegates.end())
    {
        return nullptr;
    }

    return it->second;
}
}    // namespace MGL::File

// vim: et ts=4 sw=4 sts=4
