// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_handle.cc
 *  \brief      MGL ファイルハンドル
 *  \date       Since: September 29, 2022. 15:59:23 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/file/mgl_file_handle.h>

#include <mgl/file/mgl_file_accessor.h>

namespace MGL::File
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      コンストラクタ
 *  \param[in]  path    オープンするファイルのパス
 *  \param[in]  mode    オープンモード
 */
/* ------------------------------------------------------------------------- */
Handle::Handle(const PathView &path, OpenMode mode) noexcept
    : Handle()
{
    _work = Accessor::Open(_result, path, mode);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルをオープン
 *  \param[in]  path    オープンするファイルのパス
 *  \param[in]  mode    オープンモード
 *  \return     処理結果
 */
/* ------------------------------------------------------------------------- */
Result Handle::Open(const PathView &path, OpenMode mode) noexcept
{
    _work = Accessor::Open(_result, path, mode);

    return _result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルをクローズ
 *  \return     処理結果
 */
/* ------------------------------------------------------------------------- */
Result Handle::Close() noexcept
{
    if (!IsOpen())
    {
        return Result::Succeeded();
    }

    _result = Accessor::Close(_work);

    return _result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルの読み込み
 *  \param[out] buffer  読み込み先のバッファ
 *  \param[in]  size    読み込むサイズ
 *  \return     実際に読み込んだサイズ．失敗時はいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
size_t Handle::Read(void *buffer, size_t size) noexcept
{
    return Accessor::Read(_work, _result, buffer, size);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルに書き込み
 *  \param[out] buffer  書き込むデータを格納したバッファ
 *  \param[in]  size    書き込むサイズ
 *  \return     実際に書き込んだサイズ．失敗時はいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
size_t Handle::Write(const void *buffer, size_t size) noexcept
{
    return Accessor::Write(_work, _result, buffer, size);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ストリーム位置を設定
 *  \param[in]  seekType    シークタイプ
 *  \param[in]  offset      オフセット
 *  \return     設定後のストリーム位置．失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
size_t Handle::Seek(SeekType seekType, int32_t offset) noexcept
{
    return Accessor::Seek(_work, _result, seekType, offset);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ストリーム位置を設定
 *  \param[in]  seekType    シークタイプ
 *  \param[in]  offset      オフセット
 *  \return     設定後のストリーム位置．失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
size_t Handle::Seek(SeekType seekType, size_t offset) noexcept
{
    return Seek(seekType, static_cast<int32_t>(offset));
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ストリーム位置をスキップ
 *  \param[in]  size    スキップするサイズ
 *  \return     スキップ後のストリーム位置．失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
size_t Handle::Skip(size_t size) noexcept
{
    return Seek(SeekType::Current, static_cast<int32_t>(size));
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ストリーム位置を取得
 *  \return     現在のストリーム位置．失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
size_t Handle::GetOffset() noexcept
{
    return Accessor::GetOffset(_work, _result);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルストリームが終端に達しているかを取得
 *  \retval     true    終端に達している
 *  \retval     false   終端に達していない
 *  \note       失敗時には戻り値にいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
bool Handle::IsEOF() noexcept
{
    return Accessor::IsEOF(_work, _result);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      オープンしているファイルのサイズを取得
 *  \return     ファイルサイズ．失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
size_t Handle::GetSize() noexcept
{
    return Accessor::GetSize(_work, _result);
}
}    // namespace MGL::File
// vim: et ts=4 sw=4 sts=4
