// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_accessor.cc
 *  \brief      MGL ファイルアクセサ
 *  \date       Since: January 20, 2021. 11:21:51 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/file/mgl_file_accessor.h>

namespace MGL::File::Accessor
{
namespace
{
//! コピーに使用するバッファのサイズ
constexpr size_t kCopyBufferSize = static_cast<size_t>(1) * 1024 * 1024;

//! ファイルパスを格納するためのアロケートサイズ
constexpr size_t kPathAllocateSize = 512 - 32;

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      パスからマウント情報とデリゲート用のパスを取得
 *  \param[out] property        マウント情報の格納先
 *  \param[out] delegatePath    使用するデリゲート用のパスの格納先
 *  \param[in]  path            入力パス
 *  \return     失敗時にはいずれかのエラーが格納される
 */
/* ------------------------------------------------------------------------- */
Result GetMountProperty(Mounter::MountProperty &property, Path &delegatePath, const PathView &path) noexcept
{
    auto &mounter = Mounter::GetInstance();

    // 入力パスからマウント名とパスを取得
    auto mountName = path.GetMountName();
    auto filePath = path.GetRelativePath();

    if (mountName.IsEmpty())
    {
        property.delegate = mounter.GetDelegate();
        if (property.delegate == nullptr)
        {
            return Error::InvalidDelegate;
        }

        property.name.Clear();
        property.path.Set(path);
        property.accessType = MountAccessType::Writable;
        delegatePath.Set(property.path);

        return Result::Succeeded();
    }

    // マウント名からマウント情報を取得
    auto mountProperty = mounter.Get(mountName);
    if (mountProperty == nullptr)
    {
        return Error::NotMounted;
    }
    if (mountProperty->delegate == nullptr)
    {
        return Error::InvalidDelegate;
    }

    // ファイルを開くためのパスを設定
    if (mountProperty->delegate->IsManagedSystemNativeFile())
    {
        delegatePath.Set(mountProperty->path);
        delegatePath /= filePath;
    }
    else
    {
        delegatePath.Set(filePath);
    }

    property = *mountProperty;

    return Result::Succeeded();
}
}    // namespace


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルをオープン
 *  \param[out] result  処理結果
 *  \param[in]  path    オープンするファイルのパス
 *  \param[in]  mode    オープンモード
 *  \return     オープンしたファイルのワーク．失敗時にはnullptrが返り，resultにエラーが設定される
 */
/* ------------------------------------------------------------------------- */
AccessWorkPtr Open(Result &result, const PathView &path, OpenMode mode) noexcept
{
    // OpenMode::Noneは指定できない
    if (mode == OpenMode::None)
    {
        result = Error::InvalidArgument;
        return nullptr;
    }

    // マウント情報を取得
    Mounter::MountProperty mountProperty;
    Path delegatePath(kPathAllocateSize);
    result = GetMountProperty(mountProperty, delegatePath, path);
    if (result.HasError())
    {
        return nullptr;
    }

    // 読み込み専用マウントに対してライトアクセスはできない
    if ((mountProperty.accessType == MountAccessType::ReadOnly) && (mode == OpenMode::Write))
    {
        result = Error::MountNotWritable;
        return nullptr;
    }

    // ファイルをオープン
    auto work = mountProperty.delegate->Open(result, mountProperty.mountWork, delegatePath, mode);
    if (work == nullptr)
    {
        return nullptr;
    }
    work->SetDelegate(mountProperty.delegate);

    return work;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルをクローズ
 *  \param[in,out]  work    ファイルアクセス用のワーク
 *  \return         失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result Close(AccessWorkPtr &work) noexcept
{
    // 引数が不正なら失敗
    if (work == nullptr)
    {
        return Error::InvalidArgument;
    }

    // オープンされていないワークは扱わない
    if (!work->IsOpen())
    {
        return Error::NotOpened;
    }

    // デリゲートが不正なら失敗
    if (work->GetDelegate() == nullptr)
    {
        return Error::InvalidDelegate;
    }

    // デリゲートを用いてクローズ
    auto result = work->GetDelegate()->Close(work.get());
    if (result.HasError())
    {
        return result;
    }

    // アクセスワークを解放
    work.reset();
    return Result::Succeeded();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルを読み込み
 *  \param[in,out]  work    ファイルアクセス用のワーク
 *  \param[out]     result  処理結果
 *  \param[out]     buffer  読み込み先のバッファ
 *  \param[in]      size    読み込むサイズ
 *  \return         実際に読み込んだサイズ．失敗時はいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
size_t Read(AccessWorkPtr &work, Result &result, void *buffer, size_t size) noexcept
{
    // 引数が不正なら失敗
    if (work == nullptr)
    {
        result = Error::InvalidArgument;
        return 0;
    }

    // 読み込み用にオープンされていなければ失敗
    if (!work->IsReadable())
    {
        result = Error::NotReadableMode;
        return 0;
    }

    // デリゲートが不正なら失敗
    if (work->GetDelegate() == nullptr)
    {
        result = Error::InvalidDelegate;
        return 0;
    }

    // デリゲートを用いてファイルを読み込み
    return work->GetDelegate()->Read(work.get(), result, buffer, size);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルに書き込み
 *  \param[in,out]  work    ファイルアクセス用のワーク
 *  \param[out]     result  処理結果
 *  \param[out]     buffer  書き込むデータを格納したバッファ
 *  \param[in]      size    書き込むサイズ
 *  \return         実際に書き込んだサイズ．失敗時はいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
size_t Write(AccessWorkPtr &work, Result &result, const void *buffer, size_t size) noexcept
{
    // 引数が不正なら失敗
    if (work == nullptr)
    {
        result = Error::InvalidArgument;
        return 0;
    }

    // 書き込み用にオープンされていなければ失敗
    if (!work->IsWritable())
    {
        result = Error::NotWritableMode;
        return 0;
    }

    // デリゲートが不正なら失敗
    if (work->GetDelegate() == nullptr)
    {
        result = Error::InvalidDelegate;
        return 0;
    }

    // デリゲートを用いてファイルを読み込み
    return work->GetDelegate()->Write(work.get(), result, buffer, size);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ストリーム位置を設定
 *  \param[in,out]  work        アクセスするファイルのワーク
 *  \param[out]     result      処理結果
 *  \param[in]      seekType    シークタイプ
 *  \param[in]      offset      オフセット
 *  \return         設定後のストリーム位置．失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
size_t Seek(AccessWorkPtr &work, Result &result, SeekType seekType, int32_t offset) noexcept
{
    // 引数が不正なら失敗
    if (work == nullptr)
    {
        result = Error::InvalidArgument;
        return 0;
    }

    // オープンされていないワークは扱わない
    if (!work->IsOpen())
    {
        result = Error::NotOpened;
        return 0;
    }

    // 絶対位置のシークに対して負数のオフセットは指定できない
    if ((seekType != SeekType::Current) && (offset < 0))
    {
        result = Error::InvalidArgument;
        return 0;
    }

    // デリゲートが不正なら失敗
    if (work->GetDelegate() == nullptr)
    {
        result = Error::InvalidDelegate;
        return 0;
    }

    // デリゲートを用いてシーク
    return work->GetDelegate()->Seek(work.get(), result, seekType, offset);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ストリーム位置を取得
 *  \param[in,out]  work    アクセスするファイルのワーク
 *  \param[out]     result  処理結果
 *  \return         現在のストリーム位置．失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
size_t GetOffset(AccessWorkPtr &work, Result &result) noexcept
{
    // 引数が不正なら失敗
    if (work == nullptr)
    {
        result = Error::InvalidArgument;
        return 0;
    }

    // オープンされていないワークは扱わない
    if (!work->IsOpen())
    {
        result = Error::NotOpened;
        return 0;
    }

    // デリゲートが不正なら失敗
    if (work->GetDelegate() == nullptr)
    {
        result = Error::InvalidDelegate;
        return 0;
    }

    // デリゲートを用いて取得
    return work->GetDelegate()->GetOffset(work.get(), result);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルストリームが終端に達しているかを取得
 *  \param[in,out]  work    アクセスするファイルのワーク
 *  \param[out]     result  処理結果
 *  \retval         true    終端に達している
 *  \retval         false   終端に達していない
 *  \note           失敗時には戻り値にいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
bool IsEOF(AccessWorkPtr &work, Result &result) noexcept
{
    // 引数が不正なら失敗
    if (work == nullptr)
    {
        result = Error::InvalidArgument;
        return true;
    }

    // オープンされていないワークは扱わない
    if (!work->IsOpen())
    {
        result = Error::NotOpened;
        return true;
    }

    // デリゲートが不正なら失敗
    if (work->GetDelegate() == nullptr)
    {
        result = Error::InvalidDelegate;
        return true;
    }

    // デリゲートを用いて取得
    return work->GetDelegate()->IsEOF(work.get(), result);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          オープンしているファイルのサイズを取得
 *  \param[in,out]  work    アクセスするファイルのワーク
 *  \param[out]     result  処理結果
 *  \return         ファイルサイズ．失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
size_t GetSize(AccessWorkPtr &work, Result &result) noexcept
{
    // 引数が不正なら失敗
    if (work == nullptr)
    {
        result = Error::InvalidArgument;
        return 0;
    }

    // オープンされていないワークは扱わない
    if (!work->IsOpen())
    {
        result = Error::NotOpened;
        return 0;
    }

    // デリゲートが不正なら失敗
    if (work->GetDelegate() == nullptr)
    {
        result = Error::InvalidDelegate;
        return 0;
    }

    // デリゲートを用いて取得
    return work->GetDelegate()->GetSize(work.get(), result);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルのサイズを取得
 *  \param[out] result  処理結果
 *  \param[in]  path    サイズを取得するファイルのパス
 *  \return     ファイルサイズ．失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
size_t GetSize(Result &result, const PathView &path) noexcept
{
    // マウント情報を取得
    Mounter::MountProperty mountProperty;
    Path delegatePath(kPathAllocateSize);
    result = GetMountProperty(mountProperty, delegatePath, path);
    if (result.HasError())
    {
        return 0;
    }

    // デリゲートを用いてディレクトリを作成
    auto size = mountProperty.delegate->GetSize(mountProperty.mountWork.get(), result, delegatePath);
    if (result.HasError())
    {
        return 0;
    }

    return size;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ディレクトリを作成する
 *  \param[in]  path    作成するディレクトリのパス
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result MakeDirectory(const PathView &path) noexcept
{
    // マウント情報を取得
    Mounter::MountProperty mountProperty;
    Path delegatePath(kPathAllocateSize);
    auto result = GetMountProperty(mountProperty, delegatePath, path);
    if (result.HasError())
    {
        return result;
    }

    // デリゲートを用いてディレクトリを作成
    return mountProperty.delegate->MakeDirectory(mountProperty.mountWork.get(), delegatePath);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルの移動・リネーム
 *  \param[in]  sourcePath  移動元のパス
 *  \param[in]  destPath    移動先のパス
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result Move(const PathView &sourcePath, const PathView &destPath) noexcept
{
    // 移動先のマウント情報を取得
    Mounter::MountProperty destMountProperty;
    Path destDelegatePath(kPathAllocateSize);
    if (auto result = GetMountProperty(destMountProperty, destDelegatePath, destPath); result.HasError())
    {
        return result;
    }

    // 移動先が書き込みできなければ失敗
    if (destMountProperty.accessType == MountAccessType::ReadOnly)
    {
        return Error::MountNotWritable;
    }

    // 移動元のマウント情報を取得
    Mounter::MountProperty srcMountProperty;
    Path srcDelegatePath(kPathAllocateSize);
    if (auto result = GetMountProperty(srcMountProperty, srcDelegatePath, sourcePath); result.HasError())
    {
        return result;
    }

    // 移動元と移動先のデリゲートとマウントが同じならデリゲートのMove()を呼ぶ
    if ((srcMountProperty.delegate == destMountProperty.delegate) && (srcMountProperty.mountWork == destMountProperty.mountWork))
    {
        return srcMountProperty.delegate->Move(srcMountProperty.mountWork.get(), srcDelegatePath, destDelegatePath);
    }

    // 移動元と移動先が異なる場合はコピーした後に移動元を削除する
    if (auto result = Copy(sourcePath, destPath); result.HasError())
    {
        return result;
    }

    return Remove(sourcePath);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルの削除
 *  \param[in]  path    削除するファイルのパス
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result Remove(const PathView &path) noexcept
{
    // マウント情報を取得
    Mounter::MountProperty mountProperty;
    Path delegatePath(kPathAllocateSize);
    if (auto result = GetMountProperty(mountProperty, delegatePath, path); result.HasError())
    {
        return result;
    }

    // 書き込み不可能であれば失敗
    if (mountProperty.accessType == MountAccessType::ReadOnly)
    {
        return Error::MountNotWritable;
    }

    // 削除
    return mountProperty.delegate->Remove(mountProperty.mountWork.get(), delegatePath);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルのコピー
 *  \param[in]  sourcePath  コピー元のパス
 *  \param[in]  destPath    コピー先のパス
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result Copy(const PathView &sourcePath, const PathView &destPath) noexcept
{
    Result result;

    // コピー元のファイルをオープン
    auto srcFile = Open(result, sourcePath, OpenMode::Read);
    if (srcFile == nullptr)
    {
        return result;
    }

    // コピー先のファイルをオープン
    auto destFile = Open(result, destPath, OpenMode::Write);
    if (destFile == nullptr)
    {
        return result;
    }

    // コピー用のバッファを確保
    auto copyBuffer = STL::make_unique<std::byte[]>(kCopyBufferSize);

    // ファイルの終端に達するまで読み書きを繰り返す
    for (;;)
    {
        // 終端に達していたらループから抜ける
        if (auto isEOF = IsEOF(srcFile, result); isEOF || result.HasError())
        {
            break;
        }

        // コピー元からファイルの内容を読み込み
        auto readSize = Read(srcFile, result, copyBuffer.get(), kCopyBufferSize);
        if (result.HasError())
        {
            return result;
        }

        // 読み込んだサイズ分だけコピー先のファイルに書き込み
        Write(destFile, result, copyBuffer.get(), readSize);
        if (result.HasError())
        {
            return result;
        }
    }

    return result;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ファイルの存在をチェック
 *  \param[out] result      処理結果
 *  \param[in]  path        チェックするファイルのパス
 *  \retval     true        存在する
 *  \retval     false       存在しない
 *  \note       失敗時には戻り値にいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
bool Exists(Result &result, const PathView &path) noexcept
{
    // マウント情報を取得
    Mounter::MountProperty mountProperty;
    Path delegatePath(kPathAllocateSize);
    result = GetMountProperty(mountProperty, delegatePath, path);
    if (result.HasError())
    {
        return false;
    }

    return mountProperty.delegate->Exists(mountProperty.mountWork.get(), result, delegatePath);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      パスがシステム標準のファイルかを取得
 *  \param[out] result      処理結果
 *  \param[in]  path        パス
 *  \retval     true        システム標準のファイルである
 *  \retval     false       システム標準のファイルではない
 *  \note       失敗時には戻り値にいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
bool IsSystemNativeFile(Result &result, const PathView &path) noexcept
{
    // 入力パスからマウント名とパスを取得
    auto mountName = path.GetMountName();

    // マウント名からマウント情報を取得
    auto property = Mounter::GetInstance().Get(mountName);
    if (property == nullptr)
    {
        result = Error::NotMounted;
        return false;
    }

    // デリゲートが無効なら失敗
    if (property->delegate == nullptr)
    {
        result = Error::InvalidDelegate;
        return false;
    }

    result.Success();
    return property->delegate->IsManagedSystemNativeFile();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウント
 *  \param[in]  mountName   マウント名
 *  \param[in]  path        マウント先のパス
 *  \param[in]  accessType  アクセスタイプ
 *  \param[in]  delegateKey デリゲートキー
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result Mount(const PathView &mountName, const PathView &path, MountAccessType accessType, DelegateKey delegateKey) noexcept
{
    Result result;
    auto filesystemPath = GetSystemNativePath(result, path);
    if (result.HasError())
    {
        return result;
    }

    return Mounter::GetInstance().Mount(mountName, filesystemPath.c_str(), accessType, delegateKey);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウント解除
 *  \param[in]  mountName   マウント名
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result Unmount(const PathView &mountName) noexcept
{
    return Mounter::GetInstance().Unmount(mountName);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      再マウント
 *  \param[in]  mountName   マウント名
 *  \param[in]  path        マウント先のパス
 *  \param[in]  accessType  アクセスタイプ
 *  \param[in]  delegateKey デリゲートキー
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result Remount(const PathView &mountName, const PathView &path, MountAccessType accessType, DelegateKey delegateKey) noexcept
{
    Unmount(mountName);

    return Mount(mountName, path, accessType, delegateKey);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウントパスからシステム標準のパスに変換
 *  \param[out] result  処理結果
 *  \param[in]  path    パス
 *  \return     システム標準のパス
 */
/* ------------------------------------------------------------------------- */
STL::string GetSystemNativePath(Result &result, const PathView &path) noexcept
{
    // 入力パスからマウント名とパスを取得
    auto mountName = path.GetMountName();
    auto filePath = path.GetRelativePath();

    // マウント名が無ければ入力をそのまま返す
    if (mountName.IsEmpty())
    {
        result.Success();
        return {filePath.GetCString()};
    }

    // マウント名からマウント情報を取得
    auto property = Mounter::GetInstance().Get(mountName);
    if (property == nullptr)
    {
        result = Error::NotMounted;
        return {};
    }

    // デリゲートが無効なら失敗
    if (property->delegate == nullptr)
    {
        result = Error::InvalidDelegate;
        return {};
    }

    // システム標準のファイルを扱わないデリゲートからはパスを取得できない
    if (!property->delegate->IsManagedSystemNativeFile())
    {
        result = Error::NoSystemNativeFile;
        return {};
    }

    result.Success();
    Path nativePath(property->path, kPathAllocateSize);
    nativePath /= filePath;
    return {nativePath.GetCString()};
}
}    // namespace MGL::File::Accessor

// vim: et ts=4 sw=4 sts=4
