// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_delegate_win32.cc
 *  \brief      MGL Win32用ファイルデリゲート
 *  \date       Since: March 27, 2021. 4:25:48 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#include <mgl/file/delegate/mgl_file_delegate_win32.h>
#if defined(MGL_FILE_DELEGATE_ENABLE_WIN32)

#include <Windows.h>
#include <atlstr.h>

namespace MGL::File
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルをオープン
 *  \param[out]     result      処理結果
 *  \param[in,out]  mountWork   マウント毎のワーク
 *  \param[in]      path        オープンするファイルのパス
 *  \param[in]      mode        オープンモード
 *  \return         オープンしたファイルのワーク
 */
/* ------------------------------------------------------------------------- */
AccessWorkPtr Win32Delegate::Open(Result &result, [[maybe_unused]] SharedMountWork &mountWork, const PathView &path, OpenMode mode) noexcept
{
    FILE *fp = nullptr;
    auto error = _wfopen_s(&fp, ConvertPath(path).c_str(), GetModeString(mode));
    if (error != 0)
    {
        result = GetError(error);
        return nullptr;
    }

    auto work = STL::make_unique<Win32Work>(mode);
    work->fp = fp;
    
    work->size = GetSize(work.get(), result);
    if (result.HasError())
    {
        return nullptr;
    }
    
    result.Success();

    return work;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルをクローズ
 *  \param[in,out]  work    クローズするファイルのワーク
 *  \return         エラー発生時にいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result Win32Delegate::Close(AccessWork *work) noexcept
{
    auto *win32Work = GetWork(work);
    if (win32Work == nullptr)
    {
        return Error::InvalidArgument;
    }

    if (fclose(win32Work->fp) != 0)
    {
        auto result = GetError(errno);
        return result;
    }
    win32Work->fp = nullptr;
    
    return Result::Succeeded();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルを読み込み
 *  \param[in,out]  work        アクセスするファイルのワーク
 *  \param[out]     result      処理結果
 *  \param[in]      buffer      読み込み先のアドレス
 *  \param[in]      size        読み込みサイズ
 *  \return         実際に読み込んだバイト数
 */
/* ------------------------------------------------------------------------- */
size_t Win32Delegate::Read(AccessWork *work, Result &result, void *buffer, size_t size) noexcept
{
    auto *win32Work = GetWork(work, OpenMode::Read);
    if (win32Work == nullptr)
    {
        result = Error::InvalidArgument;
        return 0;
    }

    clearerr(win32Work->fp);
    auto readSize = fread(buffer, 1, size, win32Work->fp);

    if (ferror(win32Work->fp) != 0)
    {
        result = Error::ReadError;
    }

    result.Success();
    return readSize;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルを書き込み
 *  \param[in,out]  work        アクセスするファイルのワーク
 *  \param[out]     result      処理結果
 *  \param[out]     buffer      書き込み先のアドレス
 *  \param[in]      size        書き込みサイズ
 *  \return         実際に書き込んだバイト数
 */
/* ------------------------------------------------------------------------- */
size_t Win32Delegate::Write(AccessWork *work, Result &result, const void *buffer, size_t size) noexcept
{
    auto *win32Work = GetWork(work, OpenMode::Write);
    if (win32Work == nullptr)
    {
        result = Error::InvalidArgument;
        return 0;
    }

    clearerr(win32Work->fp);
    auto writeSize = fwrite(buffer, 1, size, win32Work->fp);
    if (writeSize < size)
    {
        result = Error::WriteError;
    }
    else
    {
        result.Success();
    }
    
    return writeSize;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ストリーム位置を設定
 *  \param[in,out]  work        アクセスするファイルのワーク
 *  \param[out]     result      処理結果
 *  \param[in]      seekType    シークタイプ
 *  \param[in]      offset      オフセット
 *  \return         エラー発生時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
size_t Win32Delegate::Seek(AccessWork *work, Result &result, SeekType seekType, int32_t offset) noexcept
{
    auto *win32Work = GetWork(work);
    if (win32Work == nullptr)
    {
        result = Error::InvalidArgument;
        return 0;
    }

    int seek = SEEK_SET;    
    switch (seekType)
    {
        case SeekType::Top:
            seek = SEEK_SET;
            break;
            
        case SeekType::End:
            seek = SEEK_END;
            break;
            
        case SeekType::Current:
            seek = SEEK_CUR;
            break;
    }
    
    if (fseek(win32Work->fp, offset, seek) != 0)
    {
        result = GetError(errno);
    }
    else
    {
        result.Success();
    }

    return static_cast<size_t>(ftell(win32Work->fp));
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ストリーム位置を取得
 *  \param[in]  work    アクセスするファイルのワーク
 *  \param[out] result  エラー発生時の結果の格納先
 *  \return     現在のストリーム位置
 */
/* ------------------------------------------------------------------------- */
size_t Win32Delegate::GetOffset(AccessWork *work, Result &result) const noexcept
{
    auto *win32Work = GetWork(work);
    if (win32Work == nullptr)
    {
        result = Error::InvalidArgument;
        return 0;
    }
    
    auto offset = ftell(win32Work->fp);
    if (offset < 0)
    {
        result = Error::UnknownError;
        return 0;
    }

    result.Success();
    return static_cast<size_t>(offset);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルストリームが終端に達しているかを取得
 *  \param[in,out]  work    アクセスするファイルのワーク
 *  \param[out]     result  エラー発生時の結果の格納先
 *  \retval         true    終端に達している
 *  \retval         false   終端に達していない
 */
/* ------------------------------------------------------------------------- */
bool Win32Delegate::IsEOF(AccessWork *work, Result &result) const noexcept
{
    auto *win32Work = GetWork(work);
    if (win32Work == nullptr)
    {
        result = Error::InvalidArgument;
        return false;
    }

    auto offset = GetOffset(work, result);
    if (result.HasError())
    {
        return false;
    }
    if (offset >= win32Work->size)
    {
        return true;
    }
    
    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      オープンしているファイルのサイズを取得
 *  \param[in]  work    アクセスするファイルのワーク
 *  \param[out] result  エラー発生時の結果の格納先
 *  \return     ファイルサイズ
 */
/* ------------------------------------------------------------------------- */
size_t Win32Delegate::GetSize(AccessWork *work, Result &result) const noexcept
{
    auto *win32Work = GetWork(work);
    if (win32Work == nullptr)
    {
        result = Error::InvalidArgument;
        return 0;
    }

    auto offset = ftell(win32Work->fp);
    if (fseek(win32Work->fp, 0, SEEK_END) != 0)
    {
        result = GetError(errno);
        return 0;
    }
    auto size = ftell(win32Work->fp);
    if (fseek(win32Work->fp, offset, SEEK_SET) != 0)
    {
        result = GetError(errno);
        return 0;
    }
    
    if (size < 0)
    {
        result = Error::UnknownError;
        return 0;
    }

    result.Success();
    return static_cast<size_t>(size);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルのサイズを取得
 *  \param[in,out]  mountWork   マウント毎のワーク
 *  \param[out]     result      エラー発生時の結果の格納先
 *  \param[in]      path        サイズを取得するファイルのパス
 *  \return         ファイルサイズ
 */
/* ------------------------------------------------------------------------- */
size_t Win32Delegate::GetSize([[maybe_unused]] MountWork *mountWork, Result &result, const PathView &path) const noexcept
{
    struct _stat st = {};
    
    if (_wstat(ConvertPath(path).c_str(), &st) != 0)
    {
        result = GetError(errno);
        return 0;
    }
    
    result.Success();
    return static_cast<size_t>(st.st_size);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ディレクトリを作成する
 *  \param[in,out]  mountWork   マウント毎のワーク
 *  \param[in]      path        作成するディレクトリのパス
 *  \return         エラー発生時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result Win32Delegate::MakeDirectory([[maybe_unused]] MountWork *mountWork, const PathView &path) noexcept
{
    if (!CreateDirectory(ConvertPath(path).c_str(), nullptr))
    {
        switch (GetLastError())
        {
            case ERROR_ALREADY_EXISTS:
                return Result::Succeeded();

            case ERROR_PATH_NOT_FOUND:
                return Error::PathNotFound;

            default:
                return Error::UnknownError;
        }
    }

    return Result::Succeeded();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルの移動・リネーム
 *  \param[in,out]  mountWork   マウント毎のワーク
 *  \param[in]      sourcePath  移動元のパス
 *  \param[in]      destPath    移動先のパス
 *  \return         エラー発生時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result Win32Delegate::Move([[maybe_unused]] MountWork *mountWork, const PathView &sourcePath, const PathView &destPath) noexcept
{
    const int result = _wrename(ConvertPath(sourcePath).c_str(), ConvertPath(destPath).c_str());
    if (result == -1)
    {
        return GetError(errno);
    }

    return Result::Succeeded();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルの削除
 *  \param[in,out]  mountWork   マウント毎のワーク
 *  \param[in]      path        削除するファイルのパス
 *  \return         エラー発生時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
Result Win32Delegate::Remove([[maybe_unused]] MountWork *mountWork, const PathView &path) noexcept
{
    const int result = _wremove(ConvertPath(path).c_str());
    if (result != 0)
    {
        return GetError(errno);
    }

    return Result::Succeeded();
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief          ファイルの存在をチェック
 *  \param[in,out]  mountWork   マウント毎のワーク
 *  \param[out]     result      エラー発生時の結果の格納先
 *  \param[in]      path        チェックするファイルのパス
 *  \return         存在する場合はエラーなしのリザルトが返る
 */
/* ------------------------------------------------------------------------- */
bool Win32Delegate::Exists([[maybe_unused]] MountWork *mountWork, Result &result, const PathView &path) noexcept
{
    struct _stat st = {};
    if (_wstat(ConvertPath(path).c_str(), &st) == 0)
    {
        result.Success();
        return true;
    }
    
    if (errno == ENOENT)
    {
        result.Success();
    }
    else
    {
        result = GetError(errno);
    }
    
    return false;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      作業用構造体からこのクラス用の構造体を取得する
 *  \param[in]  work    作業用構造体
 *  \param[in]  mode    オープンモード指定．OpenMode::Noneを指定した場合は無視される
 *  \return     作業用構造体がこのクラス用に作られていれば有効なアドレスを返す
 */
/* ------------------------------------------------------------------------- */
Win32Delegate::Win32Work *Win32Delegate::GetWork(AccessWork *work, OpenMode mode) noexcept
{
    if (work != nullptr)
    {
        if (work->IsOpen())
        {
            if ((mode == OpenMode::None) || (mode == work->GetOpenMode()))
            {
                if (work->GetDelegateKey() == kDelegateKey)
                {
                    return static_cast<Win32Work *>(work);
                }
            }
        }
    }
    
    return nullptr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      オープンモードからモード指定用の文字列を取得
 *  \param[in]  mode    オープンモード
 *  \return     オープンモードに対応した文字列
 */
/* ------------------------------------------------------------------------- */
const wchar_t *Win32Delegate::GetModeString(OpenMode mode) noexcept
{
    static const wchar_t *modeReadBinary = L"rb";
    static const wchar_t *modeWriteBinary = L"wb";

    switch (mode)
    {
        case OpenMode::Read:
            return modeReadBinary;

        case OpenMode::Write:
            return modeWriteBinary;

        default:
            break;
    }

    return nullptr;
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      パスをwstringに変換
 *  \param[in]  path    変換元のパス
 *  \return     wstringに変換したパス
 */
/* ------------------------------------------------------------------------- */
STL::wstring Win32Delegate::ConvertPath(const PathView &path) noexcept
{
    auto srcSize = strlen(path) + 1;
    const size_t bufferSize = srcSize * 3;
    auto buffer = STL::make_unique<wchar_t[]>(bufferSize);

    MultiByteToWideChar(CP_UTF8, 0, path, static_cast<int>(srcSize), buffer.get(), static_cast<int>(bufferSize));

    return {buffer.get()};
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      エラー番号から処理結果を取得する
 *  \param[in]  errorNumber     ファイル操作後のerrno
 *  \return     エラー番号に対応した処理結果
 */
/* ------------------------------------------------------------------------- */
Error Win32Delegate::GetError(int errorNumber) noexcept
{
    switch (errorNumber)
    {
        case ENOENT:
            return Error::FileNotExist;
            
        case EPERM:
        case EACCES:
            return Error::NotPermitted;
            
        case EINVAL:
            return Error::InvalidArgument;
            
        default:
            break;
    }
    
    return Error::UnknownError;
}

}   // namespace MGL::File

#endif  // MGL_FILE_DELEGATE_ENABLE_WIN32
// vim: et ts=4 sw=4 sts=4
